import * as React from 'react';
import { Background, LoadingSpinner, StyledLayout, Table, TableHeader, TableHeading } from 'twitch-core-ui';
import { config, vienna } from 'vienna';
import { RBAC } from 'vienna/common/clients/rbacClient';
import { GameApplication } from 'vienna/common/clients/rbacrpc';
import { ModalShown } from 'vienna/core/actions/modal';
import { PublicProps as ManageGameApplicationProps } from 'vienna/features/manage-game-application';
import {
  GameApplicationRow,
  GameApplicationTableHeader,
} from 'vienna/pages/game-application-directory/components/game-application-row';

export interface PublicProps {
  company_id: string;
  gameApps: GameApplication[];
  onAppChange: () => void;
}

interface RouteProps {}

export interface ReduxDispatchProps {
  showManageGameApplicationModal: (props: ManageGameApplicationProps) => ModalShown;
}

type Props = PublicProps & ReduxDispatchProps;

export interface State {
  gameApplications: GameApplication[];
  showManageModal: boolean;
}

export class CompanyPendingGamesTabComponent extends React.Component<Props> {
  // Create initial state
  public state: State = {
    gameApplications: [],
    showManageModal: false,
  };

  public render() {
    if (!this.props.gameApps) {
      return <LoadingSpinner />;
    }

    return (
      <StyledLayout background={Background.Base} margin={{ y: 1 }} elevation={1}>
        <Table>
          {GameApplicationTableHeader(true)}
          {this.props.gameApps.map(gameApplication => (
            <GameApplicationRow
              key={gameApplication.id}
              gameApp={gameApplication}
              onManage={this.onManage}
              excludeCompanyCell={true}
            />
          ))}
        </Table>
      </StyledLayout>
    );
  }

  private onManage = (currentGameApplication: GameApplication) => {
    if (!currentGameApplication) {
      return;
    }

    this.props.showManageGameApplicationModal({
      gameApp: currentGameApplication,
      onApprove: this.onApprove,
      onDelete: this.onDelete,
    });
  };

  private onApprove = async (gameApp: GameApplication) => {
    const { error } = await RBAC.onboardGame({
      gameApplicationId: gameApp.id,
    });
    if (error) {
      return error.message;
    }
    this.props.onAppChange();
    return '';
  };

  private onDelete = async (gameApp: GameApplication) => {
    const { error } = await RBAC.deleteGameApplication({
      id: gameApp.id,
      skipEmail: false,
    });
    if (error) {
      return error.message;
    }
    this.props.onAppChange();
    return '';
  };
}
