import * as React from 'react';
import { Background, LoadingSpinner, StyledLayout, Table, TableHeader, TableHeading } from 'twitch-core-ui';
import { RBAC } from 'vienna/common/clients/rbacClient';
import { Membership } from 'vienna/common/clients/rbacrpc';
import { ModalShown } from 'vienna/core/actions/modal';
import { PublicProps as DeleteUserModalProps } from 'vienna/features/delete-user';
import { PublicProps as ManageUserModalProps, State as ModalState, SubmitResult } from 'vienna/features/manage-user';
import { CompanyUsersTableRow } from 'vienna/pages/manage-company/components/company-users-tab/components/company-users-table-row';

export interface PublicProps {
  company_id: string;
  memberships: Membership[];
  onUsersUpdated: () => void;
}

export interface ReduxDispatchProps {
  showDeleteUserModal: (props: DeleteUserModalProps) => ModalShown;
  showManageUserModal: (props: ManageUserModalProps) => ModalShown;
}

type Props = PublicProps & ReduxDispatchProps;

export class CompanyUsersTabComponent extends React.Component<Props> {
  public render() {
    if (!this.props.memberships) {
      return <LoadingSpinner />;
    }

    return (
      <StyledLayout background={Background.Base} margin={{ y: 1 }} elevation={1}>
        <Table>
          <TableHeader>
            <TableHeading label="First Name" />
            <TableHeading label="Last Name" />
            <TableHeading label="Title" />
            <TableHeading label="Role" />
            <TableHeading label="Email" />
            <TableHeading label="Twitch ID" />
            <TableHeading label="Date Created" />
            <TableHeading label="Actions" />
          </TableHeader>
          {this.props.memberships.map(memb => (
            <CompanyUsersTableRow
              key={memb.twitchId}
              membership={memb}
              onEdit={this.showManageUserModal}
              onDelete={this.showDeleteUserModal}
            />
          ))}
        </Table>
      </StyledLayout>
    );
  }

  public showManageUserModal = (user: Membership) => {
    this.props.showManageUserModal({
      user,
      hasChanged: () => true,
      submitChanges: this.submitUserChanges,
    });
  };

  public showDeleteUserModal = (user: Membership) => {
    this.props.showDeleteUserModal({
      user,
      onDelete: this.onDeleteUser,
    });
  };

  private submitUserChanges = async (
    modalState: ModalState,
    selectedUser: Membership | null,
  ): Promise<SubmitResult> => {
    let hasChanged = true;
    let errorMessage = '';

    if (selectedUser && modalState.role !== selectedUser.role) {
      const { error } = await RBAC.updateUserRole({
        twitchId: selectedUser.twitchId,
        requestingTwitchId: '', // will get value from OAuth token
        companyId: this.props.company_id,
        role: modalState.role,
      });
      if (error) {
        errorMessage = error.message;
      } else {
        hasChanged = false;
      }
    }

    if (
      selectedUser &&
      (modalState.devEmail !== selectedUser.devEmail ||
        modalState.devTitle !== selectedUser.devTitle ||
        modalState.firstName !== selectedUser.firstName ||
        modalState.lastName !== selectedUser.lastName ||
        modalState.twitchId !== selectedUser.twitchId)
    ) {
      const { error } = await RBAC.updateUser({
        devEmail: modalState.devEmail,
        devTitle: modalState.devTitle,
        firstName: modalState.firstName,
        lastName: modalState.lastName,
        twitchId: modalState.twitchId,
        companyId: this.props.company_id,
      });
      if (error) {
        errorMessage = error.message;
      } else {
        hasChanged = false;
      }
    }

    if (!hasChanged) {
      this.props.onUsersUpdated();
    }

    return {
      hasChanged,
      errorMessage,
    };
  };

  private onDeleteUser = async (user: Membership): Promise<string> => {
    const { error } = await RBAC.removeCompanyMembership({
      twitchId: user.twitchId,
      companyId: this.props.company_id,
      requestingTwitchId: '', // will be set with the OAuth token
    });
    if (error) {
      return error.message;
    }

    this.props.onUsersUpdated();
    return '';
  };
}
