import * as React from 'react';
import { connect } from 'react-redux';
import { RouteComponentProps } from 'react-router';
import { compose } from 'redux';
import { Dispatch } from 'redux';
import { bindActionCreators } from 'redux';
import { Action } from 'redux';
import { Button, Display, FlexDirection, Layout, Tab, TabWrapper } from 'twitch-core-ui';
import { KrakenExtension, searchExtensions } from 'vienna/common/clients/extensions';
import { RBAC } from 'vienna/common/clients/rbacClient';
import { DeveloperApplication, Game, GameApplication, Membership } from 'vienna/common/clients/rbacrpc';
import { showModal } from 'vienna/core/actions/modal';
import { withUser } from 'vienna/core/auth/with-user';
import { ActivityLogModal, PublicProps as ActivityLogProps } from 'vienna/features/activity-log';
import { AddGame } from 'vienna/features/add-game/components/add-game';
import { AddUser } from 'vienna/features/add-user';
import { TransferExtension } from 'vienna/features/transfer-extension';
import { TransferOwnershipBtn } from 'vienna/features/transfer-ownership/components/btn';
import { wrapDevloperApplication, wrapMembership } from 'vienna/pages/common/data-model-wrap';
import { CompanyEdit } from 'vienna/pages/manage-company/components/company-edit';
import { CompanyExtensionsTab } from 'vienna/pages/manage-company/components/company-extensions-tab/component';
import { CompanyGamesTab } from 'vienna/pages/manage-company/components/company-games-tab';
import { CompanyPendingDevelopersTab } from 'vienna/pages/manage-company/components/company-pending-developers-tab';
import { CompanyPendingGamesTab } from 'vienna/pages/manage-company/components/company-pending-games-tab';
import { CompanyUsersTab } from 'vienna/pages/manage-company/components/company-users-tab';
import { CreateShadowAccountButton } from 'vienna/pages/manage-company/components/create-shadow-account-button';

export interface PublicProps {}

interface RouteProps {
  companyID: string;
}

export interface ReduxDispatchProps {
  showActivityLogModal: (props: ActivityLogProps) => void;
}

type Props = PublicProps & RouteComponentProps<RouteProps> & ReduxDispatchProps;

export interface State {
  activeTab: number;

  // users tab
  memberships: Membership[];

  // games tab
  games: Game[] | null;

  // game applications tab
  gameApps: GameApplication[];

  // dev applications tab
  devApps: DeveloperApplication[];

  // company extensions
  extensions: KrakenExtension[];
}

// CompanyManagePageComponent loads games and users so it can refresh the lists as
// necessary i.e. user or game is added.
class CompanyManagePageComponent extends React.Component<Props, State> {
  // Create initial state
  public state: State = {
    activeTab: 1,

    memberships: [],
    games: [],
    gameApps: [],
    devApps: [],
    extensions: [],
  };

  public componentDidMount() {
    this.fetchUsers();
    // we normally only want to fetch data for visible tabs. the users
    // tab is visible by default. in order to support conditionally
    // showing extensions when a company has them, we have to fetch them
    // here.
    // TODO: Remove when the extensions tab is available for companies
    // without extensions.
    this.fetchExtensions();
  }

  public render() {
    const companyID = this.companyID();

    const activeTab = this.showActiveTab();

    const { activeTab: activeTabNumber, extensions } = this.state;

    return (
      <Layout>
        <CompanyEdit company_id={companyID} />
        <Layout margin={{ y: 2 }} display={Display.Flex} flexDirection={FlexDirection.Row}>
          <AddUser onUserAdded={this.switchToUsersTab} companyId={companyID} /> &nbsp;
          <AddGame onGameAdded={this.switchToGamesTab} companyId={companyID} /> &nbsp;
          <TransferExtension onExtensionAdded={this.switchToExtensionsTab} companyId={companyID} />
          &nbsp;
          <TransferOwnershipBtn onTransfer={this.switchToUsersTab} companyId={companyID} />
          &nbsp;
          <CreateShadowAccountButton companyId={companyID} onCreate={this.switchToUsersTab} /> &nbsp;
          <Button onClick={this.showCompanyActivity}>Activity Log</Button>
        </Layout>

        <Layout margin={{ y: 2 }}>
          <TabWrapper>
            <Tab active={activeTabNumber === 1} onClick={this.switchTab} tabIndex={1}>
              Users
            </Tab>
            <Tab active={activeTabNumber === 2} onClick={this.switchTab} tabIndex={2}>
              Games
            </Tab>
            <Tab active={activeTabNumber === 3} onClick={this.switchTab} tabIndex={3}>
              Pending Games
            </Tab>
            <Tab active={activeTabNumber === 4} onClick={this.switchTab} tabIndex={4}>
              Pending Developers
            </Tab>
            {/* Before org extension GA only show in vienna when company has extensions */}
            {extensions.length > 0 && (
              <Tab active={activeTabNumber === 5} onClick={this.switchTab} tabIndex={5}>
                Extensions
              </Tab>
            )}
          </TabWrapper>
        </Layout>

        <Layout margin={{ y: 2 }}>{activeTab}</Layout>
      </Layout>
    );
  }

  private companyID() {
    return this.props.match.params.companyID;
  }

  private fetchUsers = async () => {
    if (!this.companyID()) {
      return;
    }
    const {
      error,
      data: { memberships },
    } = await RBAC.getUsersByCompanyId({
      role: '',
      id: this.companyID(),
      offset: 0,
      limit: 1000,
      sortBy: 'role',
      orderBy: 'ASC',
    });

    if (error) {
      return;
    }

    this.setState({
      memberships: memberships.map(m => wrapMembership(m)),
    });
  };

  private fetchGames = async () => {
    if (!this.companyID()) {
      return;
    }
    const {
      data: { games },
    } = await RBAC.getGamesByCompany({
      id: this.companyID(),
    });
    this.setState({ games });
  };

  private onGameRemoved = () => {
    this.fetchGames();
  };

  private fetchGameApplications = async () => {
    const {
      data: { gameapplications: gameApps },
    } = await RBAC.getGameApplicationsByCompany({
      id: this.companyID(),
    });

    this.setState({ gameApps });
  };

  private fetchDeveloperApplications = async () => {
    const {
      data: { developerApplications: devApps },
    } = await RBAC.getDeveloperApplications({
      companyId: this.companyID(),
      limit: 0,
      offset: 0,
    });

    this.setState({
      devApps: devApps.map(app => {
        return wrapDevloperApplication(app);
      }),
    });
  };

  private fetchExtensions = async () => {
    const {
      data: { resources },
    } = await RBAC.listResources({
      companyId: this.companyID(),
      resourceType: 'extension',
      limit: 100,
      offset: 0,
    });

    if (!resources || !resources.length) {
      return;
    }

    const extensions = await Promise.all(
      resources.map(async resource => {
        const { extensions: exts, status } = await searchExtensions('', resource.externalId, 1, 0);

        if (status !== 200 || !exts || !exts.length) {
          return null;
        }

        return exts[0];
      }),
    );

    this.setState({
      // filter any null extensions;
      extensions: extensions.filter(extension => extension) as KrakenExtension[],
    });
  };

  private showActiveTab = () => {
    switch (this.state.activeTab) {
      case 2:
        return (
          <CompanyGamesTab company_id={this.companyID()} games={this.state.games} onGameRemoved={this.onGameRemoved} />
        );
      case 3:
        return (
          <CompanyPendingGamesTab
            company_id={this.companyID()}
            gameApps={this.state.gameApps}
            onAppChange={this.fetchGameApplications}
          />
        );
      case 4:
        return (
          <CompanyPendingDevelopersTab
            company_id={this.companyID()}
            devApps={this.state.devApps}
            onAppChange={this.fetchDeveloperApplications}
          />
        );
      case 5:
        return (
          <CompanyExtensionsTab
            extensions={this.state.extensions}
            companyId={this.companyID()}
            onExtensionsUpdated={this.fetchExtensions}
          />
        );
      default:
        return (
          <CompanyUsersTab
            company_id={this.companyID()}
            memberships={this.state.memberships}
            onUsersUpdated={this.fetchUsers}
          />
        );
    }
  };

  private switchTab = (event: React.MouseEvent<HTMLInputElement>) => {
    switch (event.currentTarget.tabIndex) {
      case 2:
        return this.setState({ activeTab: event.currentTarget.tabIndex }, this.fetchGames);
      case 3:
        return this.setState({ activeTab: event.currentTarget.tabIndex }, this.fetchGameApplications);
      case 4:
        return this.setState({ activeTab: event.currentTarget.tabIndex }, this.fetchDeveloperApplications);
      default:
        return this.setState({ activeTab: event.currentTarget.tabIndex }, this.fetchUsers);
    }
  };

  private switchToUsersTab = () => {
    this.setState(
      {
        activeTab: 1,
      },
      this.fetchUsers,
    );
  };

  private switchToGamesTab = () => {
    this.setState(
      {
        activeTab: 2,
      },
      this.fetchGames,
    );
  };
  private switchToExtensionsTab = () => {
    this.setState(
      {
        activeTab: 5,
      },
      this.fetchExtensions,
    );
  };

  private showCompanyActivity = () => {
    this.props.showActivityLogModal({
      title: this.companyID(),
      companyId: this.companyID(),
    });
  };
}

function mapDispatchToProps(dispatch: Dispatch<Action>): ReduxDispatchProps {
  return bindActionCreators(
    {
      showActivityLogModal: ({ ...props }) => showModal(ActivityLogModal, props),
    },
    dispatch,
  );
}

export const CompanyManagePage: React.ComponentClass<PublicProps> = compose(
  connect(
    null,
    mapDispatchToProps,
  ),
  withUser(),
)(CompanyManagePageComponent);
