import { RBACAdminMock } from 'vienna/common/clients/rbacrpcMock';
const adminMock = new RBACAdminMock();
jest.doMock('vienna/common/clients/rbacClient', () => ({
  RBACAdmin: adminMock,
}));

import { FormGroup, Modal, ModalFooter } from 'twitch-core-ui';
import { setupShallowTest } from 'vienna/tests/utils/setup';
import { CreateShadowAccountModal, CSVRow, PublicProps, State, testSelector } from './component';

const mockCloseModal = jest.fn();
const mockOnCreate = jest.fn();
const testCompanyId = 'test_company_id';

const pause = (milliseconds: number) => {
  return new Promise(resolve => setTimeout(resolve, milliseconds));
};

const setupShallow = setupShallowTest<PublicProps, State>(
  CreateShadowAccountModal,
  (): PublicProps => ({
    closeModal: mockCloseModal,
    onCreate: mockOnCreate,
    companyId: testCompanyId,
  }),
);

const mockCSVFile = (): File => {
  const content = 'firstname (optional),lastname (optional),title (optional),email (required)\n,,, testemail@test.com';
  const blob = new Blob([content], { type: 'text/csv' });
  blob['lastModified'] = new Date();
  blob['name'] = 'test.csv';
  return blob as File;
};

describe('Test CreateShadowAccountModal', () => {
  it('renders download template button', () => {
    const { wrapper } = setupShallow();

    expect(
      wrapper.find({
        'data-test-selector': testSelector.downloadTemplateBtn,
      }),
    ).toExist();
  });

  it('renders upload csv button', () => {
    const { wrapper } = setupShallow();

    expect(
      wrapper.find({
        'data-test-selector': testSelector.uploadCSVBtn,
      }),
    ).toExist();
  });

  it('renders generate and cancel button, and generate button by default is disabled', () => {
    const { wrapper } = setupShallow();

    const footer = wrapper
      .find(Modal)
      .dive()
      .find(ModalFooter);

    expect(footer).toExist();
    expect(footer.dive().find({ children: 'Generate' })).toExist();
    expect(footer.dive().find({ children: 'Cancel' })).toExist();
  });

  it('renders disabled generate button when no file is uploaded', () => {
    const { wrapper } = setupShallow();

    const generateButton = wrapper
      .find(Modal)
      .dive()
      .find(ModalFooter)
      .dive()
      .find({ children: 'Generate' });

    expect(generateButton).toExist();
    expect(generateButton.props().disabled).toBeTruthy();
  });

  it('renders file name in FromGroup when file is uploaded', () => {
    const { wrapper } = setupShallow();

    wrapper.find('input').simulate('change', { target: { files: [mockCSVFile()] } });
    expect(wrapper.find(FormGroup).props().hint).toEqual('test.csv');
  });

  it('parses csv file data into CSVRow[] into local state when file is uploaded', async () => {
    const { wrapper } = setupShallow();
    const expectedCSVRows: CSVRow[] = [
      {
        firstname: '',
        lastname: '',
        title: '',
        email: 'testemail@test.com',
      },
    ];
    wrapper.find('input').simulate('change', { target: { files: [mockCSVFile()] } });
    // wait for async file reading
    await pause(10);
    expect(wrapper.state().csvRows).toEqual(expectedCSVRows);
  });

  it('calls RBACAdmin.createShadowAccout when generate button is clicked', async () => {
    const { wrapper } = setupShallow();
    const testEmail = 'testemail@email.com';
    const testTwitchID = '12412415';
    const testUsername = 'test_username';
    const testErrorMessage = '';
    const expectedCSVRows: CSVRow[] = [
      {
        firstname: '',
        lastname: '',
        title: '',
        email: 'testemail@test.com',
      },
    ];
    URL.createObjectURL = jest.fn(() => 'test url');
    const mockRedirection = jest.fn();
    window.location.assign = mockRedirection;
    wrapper.find('input').simulate('change', { target: { files: [mockCSVFile()] } });
    // wait for async file reading
    await pause(10);
    expect(wrapper.state().csvRows).toEqual(expectedCSVRows);

    const generateButton = wrapper
      .find(Modal)
      .dive()
      .find(ModalFooter)
      .dive()
      .find({ children: 'Generate' });
    expect(generateButton.props().disabled).toBeFalsy();

    adminMock.createShadowAccountMock.mockReturnValue({
      data: {
        error: '',
        records: [
          {
            email: testEmail,
            twitchId: testTwitchID,
            username: testUsername,
            errorMessage: testErrorMessage,
          },
        ],
      },
    });

    generateButton.simulate('click');
    await pause(10);
    expect(adminMock.createShadowAccountMock.mock.calls.length).toBe(1);
    expect(mockOnCreate.mock.calls.length).toBe(1);
    expect(mockRedirection.mock.calls.length).toBe(1);
  });

  it('calls RBACAdmin.createShadowAccout when generate button is clicked, and disable generate button when response has error', async () => {
    const { wrapper } = setupShallow();
    const testEmail = 'testemail@email.com';
    const testTwitchID = '12412415';
    const testUsername = 'test_username';
    const testErrorMessage = '';
    const expectedCSVRows: CSVRow[] = [
      {
        firstname: '',
        lastname: '',
        title: '',
        email: 'testemail@test.com',
      },
    ];
    URL.createObjectURL = jest.fn(() => 'test url');
    const mockRedirection = jest.fn();
    window.location.assign = mockRedirection;
    wrapper.find('input').simulate('change', { target: { files: [mockCSVFile()] } });
    // wait for async file reading
    await pause(10);
    expect(wrapper.state().csvRows).toEqual(expectedCSVRows);

    const generateButton = wrapper
      .find(Modal)
      .dive()
      .find(ModalFooter)
      .dive()
      .find({ children: 'Generate' });
    expect(generateButton.props().disabled).toBeFalsy();

    const errMsg = 'some backend error';
    adminMock.createShadowAccountMock.mockReturnValue({
      data: {
        error: errMsg,
        records: [
          {
            email: testEmail,
            twitchId: testTwitchID,
            username: testUsername,
            errorMessage: testErrorMessage,
          },
        ],
      },
    });

    generateButton.simulate('click');
    await pause(10);
    expect(adminMock.createShadowAccountMock.mock.calls.length).toBe(1);
    expect(mockOnCreate.mock.calls.length).toBe(0);
    expect(mockRedirection.mock.calls.length).toBe(1);

    expect(wrapper.state().errorMessage).toEqual('Some errors happened, please check the csv file for details');
    expect(
      wrapper
        .find(Modal)
        .dive()
        .find(ModalFooter)
        .dive()
        .find({ children: 'Generate' })
        .props().disabled,
    ).toBeTruthy();
  });
});
