import * as React from 'react';
import {
  Button,
  ButtonState,
  ButtonType,
  CoreText,
  FontSize,
  FormGroup,
  InjectLayout,
  Layout,
  Modal,
  ModalFooterProps,
  ModalHeaderProps,
  ModalSize,
  Position,
  SVGAsset,
  TextAlign,
} from 'twitch-core-ui';
import { config } from 'vienna';
import { RBACAdmin } from 'vienna/common/clients/rbacClient';
import { CreateShadowAccountRecord, CreateShadowAccountResponse } from 'vienna/common/clients/rbacrpc';

export const testSelector = {
  downloadTemplateBtn: 'download-csv-btn',
  uploadCSVBtn: 'upload-csv-btn',
};

export interface PublicProps {
  closeModal: () => void;
  companyId: string;
  onCreate: () => void;
}

export interface CSVRow {
  firstname: string;
  lastname: string;
  title: string;
  email: string;
}

const header: CSVRow = {
  firstname: 'firstname (optional)',
  lastname: 'lastname (optional)',
  title: 'title (optional)',
  email: 'email (required)',
};

const responseHeader: CreateShadowAccountRecord = {
  email: 'email',
  twitchId: 'twitch ID',
  username: 'username',
  errorMessage: 'error message',
};

export interface State {
  errorMessage: string;
  file: File | null;
  csvRows: CSVRow[];
  pendingCreation: boolean;
}

export class CreateShadowAccountModal extends React.Component<PublicProps, State> {
  public state: State = {
    errorMessage: '',
    file: null,
    csvRows: [],
    pendingCreation: false,
  };

  public render() {
    const { closeModal } = this.props;
    const { pendingCreation } = this.state;
    const headerProps: ModalHeaderProps = {
      title: 'Add Shadow Accounts',
      closeButtonAriaLabel: 'close',
    };
    const buttonState = pendingCreation ? ButtonState.Loading : ButtonState.Default;

    const footerProps: ModalFooterProps = {
      primaryButtonProps: {
        children: 'Generate',
        disabled: this.shouldDisableGenerateButton(),
        onClick: this.createShadowAccounts,
        state: buttonState,
      },
      secondaryButtonProps: {
        children: 'Cancel',
        onClick: closeModal,
      },
    };
    return (
      <Modal headerProps={headerProps} footerProps={footerProps} size={ModalSize.Small}>
        {this.renderModalBody()}
      </Modal>
    );
  }

  private shouldDisableGenerateButton = (): boolean => {
    const { csvRows, errorMessage } = this.state;
    return csvRows.length === 0 || !!errorMessage;
  };

  private renderModalBody = () => {
    const { errorMessage, file } = this.state;
    return (
      <Layout padding={{ x: 1, y: 1 }} textAlign={TextAlign.Left}>
        <FormGroup label={''} error={!!errorMessage} errorMessage={errorMessage} hint={file ? file.name : ''}>
          <Layout>
            <CoreText bold fontSize={FontSize.Size5}>
              Shadow Accounts
            </CoreText>
            <CoreText fontSize={FontSize.Size6}>
              Shadow accounts allow people in an organization to test their products on Twitch safely. Emails provided
              for the creation of shadow accounts need to be tied to people within the requesting organization.
            </CoreText>
          </Layout>
          <Layout margin={{ top: 2 }}>
            <CoreText bold fontSize={FontSize.Size5}>
              CSV Template
            </CoreText>
            <Button
              icon={SVGAsset.Download}
              type={ButtonType.Hollow}
              onClick={this.handleDownloadTemplate}
              download={'template.csv'}
              data-test-selector={testSelector.downloadTemplateBtn}
            >
              Download CSV Template
            </Button>
          </Layout>
          <Layout margin={{ top: 2 }} position={Position.Relative}>
            <CoreText bold fontSize={FontSize.Size5}>
              Upload File
            </CoreText>
            <Button icon={SVGAsset.Upload} type={ButtonType.Default}>
              Upload CSV
            </Button>
            <InjectLayout position={Position.Absolute} fullWidth fullHeight attachLeft attachTop>
              <input
                data-test-selector={testSelector.uploadCSVBtn}
                style={{ opacity: 0 }}
                disabled={false}
                accept={'.csv'}
                type="file"
                onChange={this.handleFileUpload}
              />
            </InjectLayout>
          </Layout>
        </FormGroup>
      </Layout>
    );
  };

  private handleDownloadTemplate = () => {
    const exampleRow: CSVRow = {
      firstname: 'example_firstname',
      lastname: 'example_lastname',
      title: '',
      email: 'example@test.com',
    };
    const dataRows = [header, exampleRow];
    const csvString = dataRows
      .map(row => {
        return this.convertToCSVString(row);
      })
      .join('');
    const url = this.generateCVSDownloadLink(csvString);
    window.location.assign(url);
  };

  private handleFileUpload = async (e: React.FormEvent<HTMLInputElement>) => {
    const files: FileList | null = (e.target as HTMLInputElement).files;
    if (files && files[0]) {
      const uploadedFile: File = files[0];
      this.setState({
        file: uploadedFile,
        errorMessage: '',
      });

      const fileReader = new FileReader();
      fileReader.onloadend = () => {
        const result = fileReader.result;
        if (result) {
          const content: string = result.toString();
          this.convertFileContentToCSVRows(content);
        }
      };
      fileReader.readAsText(uploadedFile as Blob);
    }
  };

  private generateCVSDownloadLink = (csvString: string): string => {
    const blob = new Blob([csvString], { type: 'text/csv;charset=utf-8;' });
    return URL.createObjectURL(blob);
  };

  private convertToCSVString = (obj: CSVRow): string => {
    return `${obj.firstname},${obj.lastname},${obj.title},${obj.email}\n`;
  };

  private convertRecordToCSVString = (obj: CreateShadowAccountRecord): string => {
    return `${obj.email},${obj.twitchId},${obj.username},${obj.errorMessage}\n`;
  };

  private convertFileContentToCSVRows = (content: string | undefined) => {
    if (!content) {
      this.setState({
        csvRows: [],
      });
      return;
    }
    const rows = content.split('\n');
    let result: CSVRow[] = rows.map(row => {
      return this.convertStringToCSVRow(row);
    });
    const firstRow = result[0];
    if (
      firstRow.firstname === header.firstname &&
      firstRow.lastname === header.lastname &&
      firstRow.title === header.title &&
      firstRow.email === header.email
    ) {
      // remove the first row if it's header
      result = result.slice(1);
    }
    this.setState({
      csvRows: result,
    });
  };

  private convertStringToCSVRow = (text: string): CSVRow => {
    const values: string[] = text.split(',');
    return {
      firstname: values[0].trim(),
      lastname: values[1].trim(),
      title: values[2].trim(),
      email: values[3].trim(),
    };
  };

  private createShadowAccounts = async () => {
    const { companyId, onCreate } = this.props;
    const clientID = config.twitchClientId;
    const { csvRows } = this.state;
    if (csvRows.length === 0) {
      return;
    }

    this.setState({
      pendingCreation: true,
      errorMessage: '',
    });

    const { data, error } = await RBACAdmin.createShadowAccount({
      companyId,
      clientId: clientID,
      userInput: csvRows.map(row => {
        return {
          firstName: row.firstname,
          lastName: row.lastname,
          email: row.email,
          title: row.title,
        };
      }),
    });

    this.setState({
      pendingCreation: false,
    });

    this.downloadRecords(data);
    if (error || !data || data.error) {
      const errMsg = !error ? 'Some errors happened, please check the csv file for details' : error.message;
      this.setState({
        errorMessage: errMsg,
      });
      return;
    }
    onCreate();
  };

  private downloadRecords(data: CreateShadowAccountResponse) {
    let records = [responseHeader];
    if (!!data && !!data.records && data.records.length > 0) {
      records = records.concat(data.records);
    }
    if (records.length === 1) {
      return;
    }
    const csvString = records
      .map(record => {
        return this.convertRecordToCSVString(record);
      })
      .join('');
    const url = this.generateCVSDownloadLink(csvString);
    window.location.assign(url);
  }
}
