import * as History from 'history';
import createHistory from 'history/createBrowserHistory';
import * as merge from 'lodash.merge';
import * as React from 'react';
import { RouteComponentProps, Router } from 'react-router';
import { vienna } from 'vienna';

/**
 * Wraps a component inside of a <Router> element to preserve routing
 * and other router based components like <Link>. This function calls
 * `vienna.wrap()` on the element parameter to ensure it is also wrapped
 * by a <Provider> and has access to the viennaStore and its state.
 *
 * See: https://github.com/ReactTraining/react-router/blob/master/packages/react-router/docs/guides/testing.md
 * for router testing strategies.
 *
 * @param element The component to wrap
 */
export function withRouter(element: JSX.Element) {
  const history = createHistory();
  return <Router history={history}>{vienna.wrap(element)}</Router>;
}

// TODO: solely utilize overrides to set values since lodash allows us to do a recursive merge
export function mockRouteProps<Props extends object>(props: Props, overrides?: {}): RouteComponentProps<Props> {
  const defaultProps: RouteComponentProps<Props> = {
    match: {
      params: props,
      isExact: true,
      path: '/',
      url: '/',
    },
    location: {
      pathname: '',
      search: '',
      hash: '',
      state: {},
      key: '',
    },
    history: {
      length: 50,
      action: 'POP',
      location: {
        pathname: '',
        search: '',
        hash: '',
        state: {},
        key: '',
      },
      createHref: jest.fn(),
      push: jest.fn(),
      replace: jest.fn(),
      go: jest.fn(),
      goBack: jest.fn(),
      goForward: jest.fn(),
      block: jest.fn(),
      listen: jest.fn(),
    },
  };

  // Performs a recursive partial merge making overriding very specific values trivial
  return merge(defaultProps, overrides);
}

/**
 * Mocks route props using the specified location and match params.
 */
export function mockRoutePropsWithLocation<MatchParams = {}>(
  location: History.LocationDescriptorObject,
  matchParams?: MatchParams,
): RouteComponentProps<MatchParams> {
  if (!matchParams) {
    matchParams = {} as MatchParams;
  }

  const mockLocation = merge(
    {
      pathname: '/',
      search: '',
      hash: '',
      state: {},
      key: '123456',
    },
    location,
  );

  return {
    history: {
      action: 'POP',
      block: jest.fn(),
      createHref: jest.fn(),
      go: jest.fn(),
      goBack: jest.fn(),
      goForward: jest.fn(),
      length: 50,
      listen: jest.fn(),
      location: mockLocation,
      push: jest.fn(),
      replace: jest.fn(),
    },
    location: mockLocation,
    match: {
      isExact: true,
      params: matchParams,
      path: mockLocation.pathname,
      url: `${mockLocation.pathname}${mockLocation.search}${mockLocation.hash}`,
    },
  };
}

/**
 * Mock the router context for a test. Use like:
 *
 *   setupShallowTest(SettingsRootPresentation, defaultPropGenerator, {
 *     // Provider a router with an initial path:
 *     context: mockRouterContext('/settings/missing/page')
 *   });
 */
export function mockRouterContext(pathname?: string) {
  pathname = pathname || '/';
  return {
    router: {
      history: {},
      route: {
        location: { pathname },
      },
    },
  };
}
