locals {
  container_log_retention = 7
  project_name            = "vienna"
  owner                   = "kevipike@twitch.tv"
  team                    = "hubsights"
}

module "core_account" {
  # NOTE: if we expand to multiple regions, only instantiate this in one region.
  # This module only contains global resources like IAM, so we do not need one per region.
  source = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//core_account?ref=add_cloudwatch_agent_to_asg_userdata"

  environment = "${var.environment}"
  team        = "${local.team}"
  xray_role   = "${var.xray_web_arn}"
}

module "core_region" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//core_region?ref=add_cloudwatch_agent_to_asg_userdata"

  container_log_retention = "${local.container_log_retention}"
  environment             = "${var.environment}"
  region                  = "${data.aws_region.current.name}"
  team                    = "${local.team}"
  xray_task_arn           = "${module.core_account.xray_task_role}"
  xray_web_arn            = "${var.xray_web_arn}"
}

locals {
  env_region = "${join(".", list(var.environment, data.aws_region.current.name))}"
}

module "cluster" {
  source = "../modules/ecs"

  container_instance_profile_id = "${module.core_account.container_instance_profile_id}"
  environment                   = "${var.environment}"
  max_size                      = 12
  min_size                      = 2
  name                          = "${local.project_name}"
  owner                         = "${local.owner}"
  ssm_agent_loggroup            = "${module.core_region.ssm_loggroup}"

  security_groups = "${module.vpc.sg_id[local.env_region]}"

  vpc_subnet_ids = "${module.vpc.private_subnets[local.env_region]}"
}

resource "aws_ecr_repository" "repo" {
  name = "${local.project_name}"
}

data "aws_elb_service_account" "main" {}

# S3 bucket for ELB logs

resource "aws_s3_bucket" "elb_logs" {
  bucket = "${data.aws_iam_account_alias.current.account_alias}-logs"
  acl    = "private"

  policy = "${data.aws_iam_policy_document.elb_logs_policy_doc.json}"
}

data "aws_iam_policy_document" "elb_logs_policy_doc" {
  policy_id = "Policy"

  statement {
    actions   = ["s3:PutObject"]
    resources = ["arn:aws:s3:::${data.aws_iam_account_alias.current.account_alias}-logs/*"]

    principals {
      type        = "AWS"
      identifiers = ["${data.aws_elb_service_account.main.arn}"]
    }
  }
}
