locals {
  cluster_name = "${var.name}-${var.environment}"
}

resource "aws_ecs_cluster" "cluster" {
  name = "${local.cluster_name}"
}

data "aws_ami" "ecs_ami" {
  most_recent = true
  owners      = ["amazon"]

  filter {
    name   = "owner-alias"
    values = ["amazon"]
  }

  filter {
    name   = "name"
    values = ["amzn-ami-*-amazon-ecs-optimized"]
  }
}

locals {
  // This is on of the nastiest hacks I've ever had to write.
  // TLDR: turn the deeply neseted hybrid-typed map into json, and look for the key we want. Apply domain knowledge
  // to assume that the `volume_size` key only appears in one place.
  // HCL only supports strongly typed maps, i.e. map[string]int or map[string]string or even map[string]map[string]int
  // data.aws_ami.ecs_ami.block_device_mappings[0] has some string values and some map values, so it can't be stored
  // directly in a local.
  // HIL doesn't support chaining indexes, like block_device_mappings[0]["ecs"]["volume_size"] so we can't
  // deftly avoid storing the hybrid-valued map.
  // The lookup(map, key) builtin is typed similarly to how maps work. It *must* return a string. Looking up a key
  // whose value is a map will cause a runtime error.
  // My understanding is that this is fixed in HCL2.
  root_volume_json = "${jsonencode(data.aws_ami.ecs_ami.block_device_mappings[0])}"

  root_volume_size = "${replace(local.root_volume_json, "/.*\\\"volume_size\\\":\\\"([[:digit:]]+)\\\".*/", "$1")}"
}

module "control_asg" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//ecsasg?ref=add_cloudwatch_agent_to_asg_userdata"

  ami                           = "${data.aws_ami.ecs_ami.id}"
  cluster_name                  = "${local.cluster_name}"
  container_instance_profile_id = "${var.container_instance_profile_id}"
  desired_capacity              = "${var.min_size}"
  environment                   = "${var.environment}"
  instance_type                 = "${var.instance_type}"
  ldap_group                    = "${var.ldap_group}"
  max_size                      = "${var.max_size}"
  min_size                      = "${var.min_size}"
  name                          = "${var.name}"
  owner                         = "${var.owner}"
  root_volume_size              = "${local.root_volume_size}"
  security_groups               = "${var.security_groups}"
  ssm_agent_loggroup            = "${var.ssm_agent_loggroup}"
  team                          = "${var.team}"
  vpc_zone_identifier           = "${join(",", var.vpc_subnet_ids)}"

  instance_attributes = {
    vpc = "standard"
  }

  scale_in_cooldown               = 180
  scale_in_cpu_percent_threshold  = 10
  scale_in_evaluation_periods     = 3
  scale_in_increment              = -1
  scale_out_cooldown              = 180
  scale_out_cpu_percent_threshold = 25
  scale_out_evaluation_periods    = 3
  scale_out_increment             = 2
}
