#!/bin/ash -e

function die() {
  echo "$@" >&2
  exit 1
}

# If we're not already there, move to the same directory as the script.
[[ -n "$(dirname $0)" ]] && cd $(dirname $0)

# AWS_PROFILE defines the environment folder
if [ "$AWS_PROFILE" = "twitch-rbac-prod" ]
then
  environment="production"
elif [ "$AWS_PROFILE" = "twitch-rbac-dev" ]
then
  environment="staging"
else
  die "AWS_PROFILE environment variable must be twitch-rbac-dev (staging) or twitch-rbac-prod (production)"
fi

# Folder with the module to run (account, service)
[[ -z "$TF_FOLDER" ]] && die "TF_FOLDER environment variable required."
[[ ! -d "$TF_FOLDER" ]] && die "TF_FOLDER environment variable must identify a terraform module to run."

tfvars="${PWD}/tfvars/$environment-$(basename $TF_FOLDER).tfvars"
[[ ! -f "$tfvars" ]] && die "Could not find file $tfvars"

cd "$TF_FOLDER"


# Clean up old runs
rm -rf .terraform *.plan

# Init providers and backend
terraform init -backend-config bucket="$AWS_PROFILE" -input=false

# Refresh can update the remote state in ways that `terraform plan -refresh=true` cannot.
terraform refresh -input=false -var-file "$tfvars"

# Do the plan and write the output to a file so we can apply it later.
# Pass -refresh=false because we manually refreshed above.
# NOTE: This can return non-zero but still be successful.
set +e
terraform plan -detailed-exitcode -out "${environment}.tf_plan" -input=false -var-file "$tfvars" -refresh=false >/dev/null
code=$?
set -e

# Based on the exit code, decide how to proceed.
# 0 = success, but no diff --> remove the plan
# 1 = failure, exit 1
# 2 = success with diff --> keep the plan
if [[ "$code" == "0" ]]
then
    echo "Terraform succeeded, but no diffs were generated."
    rm -f "${environment}.tf_plan"
elif [[ "$code" == "2" ]]
then
    echo "Terraform succeeded, plan was generated."
    terraform show "${environment}.tf_plan"
else
    die "Terraform plan failed, status code: $?"
fi

# Terraform fmt is just like go fmt! This script is meant for CI so we're going to run it in a checking mode rather
# than a fixing mode.
# A note on ordering: run the plan before checking the formatting. While developing, it's important to see the diff
# before we fail the build because you indented something wrong.
cd ..
echo 'Checking terraform formatting. If this fails, run `./docker-tf.sh terraform fmt` to reformat your code and commit the changes.'

# This is gnarly because of
# https://github.com/hashicorp/terraform/issues/6663
# https://github.com/hashicorp/terraform/issues/15607
echo -en "\e[1m\e[91m"  # bold, bright red
any_failed=0
for f in $(find . \( -name '*.tf' -or -name '*.tfvars' \) -not -path '*.terraform/modules*')
do

  if ! terraform fmt -write=false -list=true -check=true $f
  then
    any_failed=1
  fi
done

if [[ $any_failed != 0 ]]
then
  echo "Terraform linting step failed"
fi
echo -en "\e[0m\e[39m" # reset back to normal colors
exit $any_failed
