locals {
  name           = "hubsights-${var.environment}"
  vienna_max_5xx = 100
}

locals {
  env_region = "${join(".", list(var.environment, data.aws_region.current.name))}"
}

resource "aws_alb_target_group" "vienna" {
  deregistration_delay = "30"
  name                 = "${local.name}"

  # Port will be overridden per target by ECS.
  port     = "1"
  protocol = "HTTP"
  vpc_id   = "${module.vpc.vpc_id[local.env_region]}"

  health_check {
    path     = "/"
    protocol = "HTTP"
    matcher  = "200"
  }

  depends_on = ["aws_alb.vienna"]
}

resource "aws_alb_listener" "vienna_http" {
  load_balancer_arn = "${aws_alb.vienna.arn}"
  port              = "80"
  protocol          = "HTTP"

  default_action {
    target_group_arn = "${aws_alb_target_group.vienna.arn}"
    type             = "forward"
  }
}

resource "aws_alb_listener_rule" "redirect_http_to_https" {
  listener_arn = "${aws_alb_listener.vienna_http.arn}"

  action {
    type = "redirect"

    redirect {
      port        = "443"
      protocol    = "HTTPS"
      status_code = "HTTP_301"
    }
  }

  condition {
    field  = "host-header"
    values = ["*.internal.justin.tv"]
  }
}

resource "aws_alb_listener" "vienna_https" {
  certificate_arn   = "${var.alb_listener_certificate_arn}"
  load_balancer_arn = "${aws_alb.vienna.arn}"
  port              = "443"
  protocol          = "HTTPS"
  ssl_policy        = "ELBSecurityPolicy-2016-08"

  default_action {
    target_group_arn = "${aws_alb_target_group.vienna.arn}"
    type             = "forward"
  }
}

data "aws_elb_service_account" "main" {}

resource "aws_s3_bucket" "elb_logs" {
  bucket = "${data.aws_iam_account_alias.current.account_alias}-vienna-logs"
  acl    = "private"

  policy = <<POLICY
{
  "Id": "Policy",
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "s3:PutObject"
      ],
      "Effect": "Allow",
      "Resource": "arn:aws:s3:::${data.aws_iam_account_alias.current.account_alias}-vienna-logs/*",
      "Principal": {
        "AWS": [
          "${data.aws_elb_service_account.main.arn}"
        ]
      }
    }
  ]
}
POLICY
}

resource "aws_alb" "vienna" {
  enable_deletion_protection = true
  internal                   = true
  name                       = "${local.name}"
  security_groups            = ["${module.vpc.sg_id[local.env_region]}"]
  subnets                    = ["${module.vpc.private_subnets[local.env_region]}"]

  access_logs {
    bucket = "${aws_s3_bucket.elb_logs.bucket}"
    prefix = "alb-logs/${local.name}"
  }

  tags {
    Environment = "${var.environment}"
    Owner       = "${var.owner}"
    Name        = "hubsights-vienna"
    Service     = "hubsights-vienna"
  }
}

resource "aws_cloudwatch_metric_alarm" "vienna_5xx" {
  alarm_actions       = ["${aws_sns_topic.pd_high_urgency.arn}"]
  alarm_name          = "hubsights-vienna-5xx-${var.environment}"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = 3
  metric_name         = "HTTPCode_Target_5XX_Count"
  namespace           = "AWS/ApplicationELB"
  ok_actions          = ["${aws_sns_topic.pd_high_urgency.arn}"]
  period              = 60
  statistic           = "Sum"
  threshold           = "${local.vienna_max_5xx}"

  dimensions {
    LoadBalancer = "${aws_alb.vienna.arn_suffix}"
    TargetGroup  = "${aws_alb_target_group.vienna.arn_suffix}"
  }
}
