locals {
  servicePort = {
    vienna = 8080
  }

  envVars = {
    AWS_REGION  = "${data.aws_region.current.name}"
    ENVIRONMENT = "${var.ENVIRONMENT}"
  }
}

data "template_file" "vienna_taskdef" {
  template = "${file("${path.module}/taskdef.json.tmpl")}"

  vars = "${merge(local.envVars, map(
    "containerServicePort", "${local.servicePort["vienna"]}",
    "image", "${var.vienna_image}",
    "logGroup", "${data.terraform_remote_state.service.log_group["vienna"]}",
    "name", "vienna",
  ))}"
}

resource "aws_ecs_task_definition" "vienna" {
  container_definitions = "${data.template_file.vienna_taskdef.rendered}"
  family                = "hubsights-vienna"
  network_mode          = "bridge"
  task_role_arn         = "${data.terraform_remote_state.account.task_iam_role_arn}"

  lifecycle {
    ignore_changes = ["arn", "id", "revision"]
  }
}

resource "aws_ecs_service" "vienna" {
  cluster         = "${data.terraform_remote_state.account.cluster_name}"
  desired_count   = "${local.min_size}"
  iam_role        = "${data.terraform_remote_state.account.service_iam_role_arn}"
  name            = "hubsights-vienna"
  task_definition = "${aws_ecs_task_definition.vienna.arn}"

  load_balancer {
    container_name   = "vienna"
    container_port   = "${local.servicePort["vienna"]}"
    target_group_arn = "${data.terraform_remote_state.service.vienna_target_group_arn}"
  }

  health_check_grace_period_seconds  = 60
  deployment_maximum_percent         = 200
  deployment_minimum_healthy_percent = 100

  ordered_placement_strategy {
    field = "instanceId"
    type  = "spread"
  }

  lifecycle {
    ignore_changes = ["desired_count"]
  }
}

locals {
  cooldown = 180
  max_size = 12
  min_size = 2
  services = ["${aws_ecs_service.vienna.name}"]
}

resource "aws_appautoscaling_target" "target" {
  count = "${length(local.services)}"

  max_capacity = "${local.max_size}"
  min_capacity = "${local.min_size}"
  resource_id  = "service/${data.terraform_remote_state.account.cluster_name}/${element(local.services, count.index)}"

  # NOTE: we have to ignore changes to this value because AWS wraps this role with something else.
  # If you actually want to change the role_arn, comment out the lifecycle block.
  role_arn = "${data.terraform_remote_state.account.task_autoscale_iam_role_arn}"

  scalable_dimension = "ecs:service:DesiredCount"
  service_namespace  = "ecs"

  lifecycle {
    ignore_changes = ["role_arn"]
  }
}

resource "aws_appautoscaling_policy" "scaleout" {
  count = "${length(local.services)}"

  name               = "${element(local.services, count.index)}-scale-out"
  resource_id        = "service/${data.terraform_remote_state.account.cluster_name}/${element(local.services, count.index)}"
  scalable_dimension = "ecs:service:DesiredCount"
  service_namespace  = "ecs"

  step_scaling_policy_configuration {
    adjustment_type         = "ChangeInCapacity"
    cooldown                = "${local.cooldown}"
    metric_aggregation_type = "Average"

    step_adjustment {
      metric_interval_lower_bound = 0
      scaling_adjustment          = 2
    }
  }

  depends_on = ["aws_appautoscaling_target.target"]
}

resource "aws_appautoscaling_policy" "scalein" {
  count = "${length(local.services)}"

  name               = "${element(local.services, count.index)}-scale-in"
  resource_id        = "service/${data.terraform_remote_state.account.cluster_name}/${element(local.services, count.index)}"
  scalable_dimension = "ecs:service:DesiredCount"
  service_namespace  = "ecs"

  step_scaling_policy_configuration {
    adjustment_type         = "ChangeInCapacity"
    cooldown                = "${local.cooldown}"
    metric_aggregation_type = "Average"

    step_adjustment {
      metric_interval_upper_bound = 0
      scaling_adjustment          = -1
    }
  }

  depends_on = ["aws_appautoscaling_target.target"]
}

resource "aws_cloudwatch_metric_alarm" "scaleout" {
  count = "${length(local.services)}"

  alarm_name          = "${element(local.services, count.index)}-scale-out"
  comparison_operator = "GreaterThanOrEqualToThreshold"
  evaluation_periods  = "3"
  metric_name         = "CPUUtilization"
  namespace           = "AWS/ECS"
  statistic           = "Average"
  period              = "60"
  threshold           = "25"

  dimensions = {
    ClusterName = "${data.terraform_remote_state.account.cluster_name}"
    ServiceName = "${element(local.services, count.index)}"
  }

  alarm_actions = ["${aws_appautoscaling_policy.scaleout.*.arn[count.index]}"]
}

resource "aws_cloudwatch_metric_alarm" "scalein" {
  count = "${length(local.services)}"

  alarm_name          = "${element(local.services, count.index)}-scale-in"
  comparison_operator = "LessThanOrEqualToThreshold"
  evaluation_periods  = "3"
  metric_name         = "CPUUtilization"
  namespace           = "AWS/ECS"
  statistic           = "Average"
  period              = "60"
  threshold           = "10"

  dimensions = {
    ClusterName = "${data.terraform_remote_state.account.cluster_name}"
    ServiceName = "${element(local.services, count.index)}"
  }

  alarm_actions = ["${aws_appautoscaling_policy.scalein.*.arn[count.index]}"]
}
