const {
  join,
  resolve
} = require('path');
const HtmlWebpackPlugin = require('html-webpack-plugin');
const ForkTsCheckerWebpackPlugin = require('fork-ts-checker-webpack-plugin');
const TsconfigPathsPlugin = require('tsconfig-paths-webpack-plugin');
const autoprefixer = require('autoprefixer');
const {
  HotModuleReplacementPlugin
} = require('webpack');

module.exports = function (env, argv) {
  const isDev = argv['mode'] === 'development';
  const output = {
    publicPath: '/',
  };

  const stats = {
    warningsFilter: /export .* was not found in .*/,
  };

  const rules = [
    {
      test: /\.(ts|tsx)$/,
      use: {
        loader: 'ts-loader',
        options: {
          transpileOnly: true,
        },
      },
    },
    {
      test: /\.html$/,
      use: {
        loader: 'html-loader',
      },
    },
    {
      test: /\.tsx?$/,
      use: [
        {
          loader: 'ts-loader',
          options: {
            happyPackMode: true,
          },
        },
      ].filter(Boolean)
    },
    {
      test: /\.s?[ac]ss$/,
      use: [
        'style-loader',
        'css-loader',
        {
          loader: 'postcss-loader',
          options: {
            plugins: [
              require('autoprefixer')(),
              require('css-mqpacker')({ sort: true }),
            ],
          },
        },
        {
          loader: 'sass-loader',
          options: {
            includePaths: [join(__dirname, 'src/styles'), 'node_modules/twitch-core-ui/src/scss'],
          },
        },
      ].filter(Boolean),
    },
    {
      test: /\.(woff(2)?|ttf|eot|svg)(\?v=\d+\.\d+\.\d+)?$/,
      use: [{
        loader: 'file-loader',
        options: {
          name: '[name].[ext]',
          outputPath: 'fonts/'
        }
      }]
    }
  ];

  const plugins = [
    new HtmlWebpackPlugin({
      template: join(__dirname, 'src/core/bootstrap/index.html'),
      filename: './index.html',
      inject: true,
    }),
    new ForkTsCheckerWebpackPlugin({
      tslint: true,
      async: false,
    }),
  ];

  if (isDev) {
    plugins.push(new HotModuleReplacementPlugin());
  }

  const entry = {
    main: [join(__dirname, `src/core/bootstrap/init-${env.build}.ts`), join(__dirname, 'src/core/bootstrap/standalone.tsx')],
  };

  const devServer = {
    historyApiFallback: true,
    hot: true,
    port: 3000,
    contentBase: join(__dirname, 'public'),
    stats,
    clientLogLevel: 'error',
  };

  const resolve = {
    extensions: ['.js', '.jsx', '.ts', '.tsx'],
    alias: {
      'vienna': join(__dirname, 'src'),
      'twitch-core-ui': join(__dirname, 'node_modules/twitch-core-ui/src'),
    },
    plugins: [
      new TsconfigPathsPlugin({
        configFile: './tsconfig.json',
        extensions: ['.ts', '.tsx'],
      }),
    ],
  };

  const devtool = 'inline-source-map';

  return {
    entry,
    devtool,
    output,
    module: {
      rules,
    },
    plugins,
    devServer,
    resolve,
    stats,
  };
};
