import os
import sys
from flask import Flask, session, flash, render_template, request, redirect, url_for
from flask import Flask, jsonify
from flask import abort
from flask import make_response
from flask_cors import CORS, cross_origin
from flask_session import Session
import boto3
import json
import requests
import memcache
import logging
import urllib
import urllib2
from StringIO import StringIO
import random
from boto.s3.connection import S3Connection
from liveness_request import *

import time_utils as tu

log_file = '/tmp/channel_similarity_app_{0}.log'.format(tu.date_timestamp())
logging.basicConfig(filename=log_file, level=logging.DEBUG)
logger = logging.getLogger(__name__)

logging.getLogger('boto').setLevel(logging.ERROR)
logging.getLogger('boto3').setLevel(logging.ERROR)
logging.getLogger('botocore').setLevel(logging.ERROR)

'''Memcache key and expiration time (in seconds)'''
MEMCACHE_DEBUG_MODE = True
EXPIRATION = 1
DEFAULT_MEMCACHE_EXP_TIME = 3600 * 2000

'''Initialize memcache object'''
mcache = memcache.Client(['127.0.0.1:11211'], debug=MEMCACHE_DEBUG_MODE)

'''Initialize the app object'''
app = Flask(__name__)
cors = CORS(app, resources={r"/similarity/api/*": {"origins": "*.twitch.tv"}})
app.config.from_object('config')
app.config['SESSION_TYPE'] = 'filesystem'
sess = Session()
sess.init_app(app)

'''Dynamo DB connection'''
try:
    logger.info('{0} - Connecting to Dynamo'.format(tu.full_date_timestamp()))
    DYNAMO_DB = boto3.resource('dynamodb',
                           aws_access_key_id=app.config['AWS_ACCESS_KEY'],
                           aws_secret_access_key=app.config['AWS_SECRET'],
                           region_name=app.config['DYNAMO_DB_REGION'])
    logger.info('{0} - Connection to Dynamo Successful'.format(tu.full_date_timestamp()))
except Exception, e:
    print str(e)
    logger.error('{0} - Error Connecting to Dynamo - {1}'.format(tu.full_date_timestamp(),
                                                                 str(e)))

def log_msg(end_point, msg, exception_error_msg=None):
    if exception_error_msg is None:
        msg = '{0} - {1} : {2}'.format(tu.full_date_timestamp(),
                                       end_point,
                                       msg)
    else:
        msg = '{0} - {1} : {2}'.format(tu.full_date_timestamp(),
                                       end_point,
                                       exception_error_msg)
    return msg

def normalize_scores(channels):
    if len(channels) == 0:
        return channels

    for feature_name in app.config['FEATURES_TO_USE']:
        scores = []
        counts = []
        for channel in channels:
            if 'stats_per_feature' in channel and feature_name in channel['stats_per_feature']:
                if 'score' in channel['stats_per_feature'][feature_name]:
                    scores.append(channel['stats_per_feature'][feature_name]['score'])
                if 'count' in channel['stats_per_feature'][feature_name]:
                    counts.append(channel['stats_per_feature'][feature_name]['count'])

        if len(scores) == 0:
            continue

        max_score = max(scores)
        min_score = min(scores)
        score_sum = sum(scores)

        max_count = max(counts)
        min_count = min(counts)
        count_sum = sum(counts)

        for channel in channels:
            if 'stats_per_feature' in channel and feature_name in channel['stats_per_feature']:
                if 'score' in channel['stats_per_feature'][feature_name]:
                    if score_sum != 0:
                        norm_score = channel['stats_per_feature'][feature_name]['score'] / float(score_sum)
                    else:
                        norm_score = 0
                    channel['stats_per_feature'][feature_name]['normalized_score'] = norm_score

                if 'count' in channel['stats_per_feature'][feature_name]:
                    if count_sum != 0:
                        norm_count = channel['stats_per_feature'][feature_name]['count'] / float(count_sum)
                    else:
                        norm_count = 0
                    channel['stats_per_feature'][feature_name]['normalized_count'] = norm_count

    return channels

def make_response_json(json_data):
    out_channels = []

    if 'channels' not in json_data:
        return out_channels

    '''For each channel get stats per feature'''
    for channel_data in json_data['channels']:

        if 'stats' not in channel_data:
            continue

        stats_per_feature = {}
        for feature_name in app.config['FEATURES_TO_USE']:
            feature_data_dict = {}
            feature_data_dict['count'] = 0
            feature_data_dict['score'] = 0
            if feature_name in channel_data['stats'] and 'intersect' in channel_data['stats'][feature_name]:
                feature_data_dict['count'] = int(channel_data['stats'][feature_name]['intersect'])
            if feature_name in channel_data['stats'] and 'jaccard' in channel_data['stats'][feature_name]:
                feature_data_dict['score'] = float(channel_data['stats'][feature_name]['jaccard'])
            stats_per_feature[feature_name] = feature_data_dict

        channel_data_dict = {'channel_name': channel_data['channel_name'],
                             'aggregated_score': 0,
                             'stats_per_feature': stats_per_feature}

        out_channels.append(channel_data_dict)

    '''Compute aggregated score'''
    for channel in out_channels:
        if 'stats_per_feature' not in channel:
            continue

        agg_score = 0
        for feature_name in app.config['FEATURES_TO_USE']:
            if feature_name in channel['stats_per_feature'] and 'score' in channel['stats_per_feature'][feature_name]:
                agg_score += channel['stats_per_feature'][feature_name]['score']

        channel['aggregated_score'] = agg_score

    out_channels = sorted(out_channels, key=lambda x: x['aggregated_score'], reverse=True)

    return out_channels

def get_similar_channels_from_dynamo(channel_name, live_only, num_ngb):
    try:
        table = DYNAMO_DB.Table(app.config['DYNAMO_SIM_TABLE'])
        json_response = table.get_item(Key={'channel_name': channel_name})
        if 'Item' in json_response:
            return make_response_json(json_response['Item'], num_ngb,
                                      live_only)
        else:
            return None
    except Exception, e:
        logger.error('{0} - Error getting channel similarities - {1} - {2}'.format(
            tu.full_date_timestamp(), channel_name, str(e)))
        return None

def is_low_ccu_channel(channel_name):
    try:
        table = DYNAMO_DB.Table(app.config['DYNAMO_LOW_CCU_TABLE'])
        json_response = table.get_item(Key={'channel_name': channel_name})
        if 'Item' in json_response:
            logger.info(log_msg('Low CCU channel', channel_name))
            return True
        else:
            logger.info(log_msg('Not a low CCU channel', channel_name))
            return False
    except Exception, e:
        logger.error('{0} - Error getting low CCU info - {1} - {2}'.format(
            tu.full_date_timestamp(), channel_name, str(e)))
        return False

def get_similar_channels(channel_name):
    try:
        table = DYNAMO_DB.Table(app.config['DYNAMO_SIM_TABLE'])
        json_response = table.get_item(Key={'channel_name': channel_name})
        if 'Item' in json_response:
            logger.info(log_msg('get_similar_channels: Data exist in Dynamo', channel_name))
            return True, json_response['Item']
        else:
            logger.info(log_msg('get_similar_channels: Data does not exist in Dynamo', channel_name))
            return True, None
    except Exception, e:
        logger.error('{0} - get_similar_channels: Error getting channel similarities - {1} - {2}'.format(
            tu.full_date_timestamp(), channel_name, str(e)))
        return False, None

def parse_request_args(request):
    params = {}
    params['live_only'] = True
    if request.method == 'GET' and 'live' in request.args:
        try:
            params['live_only'] = (request.args.get('live') == 'true')
        except Exception, e:
            logger.error('{0} - Error reading get arguments - {1}'.format(
                tu.full_date_timestamp(), str(e)))

    params['num_ngb'] = app.config['MAX_NEIGHBORS_PER_CHANNEL']
    if request.method == 'GET' and 'n' in request.args:
        try:
            params['num_ngb'] = int(request.args.get('n'))
        except Exception, e:
            logger.error('{0} - Error reading argument n - {1}'.format(
                tu.full_date_timestamp(), str(e)))

    return params

def apply_live_status_filter(channels, live_only, num_ngb):
    if live_only == False:
        rank = 0
        for channel in channels:
            rank += 1
            channel['similarity_rank'] = rank
        return channels

    '''Generate list of all channel names'''
    all_channel_names = [x['channel_name'] for x in channels]
    channel_to_live = {}

    all_channel_names_chunks = [all_channel_names[x:x + 10] for x in xrange(0, len(all_channel_names), 10)]
    for i in range(0, len(all_channel_names_chunks)):
        temp_dict = get_live_from_channel_names(all_channel_names_chunks[i])
        channel_to_live.update(temp_dict)

        '''Check for early break if num_ngb channels are found'''
        num_live_channels = sum([value == True for value in channel_to_live.values()])
        if num_live_channels >= num_ngb:
            break

    live_channels = []
    rank = 0
    for channel in channels:
        rank += 1
        if channel['channel_name'] in channel_to_live and channel_to_live[channel['channel_name']] == True:
            channel['similarity_rank'] = rank
            live_channels.append(channel)

    return live_channels

@app.errorhandler(404)
def not_found(error):
    return make_response(jsonify({'error': 'Not found'}), 404)

@app.route("/")
def main():
    abort(404)

@app.route('/similarity/api/v1.0/channel/<channel_name>/', methods=['GET'])
@cross_origin()
def similar_channels_by_name(channel_name=None, methods=['GET']):
    try:
        logger.info(log_msg('/similarity/api/v1.0/channel/', channel_name))

        '''Low CCU check'''
        if is_low_ccu_channel(channel_name=channel_name):
            return make_response(jsonify({'error': "Oops! There isn't enough data to display relevant connections.",
                                          'query_channel_name' : channel_name}), 404)

        '''Get channel similarities'''
        status, similar_channel_response = get_similar_channels(channel_name=channel_name)
        if status == False:
            return make_response(jsonify({'error': "Oops! Something went wrong getting channel similarities.",
                                          'query_channel_name' : channel_name}), 404)
        elif status == True and similar_channel_response == None:
            return make_response(jsonify({'error': "Oops! This channel doesn't exist.",
                                          'query_channel_name' : channel_name}), 404)

        params = parse_request_args(request)
        logger.info(log_msg('params:', params))

        '''Convert Dynamo response object into JSON format of the API response'''
        channels = make_response_json(similar_channel_response)

        '''Apply live status filtering'''
        channels = apply_live_status_filter(channels=channels[:app.config['MAX_CHANNELS_FOR_JAX_QUERY']],
                                            live_only=params['live_only'],
                                            num_ngb=params['num_ngb'])

        '''Pick first n channels and normalize scores'''
        channels = channels[:params['num_ngb']]

        if params['live_only'] == True and len(channels) == 0:
            return make_response(jsonify({'error': "Oops! We don't have enough information about this user to display relvant live connections.",
                                          'query_channel_name' : channel_name}), 404)
        elif params['live_only'] == False and len(channels) == 0:
            return make_response(jsonify({'error': "Oops! We don't have enough information about this user to display relvant connections.",
                                          'query_channel_name' : channel_name}), 404)

        channels = normalize_scores(channels)

        return jsonify({'channels': channels, 'query_channel_name' : channel_name})
    except Exception, e:
        logger.error('{0} - Error processing /similarity/api/v1.0/channel/ - {1} - {2}'.format(
            tu.full_date_timestamp(), channel_name, str(e)))
        return make_response(jsonify({'error': "Oops! Something went wrong getting channel similarities.",
                                          'query_channel_name' : channel_name}), 404)

if __name__ == '__main__':
    app.secret_key = ')\xceZ\x15\xf2si k\x14\xcb3Mvug\xa4\xd2\xcbii\xf7\xdf5'
    app.config['SESSION_TYPE'] = 'filesystem'
    sess.init_app(app)
    app.run(host='0.0.0.0', port=8000, debug=True)