package multiconfig

import (
	"encoding/json"
	"flag"
	"fmt"
	"os"
)

// Config is the representation of a set of config options. Options are
// registered to a config and then once everything is registered the Parse()
// method is called.
type Config struct {
	parsed     bool
	flagSet    *flag.FlagSet
	funcs      []parseFun
	configFile string
	config     map[string]interface{}
	envGetter  envGetterFunc
	args       []string
}

type parseFun func() error
type envGetterFunc func(string) string

// NewConfig will generate a new config state. This assumes that it is running
// for the base program.
func NewConfig() *Config {
	return &Config{
		parsed:    false,
		args:      os.Args[1:],
		flagSet:   flag.NewFlagSet(os.Args[0], flag.ExitOnError),
		funcs:     []parseFun{},
		config:    make(map[string]interface{}),
		envGetter: os.Getenv,
	}
}

// Parsed returns true if the Config.Parse() method has already been called.
func (c *Config) Parsed() bool {
	return c.parsed
}

// Parse will cause multiconfig to parse command line flags, the config file
// and environment variables. It will set all of the values using the following
// priority:
//
//  - Flag
//  - Environment
//  - JSON Config
//  - Code specified Default
func (c *Config) Parse() error {
	if c.parsed {
		return nil
	}
	c.parsed = true

	c.flagSet.StringVar(&c.configFile, "config", "", "config file to load")
	if err := c.flagSet.Parse(c.args); err != nil {
		return err
	}

	if c.configFile == "" {
		c.configFile = os.Getenv("CONFIG")
	}

	if c.configFile == "" {
		c.configFile = "config.json"
	}

	if err := c.parseConfig(); err != nil {
		return err
	}

	for _, f := range c.funcs {
		if err := f(); err != nil {
			return err
		}
	}

	return nil
}

func (c *Config) parseConfig() error {
	confFile, err := os.Open(c.configFile)
	if err != nil {
		// If the config doesn't exist then we skip loading configs.
		if os.IsNotExist(err) {
			return nil
		}
		return fmt.Errorf("Error opening config %q: %v", c.configFile, err)
	}

	if err := json.NewDecoder(confFile).Decode(&c.config); err != nil {
		return fmt.Errorf("Error parsing config %q: %v", c.configFile, err)
	}
	return nil
}

// Args will return any unparsed command line args
func (c *Config) Args() []string {
	return c.flagSet.Args()
}
