package twitchhttp

import (
	"log"
	"net/http"
	// importing twitchhttp implies intent to host pprof metrics, so we automatically enable it.
	_ "net/http/pprof"
	"time"

	"code.justin.tv/common/chitin"
	"code.justin.tv/common/config"
	"code.justin.tv/common/gometrics"

	"golang.org/x/net/context"

	"github.com/zenazn/goji/web"
)

// Server allows simple definition of an HTTP service which meets Twitch requirements for production readiness.
type Server struct {
	http.Handler
	routes *web.Mux
}

type HandlerFunc func(context.Context, web.C, http.ResponseWriter, *http.Request)

func init() {
	config.Register(map[string]string{
		"bind-address":       ":8000",
		"debug-bind-address": ":6000",
	})
}

// NewServer allocates and returns a Server. The /debug/running endpoint is registered for load balancer health checks.
func NewServer() (*Server, error) {
	router := web.New()

	ctx, err := chitin.ExperimentalTraceContext(context.Background())
	if err != nil {
		return nil, err
	}

	router.Use(func(h http.Handler) http.Handler { return chitin.Handler(h, chitin.SetBaseContext(ctx)) })
	router.Get("/", home)
	router.Get("/debug/running", healthcheck)

	s := Server{
		routes: router,
	}

	s.Handler = router

	return &s, nil
}

func home(c web.C, w http.ResponseWriter, r *http.Request) {
	_, err := w.Write([]byte("twitchhttp.Server is alive"))
	if err != nil {
		log.Printf("home page failed to respond: %v", err)
	}
}

func healthcheck(c web.C, w http.ResponseWriter, r *http.Request) {
	w.Header().Add("Cache-Control", "no-cache, no-store, must-revalidate")
	_, err := w.Write([]byte("OK"))
	if err != nil {
		log.Printf("health check failed to respond: %v", err)
	}
}

func (s *Server) createHandler(f HandlerFunc) web.HandlerFunc {
	return func(c web.C, w http.ResponseWriter, r *http.Request) {
		defer func() {
			if p := recover(); p != nil {
				if config.RollbarErrorLogger() != nil {
					config.RollbarErrorLogger().RequestPanic(r, p)
				}
				panic(p)
			}
		}()

		ctx := chitin.Context(w, r)
		f(ctx, c, w, r)
	}
}

// ListenAndServe starts the application server, as well as a debug server for pprof debugging. Go runtime metrics begin emitting to statsd.
func (s *Server) ListenAndServe() error {
	go func() {
		log.Printf("debug listening on %v", config.Resolve("debug-bind-address"))
		log.Print(http.ListenAndServe(config.Resolve("debug-bind-address"), nil))
	}()

	gometrics.Monitor(config.Statsd(), time.Second*5)

	log.Printf("application listening on %v", config.Resolve("bind-address"))
	return http.ListenAndServe(config.Resolve("bind-address"), s)
}

// Delete dispatches to the given handler when the pattern matches and the HTTP method is DELETE.
// See https://godoc.org/github.com/zenazn/goji/web#Mux.Delete for details
func (s *Server) Delete(pattern web.PatternType, f HandlerFunc) {
	s.routes.Delete(pattern, s.createHandler(f))
}

// Get dispatches to the given handler when the pattern matches and the HTTP method is GET.
// See https://godoc.org/github.com/zenazn/goji/web#Mux.Get for details
func (s *Server) Get(pattern web.PatternType, f HandlerFunc) {
	s.routes.Get(pattern, s.createHandler(f))
}

// Handle dispatches to the given handler when the pattern matches, regardless of HTTP method.
// See https://godoc.org/github.com/zenazn/goji/web#Mux.Handle for details
func (s *Server) Handle(pattern web.PatternType, f HandlerFunc) {
	s.routes.Handle(pattern, s.createHandler(f))
}

// Head dispatches to the given handler when the pattern matches and the HTTP method is HEAD.
// See https://godoc.org/github.com/zenazn/goji/web#Mux.Head for details
func (s *Server) Head(pattern web.PatternType, f HandlerFunc) {
	s.routes.Head(pattern, s.createHandler(f))
}

// Options dispatches to the given handler when the pattern matches and the HTTP method is OPTIONS.
// See https://godoc.org/github.com/zenazn/goji/web#Mux.Options for details
func (s *Server) Options(pattern web.PatternType, f HandlerFunc) {
	s.routes.Options(pattern, s.createHandler(f))
}

// Patch dispatches to the given handler when the pattern matches and the HTTP method is PATCH.
// See https://godoc.org/github.com/zenazn/goji/web#Mux.Patch for details
func (s *Server) Patch(pattern web.PatternType, f HandlerFunc) {
	s.routes.Patch(pattern, s.createHandler(f))
}

// Post dispatches to the given handler when the pattern matches and the HTTP method is POST.
// See https://godoc.org/github.com/zenazn/goji/web#Mux.Post for details
func (s *Server) Post(pattern web.PatternType, f HandlerFunc) {
	s.routes.Post(pattern, s.createHandler(f))
}

// Put dispatches to the given handler when the pattern matches and the HTTP method is PUT.
// See https://godoc.org/github.com/zenazn/goji/web#Mux.Put for details
func (s *Server) Put(pattern web.PatternType, f HandlerFunc) {
	s.routes.Put(pattern, s.createHandler(f))
}

// Trace dispatches to the given handler when the pattern matches and the HTTP method is TRACE.
// See https://godoc.org/github.com/zenazn/goji/web#Mux.Trace for details
func (s *Server) Trace(pattern web.PatternType, f HandlerFunc) {
	s.routes.Trace(pattern, s.createHandler(f))
}
