# Canary ASG to allow experimental deploys with minimized impact.

resource "template_file" "canary_app_user_data" {
  template = "../../templates/asg_puppetizer.template"
  vars {
    hostname_prefix = "${var.cluster.hostname}-aws"
    cluster = "${var.cluster.name}"
    environment = "${var.cluster.environment}"
    canary = "true"
  }
}

resource "aws_launch_configuration" "canary_app_conf" {
  name_prefix = "${var.cluster.name}-canary-app"
  image_id = "${var.cluster.ami}"
  instance_type = "${var.cluster.instance_type}"
  security_groups = ["${terraform_remote_state.twitch_web_aws.output.twitch_subnets_sg}"]
  user_data = "${template_file.canary_app_user_data.rendered}"
  iam_instance_profile = "${aws_iam_instance_profile.app.id}"

  root_block_device {
    volume_size = "${var.cluster.root_volume_size}"
  }
}

resource "aws_autoscaling_group" "canary_app_asg" {
  name = "${var.cluster.name}_${var.cluster.environment}_canary_asg"
  vpc_zone_identifier = ["${terraform_remote_state.twitch_web_aws.output.service_a}",
                          "${terraform_remote_state.twitch_web_aws.output.service_b}",
                          "${terraform_remote_state.twitch_web_aws.output.service_c}"]
  max_size = "${var.cluster.canary_size}"
  min_size = "${var.cluster.canary_size}"
  desired_capacity = "${var.cluster.canary_size}"
  health_check_grace_period = 600
  health_check_type = "EC2"
  load_balancers = ["${aws_elb.internal_app_elb.name}"]
  launch_configuration = "${aws_launch_configuration.canary_app_conf.name}"

  tag {
    key = "Name"
    value = "${var.cluster.name}"
    propagate_at_launch = true
  }
  tag {
    key = "Owner"
    value = "${var.cluster.owner}"
    propagate_at_launch = true
  }
  tag {
    key = "Project"
    value = "${var.cluster.project}"
    propagate_at_launch = true
  }
}

module "canary_asg_dns" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/web/asg-r53-lambda.git//modules/notifications"
  topic_name = "${var.cluster.name}_${var.cluster.environment}_canary_topic"
  asg_name = "${aws_autoscaling_group.canary_app_asg.name}"
  lambda_arn = "${terraform_remote_state.production_asg_lambda.output.asg_lambda_arn}"
}
