# Primary application ASG and ELB

resource "template_file" "app_user_data" {
  template = "../../templates/asg_puppetizer.template"
  vars {
    hostname_prefix = "${var.cluster.hostname}-aws"
    cluster = "${var.cluster.name}"
    environment = "${var.cluster.environment}"
    canary = "false"
  }
}

resource "aws_launch_configuration" "app_conf" {
  name_prefix = "${var.cluster.name}-app"
  image_id = "${var.cluster.ami}"
  instance_type = "${var.cluster.instance_type}"
  security_groups = ["${terraform_remote_state.twitch_web_aws.output.twitch_subnets_sg}"]
  user_data = "${template_file.app_user_data.rendered}"
  iam_instance_profile = "${aws_iam_instance_profile.app.id}"

  root_block_device {
    volume_size = "${var.cluster.root_volume_size}"
  }
}

resource "aws_autoscaling_group" "app_asg" {
  name = "${var.cluster.name}_${var.cluster.environment}_asg"
  vpc_zone_identifier = ["${terraform_remote_state.twitch_web_aws.output.service_a}",
                          "${terraform_remote_state.twitch_web_aws.output.service_b}",
                          "${terraform_remote_state.twitch_web_aws.output.service_c}"]
  max_size = "${var.cluster.max_size}"
  min_size = "${var.cluster.min_size}"
  desired_capacity = "${var.cluster.desired}"
  health_check_grace_period = 600
  health_check_type = "EC2"
  load_balancers = ["${aws_elb.internal_app_elb.name}"]
  launch_configuration = "${aws_launch_configuration.app_conf.name}"

  tag {
    key = "Name"
    value = "${var.cluster.name}"
    propagate_at_launch = true
  }
  tag {
    key = "Owner"
    value = "${var.cluster.owner}"
    propagate_at_launch = true
  }
  tag {
    key = "Project"
    value = "${var.cluster.project}"
    propagate_at_launch = true
  }
}

module "asg_dns" {
  source = "git::git+ssh://git@git-aws.internal.justin.tv/web/asg-r53-lambda.git//modules/notifications"
  topic_name = "${var.cluster.name}_${var.cluster.environment}_topic"
  asg_name = "${aws_autoscaling_group.app_asg.name}"
  lambda_arn = "${terraform_remote_state.production_asg_lambda.output.asg_lambda_arn}"
}

resource "aws_elb" "internal_app_elb" {
  name = "${var.cluster.name}-internal-app-elb"
  security_groups = ["${terraform_remote_state.twitch_web_aws.output.twitch_subnets_sg}"]
  subnets = ["${split(",", "${terraform_remote_state.twitch_web_aws.output.service_subnets}")}"]
  internal = true
  cross_zone_load_balancing = true
  connection_draining = true

  listener {
    instance_port = 8000
    instance_protocol = "http"
    lb_port = 80
    lb_protocol = "http"
  }

  health_check {
    healthy_threshold = 3
    unhealthy_threshold = 5
    timeout = 10
    target = "HTTP:8000/debug/running"
    interval = 30
  }

  tags {
    Name = "${var.cluster.name}-internal-app-elb"
    Owner = "${var.cluster.owner}"
    Project = "${var.cluster.project}"
  }
}
