﻿using System;
using System.IO;
using Curse.AGS.FileReaders;
using Curse.AGS.Roles;
using Curse.AGS.Users;
using Curse.AGS.Users.Importers;
using Curse.Cobalt.Models;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Moq;

namespace Curse.AGS.UnitTests
{
    [TestClass]
    public class AGSAlphaUserImporterTests
    {
        [TestMethod]
        public void ImportReturnsFailureIfFileReadErrorOccurrs()
        {
            var ms = new MemoryStream();
            var fileReader = new Mock<IFileReader>();
            fileReader.Setup(x => x.ReadFileDataFromStream<UserToCreate>(ms))
                .Returns(new FileReaderResult<UserToCreate>
                {
                    Message = "",
                    Succeeded = false,
                    Result = null
                });
            var bulkUserCreator = new Mock<IBulkUserCreator>();
            var bulkRoleAssigner = new Mock<IBulkRoleAssigner>();
            var sut = new AGSAlphaUserImporter(fileReader.Object, bulkUserCreator.Object, bulkRoleAssigner.Object);
            var result = sut.ImportAndAssignRoles(ms, new IRole[0], GetMockedSite());
            Assert.IsFalse(result.ImportSucceeded);
        }

        [TestMethod]
        public void ImportReturnsSuccessIfNoErrorOccurrs()
        {
            var ms = new MemoryStream();
            var fileReader = new Mock<IFileReader>();
            fileReader.Setup(x => x.ReadFileDataFromStream<UserToCreate>(ms))
                .Returns(new FileReaderResult<UserToCreate>
                {
                    Message = "",
                    Succeeded = true,
                    Result = null
                });
            var site = GetMockedSite();
            var bulkUserCreatorResult = new BulkUserCreateResult();
            bulkUserCreatorResult.CobaltUserIDs.Add(1);
            var bulkUserCreator = new Mock<IBulkUserCreator>();
            bulkUserCreator.Setup(x => x.BulkCreateUsers(null)).Returns(bulkUserCreatorResult);
            var bulkRoleAssigner = new Mock<IBulkRoleAssigner>();
            bulkRoleAssigner.Setup(x => x.BulkAssignRoles(new[] { 1 }, new IRole[0], site)).Returns(new BulkRoleAssignmentResult());
            var sut = new AGSAlphaUserImporter(fileReader.Object, bulkUserCreator.Object, bulkRoleAssigner.Object);
            var result = sut.ImportAndAssignRoles(ms, new IRole[0], site);
            Assert.IsTrue(result.ImportSucceeded);
            Assert.IsNotNull(result.BulkRoleAssignmentResult);
            Assert.IsNotNull(result.BulkUserCreateResult);
        }

        [ExpectedException(typeof(ArgumentNullException))]
        [TestMethod]
        public void PassingInNullStreamThrowsException()
        {
            var fileReader = new Mock<IFileReader>();
            var bulkUserCreator = new Mock<IBulkUserCreator>();
            var bulkRoleAssigner = new Mock<IBulkRoleAssigner>();
            var sut = new AGSAlphaUserImporter(fileReader.Object, bulkUserCreator.Object, bulkRoleAssigner.Object);
            sut.ImportAndAssignRoles(null, new IRole[0], GetMockedSite());
        }

        [ExpectedException(typeof(ArgumentNullException))]
        [TestMethod]
        public void PassingInNullSiteThrowsException()
        {
            var fileReader = new Mock<IFileReader>();
            var bulkUserCreator = new Mock<IBulkUserCreator>();
            var bulkRoleAssigner = new Mock<IBulkRoleAssigner>();
            var sut = new AGSAlphaUserImporter(fileReader.Object, bulkUserCreator.Object, bulkRoleAssigner.Object);
            sut.ImportAndAssignRoles(new MemoryStream(), new IRole[0], null);
        }

        private static ISite GetMockedSite()
        {
            var site = new Mock<ISite>();
            site.Setup(x => x.ID).Returns(1);
            return site.Object;
        }
    }
}
