﻿using System;
using System.Linq;
using Curse.AGS.Roles;
using Curse.Cobalt.Models;
using Curse.Cobalt.RoleAssignments;
using Curse.Cobalt.Users;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Moq;

namespace Curse.AGS.UnitTests
{
    [TestClass]
    public class BulkRoleAssignerTests
    {
        [TestMethod]
        public void PassingInNullUsersYieldsNoResult()
        {
            var roleAssignmentRepo = new Mock<IRoleAssignmentRepository>();
            var userRepository = new Mock<IUserRepository>();
            var sut = new BulkRoleAssigner(roleAssignmentRepo.Object, userRepository.Object);
            var result = sut.BulkAssignRoles(null, new IRole[0], GetMockedSite());
            Assert.AreEqual(0, result.FailedRoleAssignments.Count);
            Assert.AreEqual(0, result.SuccessfulRoleAssignments.Count);
        }

        [TestMethod]
        public void PassingInNoUsersYieldsNoResult()
        {
            var roleAssignmentRepo = new Mock<IRoleAssignmentRepository>();
            var userRepository = new Mock<IUserRepository>();
            var sut = new BulkRoleAssigner(roleAssignmentRepo.Object, userRepository.Object);
            var result = sut.BulkAssignRoles(new int[0], new IRole[0], GetMockedSite());
            Assert.AreEqual(0, result.FailedRoleAssignments.Count);
            Assert.AreEqual(0, result.SuccessfulRoleAssignments.Count);
        }

        [TestMethod]
        public void PassingInNullRolesYieldsNoResult()
        {
            var roleAssignmentRepo = new Mock<IRoleAssignmentRepository>();
            var userRepository = new Mock<IUserRepository>();
            var sut = new BulkRoleAssigner(roleAssignmentRepo.Object, userRepository.Object);
            var result = sut.BulkAssignRoles(new[] { 1 }, null, GetMockedSite());
            Assert.AreEqual(0, result.FailedRoleAssignments.Count);
            Assert.AreEqual(0, result.SuccessfulRoleAssignments.Count);
        }

        [TestMethod]
        public void PassingInNoRolesYieldsNoResult()
        {
            var roleAssignmentRepo = new Mock<IRoleAssignmentRepository>();
            var userRepository = new Mock<IUserRepository>();
            userRepository.Setup(x => x.GetUserByID(1)).Returns(GetMockedUser(1, "daniel"));
            var sut = new BulkRoleAssigner(roleAssignmentRepo.Object, userRepository.Object);
            var result = sut.BulkAssignRoles(new[] { 1 }, new IRole[0], GetMockedSite());
            Assert.AreEqual(0, result.FailedRoleAssignments.Count);
            Assert.AreEqual(0, result.SuccessfulRoleAssignments.Count);
        }

        [ExpectedException(typeof(ArgumentNullException))]
        [TestMethod]
        public void PassingInNoSiteThrowsException()
        {
            var roleAssignmentRepo = new Mock<IRoleAssignmentRepository>();
            var userRepository = new Mock<IUserRepository>();
            userRepository.Setup(x => x.GetUserByID(1)).Returns(GetMockedUser(1, "daniel"));
            var sut = new BulkRoleAssigner(roleAssignmentRepo.Object, userRepository.Object);
            var result = sut.BulkAssignRoles(new[] { 1 }, new IRole[0], null);
            Assert.AreEqual(0, result.FailedRoleAssignments.Count);
            Assert.AreEqual(0, result.SuccessfulRoleAssignments.Count);
        }

        [TestMethod]
        public void AFailureIsRecordedWhenTryingToAssignAUserToARoleTheyAreAlreadyIn()
        {
            var daniel = GetMockedUser(1, "daniel");
            var admin = GetMockedRole(1, "admin");
            var site = GetMockedSite();
            var roleAssignmentRepo = new Mock<IRoleAssignmentRepository>();
            roleAssignmentRepo.Setup(x => x.IsUserAssignedToARole(daniel, site, admin)).Returns(true);
            var userRepository = new Mock<IUserRepository>();
            userRepository.Setup(x => x.GetUserByID(1)).Returns(daniel);
            var sut = new BulkRoleAssigner(roleAssignmentRepo.Object, userRepository.Object);
            var result = sut.BulkAssignRoles(new[] { 1 }, new[] { admin }, site);
            Assert.AreEqual(1, result.FailedRoleAssignments.Count);
            Assert.AreEqual("daniel is already assigned to 'admin'.", result.FailedRoleAssignments.First());
        }

        [TestMethod]
        public void SuccessfullyAssigningAUserToAGroupAddsToSuccessList()
        {
            var daniel = GetMockedUser(1, "daniel");
            var admin = GetMockedRole(1, "admin");
            var site = GetMockedSite();
            var roleAssignmentRepo = new Mock<IRoleAssignmentRepository>();
            roleAssignmentRepo.Setup(x => x.IsUserAssignedToARole(daniel, site, admin)).Returns(false);
            var userRepository = new Mock<IUserRepository>();
            userRepository.Setup(x => x.GetUserByID(1)).Returns(daniel);
            var sut = new BulkRoleAssigner(roleAssignmentRepo.Object, userRepository.Object);
            var result = sut.BulkAssignRoles(new[] { 1 }, new[] { admin }, site);
            Assert.AreEqual(1, result.SuccessfulRoleAssignments.Count);
            Assert.AreEqual("daniel has been assigned to 'admin'", result.SuccessfulRoleAssignments.First());
            Assert.AreEqual(0, result.FailedRoleAssignments.Count);
        }

        private static ISite GetMockedSite()
        {
            var site = new Mock<ISite>();
            site.Setup(x => x.ID).Returns(1);
            return site.Object;
        }

        private static IUser GetMockedUser(int id, string username)
        {
            var user = new Mock<IUser>();
            user.Setup(x => x.ID).Returns(id);
            user.Setup(x => x.Username).Returns(username);
            return user.Object;
        }

        private static IRole GetMockedRole(int id, string name)
        {
            var role = new Mock<IRole>();
            role.Setup(x => x.ID).Returns(id);
            role.Setup(x => x.Name).Returns(name);
            return role.Object;
        }
    }
}
