﻿using System.Web.Mvc;
using Curse.AGS.Constants;
using Curse.AGS.Web.Forms.AmazonAuth;
using Curse.AGS.Web.ViewModels.AmazonAuthentication;
using Curse.Cobalt.AmazonAuth;
using Curse.Cobalt.Application;
using Curse.Cobalt.Authentication;
using Curse.Cobalt.Cookies;
using Curse.Cobalt.Identification;
using Curse.Cobalt.Models;
using Curse.Cobalt.SessionManagement;
using Curse.Cobalt.Web.ActionFilters;
using Curse.Cobalt.Encryption;
using Curse.Cobalt.Users;
using Curse.Cobalt.Web.Controllers;

namespace Curse.AGS.Web.Controllers
{
    [ExternalAuthProviderIsEnabled(Name = ExternalAuthProvider.AmazonProviderName)]
    [SkipSiteRoleProtectionFilter]
    public class AmazonAuthenticationController : BaseController
    {
        private readonly IExternalUserIdentifier _externalUserIdentifier;
        private readonly IUserCreator _userCreator;
        private readonly IAmazonApi _amazonApi;
        private readonly ICryptographer _cryptographer;
        private readonly IUserSessionManager _userSessionManager;
        private readonly ICookieRepository _cookieRepository;

        public AmazonAuthenticationController(IExternalUserIdentifier externalUserIdentifier, IUserCreator userCreator, IAmazonApi amazonApi, ICryptographer cryptographer, IUserSessionManager userSessionManager, ICookieRepository cookieRepository)
        {
            _externalUserIdentifier = externalUserIdentifier;
            _userCreator = userCreator;
            _amazonApi = amazonApi;
            _cryptographer = cryptographer;
            _userSessionManager = userSessionManager;
            _cookieRepository = cookieRepository;
        }

        [HttpPost]
        public ActionResult CreateUser(FormCollection formCollection)
        {
            var form = new SetDisplayNameForm(null, formCollection);
            if (formCollection == null || !form.Validate())
            {
                return View(AGSViews.Amazon.SetDisplayName, new SetDisplayNameViewModel(null, form));
            }

            var amazonUser = form.GetAmazonUser();
            amazonUser.Email = _cryptographer.DecryptString(amazonUser.Email);
            amazonUser.Name = _cryptographer.DecryptString(amazonUser.Name);
            amazonUser.ID = _cryptographer.DecryptString(amazonUser.ID);
            amazonUser.RefreshToken = _cryptographer.DecryptString(amazonUser.RefreshToken);
            amazonUser.AccessToken = _cryptographer.DecryptString(amazonUser.AccessToken);

            var newCobaltUser = new AmazonUserCreator(_userCreator).Create(amazonUser);
            _userSessionManager.Create(newCobaltUser.ID, amazonUser.AccessToken, amazonUser.RefreshToken, ExternalAuthProvider.AmazonProviderID);

            return Redirect("/");
        }
        [HttpGet]
        public ActionResult LogIn(string returnUrl)
        {
            return Redirect(_amazonApi.GetLogin(returnUrl));
        }
        [HttpGet]
        public ActionResult TokenCallback(string code, string state, string error_description, string error)
        {
            if (error != null)
            {
                OAuthState.ClearState(_cookieRepository);
                return View(AGSViews.Amazon.LogInError, new LogInErrorViewModel());
            }

            var oauthState = OAuthState.ParseStateFromString(state, _cookieRepository);
            var returnUrl = oauthState.ReturnUrl;
            if (!oauthState.Validate())
            {
                OAuthState.ClearState(_cookieRepository);
                return Redirect("/");
            }

            OAuthState.ClearState(_cookieRepository);

            var authResponse = _amazonApi.ProcessAuthorization(code);
            var tokenValidationResult = _amazonApi.ValidateAccessToken(authResponse.AccessToken);
            if (!tokenValidationResult.IsValid)
            {
                if (tokenValidationResult.EncounteredException)
                {
                    CobaltLogger.Instance.LogMessage(tokenValidationResult.Message);
                }

                return Redirect("/");
            }

            var amazonUser = _amazonApi.GetUserData(authResponse.AccessToken);
            var externalUser = _externalUserIdentifier.IdentifyExternalUser(amazonUser.ID, ExternalAuthProvider.AmazonProviderID);
            if (externalUser.IsIdentified)
            {
                // ReSharper disable once PossibleInvalidOperationException
                _userSessionManager.Create(externalUser.UserID.Value, authResponse.AccessToken,
                    authResponse.RefreshToken, ExternalAuthProvider.AmazonProviderID);
            }
            else
            {
                amazonUser.Email = _cryptographer.EncryptString(amazonUser.Email);
                amazonUser.ID = _cryptographer.EncryptString(amazonUser.ID);
                amazonUser.Name = _cryptographer.EncryptString(amazonUser.Name);
                amazonUser.AccessToken = _cryptographer.EncryptString(authResponse.AccessToken);
                amazonUser.RefreshToken = _cryptographer.EncryptString(authResponse.RefreshToken);

                return View(AGSViews.Amazon.SetDisplayName, new SetDisplayNameViewModel(amazonUser, null));
            }

            return Redirect(returnUrl);
        }
    }
}