﻿using System.IO;
using System.Net;
using System.Text;
using Curse.Cobalt.Extensions;
using Curse.Cobalt.Helpers;
using Newtonsoft.Json;
using System.Collections.Specialized;
using Curse.Cobalt.Authentication;
using Curse.Cobalt.Cookies;
using Curse.Cobalt.SessionManagement;

namespace Curse.Cobalt.AmazonAuth
{
    public class AmazonApi : IAmazonApi
    {
        private readonly IAmazonApiUrlHelper _apiUrlHelper;
        private readonly IAmazonApiConfigHelper _configHelper;
        private readonly IUserSessionManager _userSessionManager;

        public AmazonApi(IAmazonApiUrlHelper apiUrlHelper, IAmazonApiConfigHelper configHelper, IUserSessionManager userSessionManager)
        {
            _apiUrlHelper = apiUrlHelper;
            _configHelper = configHelper;
            _userSessionManager = userSessionManager;
        }

        public AmazonTokenResponse ProcessAuthorization(string code)
        {
            using (var client = new WebClient())
            {
                try
                {
                    var postData = new NameValueCollection();
                    postData.Add("grant_type", "authorization_code");
                    postData.Add("code", code);
                    postData.Add("redirect_uri", $"{_configHelper.AmazonRedirectUrl}");
                    postData.Add("client_id", _configHelper.AmazonClientID);
                    postData.Add("client_secret", _configHelper.AmazonClientSecret);
                    var response = client.UploadValues(_apiUrlHelper.TokenExchange, "POST", postData);

                    return JsonConvert.DeserializeObject<AmazonTokenResponse>(Encoding.UTF8.GetString(response));
                }
                catch
                {
                    return null;
                }
            }
        }

        public AmazonUser GetUserData(string accessToken)
        {
            if (accessToken.IsNullOrEmpty())
            {
                return null;
            }

            using (var client = new WebClient())
            {
                try
                {
                    client.Headers.Add("Authorization", "Bearer {0}".FormatWith(accessToken));
                    Stream response;
                    using (CobaltTimer.DebugCreate("Amazon Authentication"))
                    {
                        response = client.OpenRead(_apiUrlHelper.User);
                    }
                    if (response != null)
                    {
                        using (var reader = new StreamReader(response, Encoding.UTF8))
                        {
                            var amazonUser = JsonConvert.DeserializeObject<AmazonUser>(reader.ReadToEnd());
                            amazonUser.DisplayName = string.Empty;
                            return amazonUser;
                        }
                    }

                    return null;
                }
                catch
                {
                    return null;
                }
            }
        }

        public AmazonRevalidationResult ValidateAccessToken(string accessToken)
        {
            using (var client = new WebClient())
            {
                try
                {
                    var response = client.OpenRead(_apiUrlHelper.TokenInfo + accessToken);
                    if (response != null)
                    {
                        using (var reader = new StreamReader(response, Encoding.UTF8))
                        {
                            var tokenInfo = JsonConvert.DeserializeObject<AmazonTokenInfo>(reader.ReadToEnd());
                            return new AmazonRevalidationResult
                            {
                                IsValid = tokenInfo.ClientId == _configHelper.AmazonClientID
                            };
                        }
                    }
                    return new AmazonRevalidationResult { IsValid = false };
                }
                catch (WebException ex)
                {
                    return new AmazonRevalidationResult
                    {
                        IsValid = false,
                        EncounteredException = true,
                        Message = ex.Message
                    };
                }
            }
        }

        public string GetLogin(string returnUrl)
        {
            OAuthState.CreateState(new CookieRepository());
            return _apiUrlHelper.GetLogin(returnUrl);
        }

        public AmazonTokenResponse RefreshToken(string refreshToken)
        {
            using (var client = new WebClient())
            {
                try
                {
                    var postData = new NameValueCollection();
                    postData.Add("grant_type", "refresh_token");
                    postData.Add("refresh_token", refreshToken);
                    postData.Add("client_id", _configHelper.AmazonClientID);
                    postData.Add("client_secret", _configHelper.AmazonClientSecret);
                    
                    var response = client.UploadValues(_apiUrlHelper.TokenExchange, "POST", postData);

                    return JsonConvert.DeserializeObject<AmazonTokenResponse>(Encoding.UTF8.GetString(response));
                }
                catch
                {
                    return null;
                }
            }
        }

        public bool Revalidate()
        {
            var session = _userSessionManager.GetCurrent();
            var validationResult = ValidateAccessToken(session.ExternalAuthToken);
            if(!validationResult.IsValid)
            {
                var token = RefreshToken(session.ExternalRefreshToken);

                if (token != null)
                {
                    if (ValidateAccessToken(token.AccessToken).IsValid)
                    {
                        session.UpdateTokens(token.AccessToken, token.RefreshToken);
                        _userSessionManager.Save(session);
                        return true;
                    }
                }
                return false;
            }
            return true;
        }
    }

    public interface IAmazonApi
    {
        string GetLogin(string returnUrl);
        AmazonUser GetUserData(string accessToken);
        AmazonRevalidationResult ValidateAccessToken(string accessToken);
        AmazonTokenResponse ProcessAuthorization(string code);
        bool Revalidate();

    }

    public class AmazonTokenInfo
    {
        [JsonProperty(PropertyName = "aud")]
        public string ClientId { get; set; }
        [JsonProperty(PropertyName = "user_id")]
        public string UserId { get; set; }
        [JsonProperty(PropertyName = "iss")]
        public string TokenIssuer { get; set; }
        [JsonProperty(PropertyName = "exp")]
        public int TokenExpiration { get; set; }
        [JsonProperty(PropertyName = "app_id")]
        public string ApplicationId { get; set; }
        [JsonProperty(PropertyName = "iat")]
        public int IssuedAt { get; set; }

    }

    public class AmazonTokenResponse
    {
        [JsonProperty(PropertyName = "access_token")]
        public string AccessToken { get; set; }
        [JsonProperty(PropertyName = "token_type")]
        public string TokenType { get; set; }
        [JsonProperty(PropertyName = "Refresh_token")]
        public string RefreshToken { get; set; }
        [JsonProperty(PropertyName = "expires_in")]
        public int TokenExpiration { get; set; }
        [JsonProperty(PropertyName = "scope")]
        public string Scope { get; set; }
        
    }

    public class AmazonTokenRequest
    {
        [JsonProperty(PropertyName = "grant_type")]
        public string GrantType { get; set; }
        [JsonProperty(PropertyName = "code")]
        public string Code { get; set; }
        [JsonProperty(PropertyName = "client_id")]
        public string ClientID { get; set; }
        [JsonProperty(PropertyName = "client_secret")]
        public string ClientSecret { get; set; }
        [JsonProperty(PropertyName = "redirect_uri")]
        public string RedirectUri { get; set; }
        
    }

    public struct AmazonRevalidationResult
    {
        public bool IsValid { get; set; }
        public string Message { get; set; }
        public bool EncounteredException { get; set; }
    }
}
