﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using Curse.AGS.FileReaders;
using Curse.AGS.Roles;
using Curse.Cobalt.Models;

namespace Curse.AGS.LwaDirectedIds
{
    public interface ILwaPermissionAssigner
    {
        LwaPermissionAssignerResult AssignPermissions(IEnumerable<Role> roles, Stream stream, Site site);
    }

    public class LwaPermissionAssigner : ILwaPermissionAssigner
    {
        public LwaPermissionAssigner(IFileReader csvFileReader, IBulkRoleAssigner bulkRoleAssigner)
        {
            this.CsvFileReader = csvFileReader;
            this.BulkRoleAssigner = bulkRoleAssigner;
        }

        private IFileReader CsvFileReader { get; }

        private IBulkRoleAssigner BulkRoleAssigner { get; }

        public LwaPermissionAssignerResult AssignPermissions(IEnumerable<Role> roles, Stream stream, Site site)
        {
            if (stream == null)
            {
                throw new ArgumentNullException(nameof(stream));
            }

            if (site == null)
            {
                throw new ArgumentNullException(nameof(site));
            }

            var csvReaderResult = this.CsvFileReader.ReadFileDataFromStream<DirectedIdRow>(stream);
            if (!csvReaderResult.Succeeded)
            {
                return new LwaPermissionAssignerResult
                {
                    ImportSucceeded = false,
                    Messages = new []{ csvReaderResult.Message }
                };
            }

            var directedIdRows = csvReaderResult.Result;
            var directedIds = directedIdRows.Select(x => x.DirectedId).ToList();
            var cobaltUserIDs = this.GetUserIdsByDirectedIds(directedIds);
            var bulkRoleAssignmentResult = this.BulkRoleAssigner.BulkAssignRoles(cobaltUserIDs, roles, site);

            var resultMessages = new List<string>
            {
                $"Imported {bulkRoleAssignmentResult.SuccessfulRoleAssignments.Count} successfully."
            };
            if (bulkRoleAssignmentResult.FailedRoleAssignments.Any())
            {
                resultMessages.Add("Failed to import the following role assignments:");
                resultMessages.AddRange(bulkRoleAssignmentResult.FailedRoleAssignments);
            }

            var notFoundDirectedIds = GetDirectedIdsNotFound(directedIds);
            if (notFoundDirectedIds.Any())
            {
                resultMessages.Add($"Directed IDs not found: {string.Join(", ", notFoundDirectedIds)}.");
            }

            return new LwaPermissionAssignerResult
            {
                ImportSucceeded = true,
                Messages = resultMessages
            };
        }

        private IList<int> GetUserIdsByDirectedIds(IEnumerable<string> directedIds)
        {
            return (from e in ExternalUserMap.Query
                where directedIds.Contains(e.ExternalUserID)
                select e.UserID).ToList();
        }

        private IList<string> GetDirectedIdsNotFound(IEnumerable<string> directedIds)
        {
            return (from d in directedIds
                where !ExternalUserMap.Query.Any(x => x.ExternalUserID == d)
                select d).ToList();
        }
    }

    public class DirectedIdRow
    {
        public string DirectedId { get; set; }
    }

    public class LwaPermissionAssignerResult
    {
        public bool ImportSucceeded { get; set; }

        public IEnumerable<string> Messages { get; set; }
    }
}
