##########################################################################################################
# Helper functions used in the build process
##########################################################################################################

import sys
import os
import imp
import git
import pprint
import platform
import sys
import time

import build_types


def add_common_argparse_parameters(parser):

    parser.add_argument(
        '--module',
        required=True,
        metavar='<module>',
        action='append',
        help='Specifies a module to generate the project file for.'
    )

    parser.add_argument(
        '--platform',
        required=True,
        metavar='<platform>',
        help='Specifies the platform support to include.'
    )

    parser.add_argument(
        '--target',
        required=False,
        metavar='<target>',
        help='Specifies the optional target if multiple exist for a platform.'
    )

    parser.add_argument(
        '--compiler',
        required=True,
        metavar='<compiler>',
        help='Specifies the compiler the project file should be generated for.'
    )

    parser.add_argument(
        '--generator',
        required=False,
        metavar='<generator>',
        help='Specifies the type of project generator to use.'
    )

    parser.add_argument(
        '--cmake-no-generate',
        required=False,
        action='store_true',
        help="Creates CMakeLists.txt files but don't run cmake on them."
    )

    parser.add_argument(
        '--cmake-path',
        required=False,
        metavar='<path>',
        help="Specifies a custom path where cmake can be found."
    )

    parser.add_argument(
        '--architecture',
        '--arch',
        required=True,
        metavar='<architecture>',
        help='Specifies the architecture to be included in the project.'
    )

    parser.add_argument(
        '--output-dir',
        required=True,
        metavar='<output_dir>',
        help='Specifies the output directory for the final project files.'
    )

    parser.add_argument(
        '--project-name',
        required=True,
        metavar='<project_name>',
        help='Specifies the name of the final project file.'
    )

    parser.add_argument(
        '--language',
        required=False,
        metavar='<language>',
        help='Specifies the language binding to be included.'
    )

    parser.add_argument(
        '--feature',
        required=False,
        metavar='<module>',
        action='append',
        help='Specifies an additional feature to be included.'
    )

    parser.add_argument(
        '--fragments-dir',
        required=False,
        metavar='<fragments_path>',
        action='append',
        help='Specifies a directory to be used to find source fragments.  This should be the parent directory of the fragment directories, often the root of the repository.'
    )

    parser.add_argument(
        '--root-dir',
        required=False,
        metavar='<root_path>',
        action='append',
        help='Specifies a directory that build_tools will run on (it should have a modules directory where modules are located).'
    )

    parser.add_argument(
        '--force-monolithic-project',
        required=False,
        metavar='<true|false>',
        help='Whether or not to force the generation to use a single project for all source files.'
    )

    parser.add_argument(
        '--use-relative-paths',
        required=False,
        metavar='<true|false>',
        help='Whether or not to use relative paths to the output directory in project files'
    )

    parser.add_argument(
        '-settings',
        nargs='*',
        action='append',
        help='Optional settings for configuring the generation of the build.'
    )


def resolve_path(path):
    path = os.path.expanduser(path)
    path = os.path.abspath(path)
    return path


def parse_common_argparse_parameters(args, options):

    options.architecture = args.architecture
    options.compiler = args.compiler
    options.generator = args.generator
    options.output_dir = resolve_path(args.output_dir)
    options.project_name = args.project_name
    options.language = args.language

    if options.language == 'java' and 'jni' not in options.features:
        options.features.append('jni')

    if args.force_monolithic_project:
        options.monolithic_project = args.force_monolithic_project

    if args.use_relative_paths:
        options.relative_paths = args.use_relative_paths

    options.primary_platform = args.platform
    options.platforms.append(args.platform)

    if args.target:
        options.target = args.target

    if options.architecture == 'x86_64':
        options.architecture = 'x64'

    if not args.root_dir is None:
        for x in args.root_dir:
            options.root_paths.append( resolve_path(x) )

    if not args.module is None:
        for x in args.module:
            options.modules.append(x)

    if not args.fragments_dir is None:
        for x in args.fragments_dir:
            options.fragment_search_paths.append( resolve_path(x) )

    if not args.feature is None:
        for x in args.feature:
            options.features.append(x)

    # Special CMake options
    if args.cmake_no_generate:
        options.cmake_options.no_generate = args.cmake_no_generate
    if args.cmake_path:
        options.cmake_options.custom_path = args.cmake_path

    #print(args)

    if not args.settings is None:
        for arr in args.settings:
            for a in arr:
                kvp = a.split('=')
                if len(kvp) > 1:
                    options.settings[kvp[0]] = kvp[1]
                else:
                    options.settings[kvp[0]] = "1"


def setup_options(options, main_script_file_path, command_line_arguments):

    # Capture paths
    options.main_build_path = os.path.realpath(os.path.dirname(main_script_file_path))
    options.main_root_path = os.path.realpath( os.path.join(options.main_build_path, '..') )

    # Add the main root path as the first entry
    if options.main_root_path in options.root_paths:
        options.root_paths.remove(options.main_root_path)
    options.root_paths.insert(0, options.main_root_path)

    options.common_dependencies_path = os.path.realpath( os.path.join(options.main_root_path, 'modules', 'core', 'dependencies') )
    options.gtest_path = os.path.realpath( os.path.join(options.common_dependencies_path, 'gtest') )

    print('main_build_path: ' + options.main_build_path)
    for path in options.root_paths:
        print('root_path: ' + path)

    # Reconstruct the command line arguments
    options.command_line = reconstruct_command_line_string(os.path.basename(main_script_file_path), command_line_arguments)

    # Add the Core module as the first entry
    if 'core' in options.modules:
        options.modules.remove('core')
    options.modules.insert(0, 'core')

    # Automatically add fragments for included modules
    for module in options.modules:
        found = False
        for root_path in options.root_paths:
            fragment_path = resolve_path(os.path.join(root_path, 'modules', module))
            if os.path.isdir(fragment_path):
                options.fragment_search_paths.append(fragment_path)
                found = True
        if not found:
            print("Invalid module specified: " + module)
            sys.exit(1)

    # Verify fragment search paths exist
    verify_fragment_search_paths_exist(options)

    # Determine the git hashes of all the repos
    find_repositories(options)

    # Generate the version string
    options.version_string = generate_version_string(options)

    # Load the settings that are specific to the platform
    options.platform_settings = load_platform_settings(options.primary_platform, options)

    # Make sure the output directory exists
    if not os.path.exists(options.output_dir):
        os.makedirs(options.output_dir)


def load_platform_settings(platform, options):

    """Searches all build/platforms directories for the platform_<platform>.py module, loads it and returns the instance."""

    platform_settings = None

    prefix = 'platform_' + platform
    filename = prefix + '.py'

    for search_path in options.root_paths:

        path = os.path.join(search_path, 'build', 'platforms', filename)

        if os.path.isfile(path):
            print("Loading platform settings: " + path + "...")
            mod = imp.load_source(prefix, path)
            platform_settings = mod.get_platform_settings()
            print("Done loading " + path + ".")

    if platform_settings is None:
        raise Exception('Could not find platform settings for platform: ' + platform)

    return platform_settings


def determine_required_fragments(module_name, options):

    """Given the name of the module and the desired options, returns the list of fragment names to search for in the fragment search paths."""

    fragment_names = []

    fragment_names = [module_name + '_common']

    for platform in options.platforms:
        fragment_names.append(module_name + '_' + platform)

    for feature in options.features:
        fragment_names.append(module_name + '_' + feature)

    if options.language:
        fragment_names.append(module_name + '_' + options.language)

    return fragment_names


def find_fragment_directory(options, fragment_name):

    """Searches all the fragment search paths for the given fragment. Returns the full path to the fragment or None if not found."""

    for search_path in options.fragment_search_paths:

        dirs = os.listdir(search_path)

        for dir in dirs:
            if dir == fragment_name:
                return os.path.join(search_path, dir)

    return None


def find_repository_root(path):

    """Find the Git repository root directory based on a path within the repository."""

    path = os.path.abspath(path)
    print(path)

    """Get the nearest git repo from the path, searching its parent directories"""
    try:
        git_repo = git.Repo(path, search_parent_directories=True)
        git_root = git_repo.git.rev_parse("--show-toplevel")
        return git_root
    except:
        print("Could not find root: " + path)

    return None


def find_repositories(options):

    """Finds all Git repositories associated with the fragments search paths."""

    # Determine the git hashes of all the repos
    for path in options.fragment_search_paths:
        commit_hash = "?????????????"
        commit_date = '????????'
        name = ""
        # Try and determine if in a git repository
        repo_root = find_repository_root(path)

        remote = None
        try:
            repo = git.Repo(repo_root)
            commit_hash = repo.head.commit.hexsha
            commit_date = repo.head.commit.committed_date
            t = time.gmtime(commit_date)
            commit_date = time.strftime('%Y%m%d', t)
            remote = repo.remotes.origin.url
        except:
            print('Failed to extract repository info for: ' + path)

        exists = False
        for entry in options.repositories:
            if entry.path == repo_root:
                exists = True
                break

        if not exists:
            entry = build_types.SourceRepository(remote, repo_root, commit_hash, commit_date)
            options.repositories.append(entry)

    options.repositories.sort(lambda a,b: cmp(a.remote, b.remote))


def reconstruct_command_line_string(script_name, command_line_arguments):

    """Recreates an approximation of the the command line string that was used to execute the script."""

    # Reconstruct the command line arguments
    if command_line_arguments is None:
        command_line_arguments = sys.argv[1:]

    str = ''
    for x in command_line_arguments:
        if str != '':
            str = str + ' '
        str = str + '\"' + x + '\"'

    return script_name + ' ' + str


def verify_fragment_search_paths_exist(options):

    """Aborts the generation if any of the fragment search paths do not exist."""

    # Verify fragment search paths exist
    for path in options.fragment_search_paths:
        if not os.path.isdir(path):
            print("Invalid fragment search path specified: " + path)
            sys.exit(1)


def get_short_commit_hash(hash):
    return hash[0:8]


def generate_version_string(options):

    """Builds the version string for the project."""

    if len(options.repositories) == 1:
        repo = options.repositories[0]
        version_string = repo.commit_date + '_' + get_short_commit_hash(repo.commit_hash)

    else:
        version_string = options.project_name
        for repo in options.repositories:
            version_string = version_string + '_' + repo.commit_date + '_' + get_short_commit_hash(repo.commit_hash)

    return version_string


def add_global_preprocessor_definitions(module, output_object):

    options = output_object.options

    module.add_global_preprocessor_definition('TTV_OUTPUT_' + options.output_object_type.upper())
    module.add_global_preprocessor_definition('TTV_PLATFORM_' + options.primary_platform.upper())
    module.add_global_preprocessor_definition('TTV_ENABLE_ASSERTS', debug=True, release=False)

    if options.target:
        module.add_global_preprocessor_definition('TTV_TARGET_' + options.target.upper())


def load_module(module_name, project_name, output_object):

    """Generates the project for the given module."""

    print('Generating module project: ' + module_name)

    # Determine which fragments will be needed
    fragment_names = determine_required_fragments(module_name, output_object.options)

    # Determine the project location
    source_module = build_types.SourceModule(module_name)
    source_module.project_name = project_name

    # Process each fragment
    for fragment_name in fragment_names:

        # Find the fragment source directory
        fragment_dir = find_fragment_directory(output_object.options, fragment_name)

        if fragment_dir is None:
            print("Could not find fragment directory for fragment, skipping: " + fragment_name)
            continue

        fragment_file_path = os.path.join(fragment_dir, 'fragment.py')
        if not os.path.isfile(fragment_file_path):
            print("Could not find fragment.py for fragment, skipping: " + fragment_dir)
            continue

        print('Found "' + fragment_name + '" fragment at ' + fragment_dir)

        # Load the fragment file
        print("Loading " + fragment_file_path + "...")

        python_module = imp.load_source('fragment_' + fragment_name, fragment_file_path)
        fragment = output_object.load_fragment_file(python_module)

        if not fragment is None:
            print("Done loading " + fragment_file_path + ".")
            source_module.add_fragment(fragment)

            output_object.options.platform_settings.add_platform_preprocessor_definitions(output_object.options, fragment)

            #fragment.debug_print()

        else:
            print("Fragment does not contain relevant entry-point, skipping.")

    add_global_preprocessor_definitions(source_module, output_object)

    return source_module


def load_twitchsdk_fragment(python_module, primitives, options):
    if hasattr(python_module, 'load_twitchsdk_fragment'):
        return python_module.load_twitchsdk_fragment(primitives, options)
    else:
        return None


def generate_module_project(module, output_object):
    generator = output_object.options.platform_settings.create_build_generator(output_object.options)

    # Determine where to place the project
    if generator.output_hierarchical_structure():
        module.project_directory = os.path.join(output_object.options.output_dir, module.project_name)
    else:
        module.project_directory = output_object.options.output_dir

    # Create the project directory if needed
    if not os.path.exists(module.project_directory):
        os.makedirs(module.project_directory)

    # Update precompiled header mappings
    module.create_precompiled_header_mappings()

    # Produce generator files
    generator.produce_module_generator_files(module, module.project_directory, output_object)


def write_header_file_common_preamble(file, modules, options):

    file.write("/***************************************************************************************************************************************\n")
    file.write(" * \n")
    file.write(" * This file was auto-generated using Twitch build tools - Do not modify by hand\n")
    file.write(" *\n")
    file.write(" * Command line:\n")
    file.write(" *\n")
    file.write(" *  " + options.command_line + "\n")
    file.write(" *\n")
    file.write(" * Project name: " + options.project_name + "\n")
    file.write(" *\n")

    file.write(" * Version string: \"" + options.version_string + "\"\n")
    file.write(" *\n")

    if len(options.repositories) == 0:
        file.write(" * Git Repositories: None\n")

    else:
        file.write(" * Git Repositories:\n")
        for repo in options.repositories:
            file.write(" *   " + str(repo.remote) + ":\n")
            file.write(" *     Path: " + str(repo.path) + "\n")
            file.write(" *     Commit: " + str(repo.commit_hash) + "\n")
            file.write(" *\n")
    file.write(" *\n")

    file.write(" * Included fragments:\n")
    for mod in modules:
        for fragment in mod.fragments:
            file.write(" *   " + fragment.name + "\n")
    file.write(" *\n")


def write_header_file_preamble_done(file):

    file.write(" *\n")
    file.write(" ***************************************************************************************************************************************/\n")


def find_executable(program):

    """
    Attempts to find the named program similar to a 'where' command.
    """

    fpath, fname = os.path.split(program)

    def is_exe(fpath):
        return (not fpath is None) and os.path.isfile(fpath) and os.access(fpath, os.X_OK)

    if fpath:
        if is_exe(program):
            return program
    else:
        for path in os.environ["PATH"].split(os.pathsep):
            path = path.strip('"')
            exe_file = os.path.join(path, program)
            if is_exe(exe_file):
                return exe_file

    return None


def find_cmake(custom_path=None):

    """
    Tries to find cmake on the system.  If custom_path is not None then it will be the only directory searched.
    """

    result = None

    # First check for an explicit path to CMake
    if custom_path:
        if platform.system() == 'Windows':
            result = os.path.join(custom_path, 'cmake.exe')
        else:
            result = os.path.join(custom_path, 'cmake')

        result = find_executable(result)
        if result:
            return result
        else:
            raise Exception('Explicit CMake path seems invalid: ' + custom_path)

    # Gather a bunch of reasonable places for CMake to be found
    guesses = []

    if platform.system() == 'Windows':
        guesses = guesses + [
            'cmake.exe', # System path
            'C:/Program Files (x86)/CMake/bin/cmake.exe',
            'C:/Program Files/CMake/bin/cmake.exe'
        ]

    else:
        guesses = guesses + [
            'cmake', # System path
            '/Applications/CMake.app/Contents/bin/cmake'
        ]

    # Check all the guesses
    for path in guesses:
        path = find_executable(path)
        if path:
            return path

    raise Exception('Could not find cmake')


"""
Adds source files to the fragment.

source_dir: The location of the source files, relative to the fragment. File paths must use forward slash (/).
file_extension_list: Lists the file extensions that will be added to the group. Each element should start with ".", e.g. ('.cpp').
source_group_root_dir: The source group file location within the fragment. File paths must use forward slash (/).
"""
def add_source_files_recursively(fragment, source_dir, file_extension_list, source_group_root_dir):
    for root, _, _ in os.walk(os.path.join(fragment.root_path, source_dir)):
        files = []
        for file_extension in file_extension_list:
            files += fragment.glob_source_files(os.path.join(root, '*' + file_extension))

        inner_folder_path = root.split(source_dir, 1)[1]
        if inner_folder_path:
            inner_folder_path = inner_folder_path[1:] # cut off first slash character
            group_dir = os.path.join(source_group_root_dir, inner_folder_path).replace('\\', '/')
            fragment.add_source_group(group_dir, files)
        else:
            fragment.add_source_group(source_group_root_dir, files)
