##########################################################################################################
# Types used in the build process
##########################################################################################################

import pprint
import os
import glob


class Primitives:

    def __init__(self):

        self.table = []

        # TODO: These could be moved to the Core module
        self.register_primitive_type('MEMORY', None, None, None)
        self.register_primitive_type('ASSERT', None, None, None)
        self.register_primitive_type('RANDOM', None, None, None)
        self.register_primitive_type('MUTEX_FACTORY', 'ttv::IMutexFactory', 'ttv::SetMutexFactory', None)
        self.register_primitive_type('THREAD_FACTORY', 'ttv::IThreadFactory', 'ttv::SetThreadFactory', None)
        self.register_primitive_type('THREAD_SYNC_FACTORY', 'ttv::IThreadSyncFactory', 'ttv::SetThreadSyncFactory', None)
        self.register_primitive_type('THREAD_API', 'ttv::IThreadApi', 'ttv::SetThreadApi', None)
        self.register_primitive_type('SYSTEM_CLOCK', 'ttv::ISystemClock', 'ttv::SetSystemClock', None)
        self.register_primitive_type('TRACER', 'ttv::ITracer', 'ttv::SetTracer', None)
        self.register_primitive_type('HTTP_REQUEST', 'ttv::HttpRequest', 'ttv::SetHttpRequest', None)
        self.register_primitive_type('BACKGROUND_EVENTSCHEDULER_FACTORY', 'ttv::IBackgroundEventSchedulerFactory', 'ttv::SetBackgroundEventSchedulerFactory', None)
        self.register_primitive_type('MAIN_EVENTSCHEDULER_FACTORY', 'ttv::IMainEventSchedulerFactory', 'ttv::SetMainEventSchedulerFactory', None)
        self.register_primitive_type('RAW_SOCKET_FACTORY', 'ttv::ISocketFactory', 'ttv::RegisterSocketFactory', 'ttv::UnregisterSocketFactory', "No default ISocketFactory specified for \"raw\" / \"tcp\", it's expected the app will provide it")
        self.register_primitive_type('TLS_SOCKET_FACTORY', 'ttv::ISocketFactory', 'ttv::RegisterSocketFactory', 'ttv::UnregisterSocketFactory', "No default ISocketFactory specified for \"tls\", it's expected the app will provide it")
        self.register_primitive_type('WS_SOCKET_FACTORY', 'ttv::IWebSocketFactory', 'ttv::RegisterWebSocketFactory', 'ttv::UnregisterWebSocketFactory', "No default IWebSocketFactory specified for \"ws\", it's expected the app will provide it")


    def register_primitive_type(self, name, interface_name, registration_function, unregistration_function=None, missing_message=None):

        if missing_message is None:
            if not interface_name is None:
                missing_message = "No default " + interface_name + " specified, it's expected the app will provide it"

            else:
                missing_message = "No default " + name + " specified, it's expected the app will provide it"

        self.table.append({
            'name': name,
            'variable_name': 'g_' + name,
            'interface': interface_name,
            'registration_function': registration_function,
            'unregistration_function': unregistration_function,
            'header': None,
            'class': None,
            'implemented': False,
            'missing_message': missing_message,
            'flag_code': [],
            'init_code': [],
            'shutdown_code': []
        })


    def set(self, name, klass=None, header=None, flag_code=None, init_code=None, shutdown_code=None):

        entry = self.find(name)

        if entry is None:
            raise Exception("Primitive not registered: " + name)

        entry['implemented'] = True
        entry['header'] = header
        entry['class'] = klass
        entry['flag_code'] = flag_code
        entry['init_code'] = init_code
        entry['shutdown_code'] = shutdown_code


    def find(self, name):

        for x in self.table:
            if x['name'] == name:
                return x

        return None


    def implemented(self, name):

        p = self.find(name)

        if p is None:
            raise Exception("Primitive not registered: " + name)

        return p['implemented']


    def debug_print(self):
        pp = pprint.PrettyPrinter(indent=4)

        print ('Primitives:')
        for p in self.table:
            pp.pprint(p)
        print ("")


class CMakeOptions:

    def __init__(self):

        self.no_generate = False # Don't run cmake on the CMakeLists.txt files that are written
        self.custom_path = None # A custom path where cmake can be found


class BaseOptions:

    def __init__(self):

        self.main_build_path = None
        self.common_dependencies_path = None
        self.gtest_path = None
        self.root_paths = [] # The directories where modules will be found
        self.main_root_path = None

        self.command_line = None # The original command line used
        self.modules = []
        self.external_projects = []
        self.primary_platform = None
        self.platforms = []
        self.platform_settings = None
        self.target = None
        self.compiler = None
        self.generator = None
        self.language = None # The language binding
        self.settings = {}
        self.features = [] # Additional features
        self.architecture = None
        self.output_object_type = 'static_library' # static_library, dynamic_library, executable
        self.fragment_search_paths = [] # The directories to search for source fragments
        self.output_dir = None
        self.project_name = None
        self.fragments = [] # The computed list of required fragments
        self.repositories = [] # SourceRepository instances
        self.version_string = None
        self.monolithic_project = False # Whether or not to compile all source into one project
        self.relative_paths = False # Whether or not to use relative paths to the output directory in project files
        self.no_primitive_registration = False # Whether or not if primitive components should not be registered
        self.cmake_options = CMakeOptions()

    def add_external_project(self, project_directory):
        project_directory = os.path.abspath(project_directory).replace('\\', '/')
        self.external_projects.append(project_directory)

    def has_setting(self, name):
        if not name in self.settings:
            return False

        value = self.settings[name]
        if value is None or value == 0 or value == "0" or value == False or value == "False":
            return False

        else:
            return True


    def print_options(self):

        print("")
        print("Modules:               " + " ".join(str(x) for x in self.modules))
        print("Primary platform:      " + self.primary_platform)
        print("Platforms:             " + " ".join(str(x) for x in self.platforms))
        print("Compiler:              " + self.compiler)
        print("Generator:             " + (self.generator if self.generator else '<Default>'))
        print("Target:                " + (self.target if self.target else '<None>'))
        print("Architecture:          " + self.architecture)
        print("Output type:           " + self.output_object_type)
        print("Language binding:      " + (self.language if self.language else '<None>'))
        print("Features:              " + ( (" ".join(str(x) for x in self.features)) if len(self.features) > 0 else '<None>') )
        print("Project name:          " + self.project_name)
        print("Monolithic project:    " + str(self.monolithic_project))
        print("Relative paths:        " + str(self.relative_paths))
        print("No primitive registration: " + str(self.no_primitive_registration))
        print("Output directory:      " + self.output_dir)
        print("Fragment search paths:")
        for path in self.fragment_search_paths:
            print("                       " + path)
        print("")


class SourceRepository:

    """Information about a source repository."""

    def __init__(self, remote, path, commit_hash, commit_date):

        self.remote = remote
        self.path = path
        self.commit_hash = commit_hash
        self.commit_date = commit_date


class SourceFragment:

    def __init__(self, context, name, fragment_root_path):

        self.context = context # What feature the fragment is for, such as 'twitchsdk', 'unittest', etc.
        self.name = name
        self.root_path = fragment_root_path
        self.source_files = []
        self.header_files = []
        self.source_groups = {}
        self.header_search_paths = []
        self.dependency_search_paths = []
        self.link_libraries = [] # The names of libraries to pass to the linker
        self.preprocessor_defines = {}
        self.preprocessor_defines['common'] = {}
        self.preprocessor_defines['debug'] = {}
        self.preprocessor_defines['release'] = {}
        self.precompiled_header_h = None # The path to the general precompiled header
        self.precompiled_header_cpp = None  # The path to the precompiled source file backing the header
        self.symbol_exports = [] # Explicitly named symbols to export

    def get_fragment_relative_path(self, path):
        return os.path.relpath(path, self.root_path)

    def get_fragment_include_relative_path(self, path):
        return os.path.relpath( path, os.path.join(self.root_path, 'include') )

    def get_fragment_source_relative_path(self, path):
        return os.path.relpath( path, os.path.join(self.root_path, 'source') )

    def get_header_search_paths(self, public_only=False):
        paths = []
        for path in self.header_search_paths:
            if path not in paths:
                if not public_only or path['is_public']:
                    paths.append(path['path'])
        return paths

    def glob_source_files(self, pattern):
        files = glob.glob(pattern)
        self.add_source_files(files)
        return files

    def glob_header_files(self, pattern):
        files = glob.glob(pattern)
        self.add_header_files(files)
        return files

    def get_source_files(self):
        return self.source_files[:]

    def add_source_files(self, paths):

        if not isinstance(paths, list):
            paths = [paths]

        for path in paths:
            path = os.path.abspath(path).replace('\\', '/')
            if not path in self.source_files:
                self.source_files.append(path)

    def get_header_files(self):
        return self.header_files[:]

    def add_header_files(self, paths):

        if not isinstance(paths, list):
            paths = [paths]

        for path in paths:
            path = os.path.abspath(path).replace('\\', '/')
            if not path in self.header_files:
                self.header_files.append(path)

    def add_header_search_paths(self, paths, is_public=False):
        """
        Add header include paths to the module project.
        Public header search paths are exported to dependent libraries.
        """
        if not isinstance(paths, list):
            paths = [paths]

        for path in paths:
            path = os.path.abspath(path).replace('\\', '/')
            if not path in self.header_search_paths:
                path = { 'path': path, 'is_public': is_public }
                self.header_search_paths.append(path)

    def add_dependency_search_paths(self, paths):

        if not isinstance(paths, list):
            paths = [paths]

        for path in paths:
            path = os.path.abspath(path).replace('\\', '/')
            if not path in self.dependency_search_paths:
                self.dependency_search_paths.append(path)

    def add_link_libraries(self, libraries, transform_to_absolute_path=True):

        if not isinstance(libraries, list):
            libraries = [libraries]

        for path in libraries:
            if transform_to_absolute_path:
                path = os.path.abspath(path).replace('\\', '/')
            if not path in self.link_libraries:
                self.link_libraries.append(path)

    def add_preprocessor_definition(self, key, value=None, debug=True, release=True):

        if value is None:
            value = 1

        if debug and release:
            self.preprocessor_defines['common'][key] = value
        elif debug:
            self.preprocessor_defines['debug'][key] = value
        elif release:
            self.preprocessor_defines['release'][key] = value

    def add_source_group(self, group, paths):

        if not isinstance(paths, list):
            paths = [paths]

        if not group in self.source_groups:
            self.source_groups[group] = []

        for path in paths:
            path = os.path.abspath(path).replace('\\', '/')
            self.source_groups[group].append(path)

    def add_symbol_export_files(self, files):

        if not isinstance(files, list):
            files = [files]

        for path in files:
            with open(path) as file:
                lines = file.read().splitlines()
            for symbol in lines:
                self.add_symbol_export(symbol.strip())

    def add_symbol_export(self, symbol_name):

        if symbol_name != '' and not symbol_name in self.symbol_exports:
            self.symbol_exports.append(symbol_name)

    def get_preprocessor_definitions(self):
        return self.preprocessor_defines.copy()

    def get_precompiled_header_h(self):
        return self.precompiled_header_h

    def get_precompiled_header_cpp(self):
        return self.precompiled_header_cpp

    def set_precompiled_header(self, pch_h_path, pch_cpp_path):
        self.precompiled_header_h = pch_h_path
        self.precompiled_header_cpp = pch_cpp_path

    def debug_print(self):
        pp = pprint.PrettyPrinter(indent=4)

        print ('Fragment: ' + self.name)
        print ('root_path: ' + self.root_path)
        print ('source_files: ')
        pp.pprint(self.source_files)
        print ('header_files:')
        pp.pprint(self.header_files)
        print ('source_groups:')
        pp.pprint(self.source_groups)
        print ('include_paths:')
        pp.pprint(self.include_paths)
        print ('preprocessor_defines:')
        pp.pprint(self.preprocessor_defines)
        print ("")


class SampleSourceFragment(SourceFragment):

    """Sample-specific fragment data"""

    def __init__(self, context, name, fragment_root_path):
        SourceFragment.__init__(self, context, name, fragment_root_path)

        self.setup_code = []


class TestSourceFragment(SourceFragment):

    """Test-specific fragment data"""

    def __init__(self, context, name, fragment_root_path):
        SourceFragment.__init__(self, context, name, fragment_root_path)

        self.test_data_files = [] # The data files that will be copied to the project directory

    def glob_test_data(self, destination, pattern, fail_if_none_found=True):
        files = glob.glob(pattern)
        if fail_if_none_found and len(files) == 0:
            raise Exception('No source files found for ' + pattern)
        self.add_test_data(destination, files)
        return files

    def add_test_data(self, destination, files):
        if not isinstance(files, list):
            files = [files]

        if destination is None:
            destination = ''

        self.test_data_files.append({
            'destination': destination,
            'files': files
        })


class SourceModule:

    def __init__(self, name):

        self.name = name
        self.project_name = name # By default the project name is the same
        self.project_directory = None
        self.fragments = []
        self.output_directory = None
        self.generator_data = {} # Data specific to the build generator for the toolset
        self.precompiled_header_use_map = {} # Mapping of source file to the precompiled header to use for the file
        self.precompiled_header_create_map = {} # Mapping of .cpp to .h which generates precompiled headers
        self.global_preprocessor_defines = {}
        self.global_preprocessor_defines['common'] = {}
        self.global_preprocessor_defines['debug'] = {}
        self.global_preprocessor_defines['release'] = {}

    def add_fragment(self, fragment):
        self.fragments.append(fragment)

    def add_global_preprocessor_definition(self, key, value=None, debug=True, release=True):
        """Appends a preprocessor definition to the module that will be applied to all fragments."""
        if value is None:
            value = 1

        if debug and release:
            self.global_preprocessor_defines['common'][key] = value
        elif debug:
            self.global_preprocessor_defines['debug'][key] = value
        elif release:
            self.global_preprocessor_defines['release'][key] = value

    def get_fragments(self):
        return self.fragments[:]

    def get_preprocessor_definitions(self):
        """Retrieves the combined set of defines for all fragments and the module."""
        result = self.global_preprocessor_defines.copy()
        for fragment in self.fragments:
            fragment_defines = fragment.get_preprocessor_definitions()

            for config in ['common', 'debug', 'release']:
                for key, value in fragment_defines[config].iteritems():
                    if not value is None:
                        result[config][key] = value
        return result

    def get_header_search_paths(self, public_only=False):
        paths = []
        for fragment in self.fragments:
            fragment_paths = fragment.get_header_search_paths(public_only=public_only)
            for path in fragment_paths:
                if path not in paths:
                    paths.append(path)
        return paths

    def get_dependency_search_paths(self):
        paths = []
        for fragment in self.fragments:
            for path in fragment.dependency_search_paths:
                if path not in paths:
                    paths.append(path)
        return paths

    def get_link_libraries(self):
        libs = []
        for fragment in self.fragments:
            for lib in fragment.link_libraries:
                if lib not in libs:
                    libs.append(lib)
        return libs

    def get_symbol_exports(self):
        exports = []
        for fragment in self.fragments:
            for symbol in fragment.symbol_exports:
                exports.append(symbol)
        return exports

    def get_include_relative_path(self, path):
        prefix = os.path.join(self.name, 'include')
        prefix = prefix.replace('\\', '/')
        path = path.replace('\\', '/')
        arr = path.split('/')
        arr.reverse()

        result = arr[0]
        arr = arr[1:]

        for part in arr:
            if part == 'include':
                break
            result = os.path.join(part, result)

        result = result.replace('\\', '/')
        return result


    def get_precompiled_header_use_map(self):
        return self.precompiled_header_use_map.copy()

    def get_precompiled_header_for_file(self, path):
        if path in self.precompiled_header_use_map:
            return self.precompiled_header_use_map[path]
        return None

    def get_precompiled_header_create_map(self):
        return self.precompiled_header_create_map.copy()

    def create_precompiled_header_mappings(self):

        self.precompiled_header_use_map = {}
        self.precompiled_header_create_map = {}

        for fragment in self.fragments:
            path_cpp = fragment.get_precompiled_header_cpp()
            path_h = fragment.get_precompiled_header_h()

            if path_h:
                path_h = path_h.replace('\\', '/')

            if path_cpp:
                path_cpp = path_cpp.replace('\\', '/')
                if path_cpp not in self.precompiled_header_create_map:
                    self.precompiled_header_create_map[path_cpp] = path_h

            if path_h:
                source_files = fragment.get_source_files()
                for path in source_files:
                    # Don't mark the pch itself as
                    if path not in self.precompiled_header_create_map:
                        self.precompiled_header_use_map[path] = path_h


    def debug_print(self):
        pp = pprint.PrettyPrinter(indent=4)

        print ('Module: ' + self.name)
        print ('output_directory: ' + self.output_directory)
        print ('Fragments: ')
        for fragment in self.fragments:
            print ('\t' + fragment.name)
        print ('generator_data:')
        pp.pprint(self.generator_data)
        print ("")


class BaseOutputObject:

    """
    The base class for objects that know how to provide information about how to generate an output object,
    like a library or executable.
    """

    def __init__(self, options, has_internal_access=False):

        self.options = options
        self.modules = []
        self.exports_file = None
        self.has_internal_access = has_internal_access # Whether or not the object has access to private sdk internal includes, etc

    def add_module(self, module):
        self.modules.append(module)

    def clear_modules(self):
        self.modules = []

    def get_module(self, name):
        for mod in self.modules:
            if mod.name == name:
                return mod
        return None

    def get_modules(self):
        return self.modules[:]


    def get_header_search_paths(self, public_only=False):
        result = []
        for mod in self.modules:
            paths = mod.get_header_search_paths(public_only)
            for path in paths:
                if path not in result:
                    result.append(path)
        return result


    def get_dependency_search_paths(self):
        result = []
        for mod in self.modules:
            paths = mod.get_dependency_search_paths()
            for path in paths:
                if path not in result:
                    result.append(path)
        return result


    def get_symbol_exports(self):
        result = []
        for mod in self.modules:
            symbols = mod.get_symbol_exports()
            for symbol in symbols:
                if symbol not in result:
                    result.append(symbol)
        return result


    def get_link_libraries(self):
        result = []
        for mod in self.modules:
            libs = mod.get_link_libraries()
            for lib in libs:
                if lib not in result:
                    result.append(lib)
        return result


    def generate_exports_file(self, output_path_prefix):

        """Loads all the files that contain the names of symbols to export and generates a single file containing all exports."""

        symbols = self.get_symbol_exports()

        # No symbols to export so no file written
        if len(symbols) == 0:
            return None

        # Write the symbols file
        output_path = self.options.platform_settings.write_symbol_exports_file(output_path_prefix, symbols)

        self.exports_file = output_path

        return output_path


    def load_fragment_file(self, python_module):

        """Derived classes should implement this"""

        pass
