#!/usr/bin/python
#
# This script uses Doxygen to generate documentation of the SDK.
#
# To build C++ documentation for a release
#   build_documentation.py --type=public --output-dir=j:/doxygen/cpp/public
#
# To build C++ documentation for internal use which includes all source files and internal headers
#   build_documentation.py --type=internal --output-dir=j:/doxygen/cpp/internal
#
# To build Java documentation for a release
#   build_documentation.py --type=public --language=java --output-dir=j:/doxygen/java/public
#

# Import standard modules
import sys
import os
import argparse
import shutil
import subprocess
import time
import imp
import platform

# Capture paths
doxygen_dir = os.path.realpath(os.path.dirname(__file__))
repo_root = os.path.realpath(os.path.join(doxygen_dir, '..', '..'))


class DocumentationGenerationParameters:

    def __init__(self):
        self.doc_type = None # public or internal
        self.doxygen_dir = None # The explicit directory to find doxygen
        self.doxygen_output_dir = None
        self.modules = []
        self.language = 'c++' # c++, java, python
        self.source_paths = []
        self.repo_dirs = [] # The repository root directories


def find_executable(program):

    def is_exe(fpath):
        return (not fpath is None) and os.path.isfile(fpath) and os.access(fpath, os.X_OK)

    fpath, fname = os.path.split(program)

    if fpath:
        if is_exe(program):
            return program
    else:
        for path in os.environ["PATH"].split(os.pathsep):
            path = path.strip('"')
            exe_file = os.path.join(path, program)
            if is_exe(exe_file):
                return exe_file

    return None


def find_doxygen(config):

    result = None

    # First check for an explicit path to doxygen
    if config.doxygen_path:
        if platform.system() == 'Windows':
            result = os.path.join(config.doxygen_path, 'doxygen.exe')
        else:
            result = os.path.join(config.doxygen_path, 'doxygen')

        result = find_executable(result)
        if result:
            return result
        else:
            raise Exception('Explicit doxygen path seems invalid: ' + config.doxygen_path)

    # Gather a bunch of reasonable places for doxygen to be found
    guesses = []

    if platform.system() == 'Windows':
        guesses = guesses + [
            'doxygen.exe', # System path
            'C:/Program Files/doxygen/bin/doxygen.exe',
            'C:/Program Files (x86)/doxygen/bin/doxygen.exe'
        ]

    else:
        guesses = guesses + [
            'doxygen' # System path
        ]

    # Check all the guesses
    for path in guesses:
        path = find_executable(path)
        if path:
            return path

    raise Exception('Could not find doxygen')


def prepare_doxygen_configuration(config):

    result = None

    # Load the common configuration
    with open(os.path.join(doxygen_dir, 'cpp.doxyfile'), 'r') as file:
        result = file.read()

    # Setup type-specific settings
    if config.doc_type == 'internal':
        result = result + 'PROJECT_NAME = "Twitch SDK (Internal)"\n'

    elif config.doc_type == 'public':
        result = result + 'PROJECT_NAME = "Twitch SDK"\n'

        # Excluse "internal" directories
        result = result + 'EXCLUDE_PATTERNS = */internal/*\n'

    else:
        raise Exception('Unhandled doc type: ' + config.doc_type)

    # Common settings
    result = result + 'OUTPUT_DIRECTORY = ' + config.doxygen_output_dir + '\n'

    # Source directories to scan
    result = result + 'INPUT = \\ \n'
    for path in config.source_paths:
        result = result + '  ' + path + ' \\ \n'

    # Setup type-specific settings
    if config.doc_type == 'internal':
        result = result + 'PROJECT_NAME = "Twitch SDK (Internal)"\n'

    elif config.doc_type == 'public':
        result = result + 'PROJECT_NAME = "Twitch SDK"\n'

        # Excluse "internal" directories
        result = result + 'EXCLUDE_PATTERNS = */internal/*\n'

    else:
        raise Exception('Unhandled doc type: ' + config.doc_type)

    # File extensions
    if config.language == 'c++':
        result = result + 'FILE_PATTERNS = *.c, *.cpp *.cxx *.cpp *.c++ *.h *.hh *.hxx *.hpp *.h++ *.mm *.m\n'
    elif config.language == 'java':
        result = result + 'FILE_PATTERNS = *.java\n'
    elif config.language == 'python':
        result = result + 'FILE_PATTERNS = *.py\n'
    else:
        raise Exception('Unhandled language: ' + config.language)

    result = result + '\n'

    return result


def generate_documentation(config):

    print('Generating documentation...')

    # Find doxygen
    doxygen_path = find_doxygen(config)

    # Generate the doxygen config
    doxygen_config = prepare_doxygen_configuration(config)

    print(doxygen_config)

    # Run doxygen
    shell_args = [doxygen_path, '-']
    proc = subprocess.Popen(shell_args, stdin=subprocess.PIPE)
    proc.communicate(input=doxygen_config)

    if proc.returncode != 0:
        raise Exception('Doxygen failed with error code: ' + str(proc.returncode))

    print('Done generating documentation.')


def find_cpp_source_roots(config):

    for repo_dir in config.repo_dirs:
        modules_dir = os.path.join(repo_dir, 'modules')

        # Find modules
        for module in os.listdir(modules_dir):
            module_root = os.path.join(modules_dir, module)
            if os.path.isdir(module_root):

                # Find fragments in the module
                for fragment in os.listdir(module_root):
                    fragment_root = os.path.join(module_root, fragment)
                    if os.path.isdir(fragment_root):
                        source_path = os.path.join(fragment_root, 'include') #.replace('\\', '/')
                        if os.path.isdir(source_path):
                            config.source_paths.append(source_path)


def find_java_source_roots(config):

    for repo_dir in config.repo_dirs:
        modules_dir = os.path.join(repo_dir, 'modules')
        for module in os.listdir(modules_dir):
            java_fragment_root = os.path.join(modules_dir, module, module + '_java')
            if os.path.isdir(java_fragment_root):
                bindings_root = os.path.join(java_fragment_root, 'bindings')
                if os.path.isdir(bindings_root):
                    config.source_paths.append(bindings_root)


if __name__ == "__main__":

    parser = argparse.ArgumentParser(description='Documentation generation.')

    parser.add_argument(
        '--type',
        required=True,
        metavar='<type>',
        help='Specifies whether or not to generate public or internal documentation.  Options are "public", "internal"'
    )

    parser.add_argument(
        '--output-dir',
        required=True,
        metavar='<output-dir>',
        help='The output directory for generated documentation.'
    )

    parser.add_argument(
        '--language',
        required=False,
        metavar='<language>',
        help='Generate documetation for the given language bindings.'
    )

    parser.add_argument(
        '--doxygen-path',
        required=False,
        metavar='<doxygen-path>',
        help='The directory containing the doxygen executable.'
    )

    args = parser.parse_args(None)

    config = DocumentationGenerationParameters()

    config.doxygen_path = args.doxygen_path
    config.doxygen_output_dir = args.output_dir

    if args.language:
        config.language = args.language

    config.doc_type = args.type
    if config.doc_type not in [ 'public', 'internal' ]: raise Exception('Unknown type: ' + config.doc_type)

    # Use this repository for now
    config.repo_dirs.append(repo_root)

    if config.language == 'c++':
        find_cpp_source_roots(config)
    elif config.language == 'java':
        find_java_source_roots(config)

    # Generate the documentation
    if not os.path.exists(config.doxygen_output_dir):
        os.makedirs(config.doxygen_output_dir)

    generate_documentation(config)
