#!/usr/bin/python
#
# The script used to generate samples for the Twitch SDK.
#
# Requirements
# ==================
# * Python 2.7
# * gitpython

# Import standard modules
import sys
import os
import argparse
import imp # Used to import the module and fragment files dynamically
import shutil
import re
import glob

import build_types
import build_tools

# Cache the root directory of the build script
build_dir = os.path.realpath(os.path.dirname(__file__))

# Update the path so the local scripts can be found
sys.path.append( os.path.join(build_dir, 'generators') )
sys.path.append( os.path.join(build_dir, 'platforms') )


class SampleOptions(build_types.BaseOptions):

    """Generation options specific to the generation of samples."""

    def __init__(self):

        build_types.BaseOptions.__init__(self)

        self.sample = None
        self.twitchsdk_dir = None

    def print_options(self):

        build_types.BaseOptions.print_options(self)

        print("Sample:                " + self.sample)
        print("Twitch SDK Path:       " + self.twitchsdk_dir)


class SampleOutputObject(build_types.BaseOutputObject):

    """Knows how to generate the parts needed to create the sample project."""

    def __init__(self, options):

        build_types.BaseOutputObject.__init__(self, options)


    def load_fragment_file(self, python_module):

        # Try and load the sample fragment
        sample_fragment = None
        if hasattr(python_module, 'load_sample_fragment'):
            sample_fragment = python_module.load_sample_fragment(self.options.sample, self.options)
        else:
            sample_fragment = build_types.SampleSourceFragment('placeholder', 'placeholder', os.path.dirname(os.path.realpath(__file__)))

        # Capture the public properties of the sdk fragment
        sdk_fragment = build_tools.load_twitchsdk_fragment(python_module, build_types.Primitives(), self.options)

        # Merge in the public include paths
        sample_fragment.add_header_search_paths( sdk_fragment.get_header_search_paths(public_only=True) , is_public=True)

        return sample_fragment


    def generate_glue_fragment(self, options):

        """Generates the main fragment for the sample project."""

        fragment = build_types.SourceFragment('sample', options.sample, options.output_dir)

        # Add platform-specific preprocessor definitions
        options.platform_settings.add_platform_preprocessor_definitions(options, fragment)

        # Add main source files
        sample_root = os.path.realpath( os.path.join(options.main_root_path, 'modules', 'core', 'samples', options.sample) )

        files = fragment.glob_source_files( os.path.join(sample_root, 'source', '*.cpp') )
        fragment.add_source_group("Source Files/samples/" + options.sample, files)

        files = fragment.glob_header_files( os.path.join(sample_root, 'include', '*.h') )
        fragment.add_source_group("Header Files/samples/" + options.sample, files)

        # Header search paths
        fragment.add_header_search_paths( options.twitchsdk_dir )
        fragment.add_header_search_paths( os.path.join(sample_root, 'include') )

        return fragment


def generate_sample_project(options, sample_fragments, output_object):

    """Generates the project file for sample project, assuming that all the module source fragments have been added."""

    print('Generating sample executable project')

    # Create the main module
    mod = build_types.SourceModule('__main__')

    # Add the sample fragments
    for fragment in sample_fragments:
        mod.add_fragment(fragment)

    # Create the main fragment
    glue_fragment = output_object.generate_glue_fragment(options)
    mod.add_fragment(glue_fragment)

    output_object.add_module(mod)

    build_tools.add_global_preprocessor_definitions(mod, output_object)

    # Add twitchsdk as an external project
    options.add_external_project(options.twitchsdk_dir)

    # Update precompiled header mappings
    mod.create_precompiled_header_mappings()

    # Generate the project
    generator = options.platform_settings.create_build_generator(options)
    generator.produce_main_generator_files(options.output_dir, output_object)
    generator.run_generator(options.output_dir, options)


def parse_command_line(explicit_arguments, options):

    """Parses the command line for configuring the generation of a Twitch SDK sample."""

    parser = argparse.ArgumentParser(description='Generate project files for a Twitch SDK sample.')

    build_tools.add_common_argparse_parameters(parser)

    parser.add_argument(
        '--sample',
        required=True,
        metavar='<sample>',
        help='Specifies the type of sample to generate.'
    )

    parser.add_argument(
        '--twitchsdk-dir',
        required=True,
        metavar='<output_dir>',
        help='Specifies the directory in which to find the Twitch SDK project.'
    )

    args = parser.parse_args(explicit_arguments)

    build_tools.parse_common_argparse_parameters(args, options)

    options.sample = args.sample
    options.twitchsdk_dir = build_tools.resolve_path( args.twitchsdk_dir )


def generate(options):

    # Standard options setup
    build_tools.setup_options(options, __file__, None)

    # Always build an executable
    options.output_object_type = 'executable'

    # Print the options
    options.print_options()

    # Create the executable container
    executable = SampleOutputObject(options)

    # Load all sample modules and combine all the fragments into one project
    sample_fragments = []
    for module_name in options.modules:
        module = build_tools.load_module(module_name, module_name, executable)

        executable.add_module(module)

        sample_fragments = sample_fragments + module.fragments

    # Generate the executable project which includes all the module projects
    generate_sample_project(options, sample_fragments, executable)

    # Notify of completion
    print('')
    print('Generated build files have been written to ' + options.output_dir)


# Process the command line arguments if run as the primary script
if __name__ == "__main__":

    options = SampleOptions()

    # Determine what we need to do
    parse_command_line(None, options)

    generate(options)
