#!/usr/bin/python
#
# The script used to generate the test code for the Twitch SDK.
#
# Requirements
# ==================
# * Python 2.7
# * gitpython
#


# Import standard modules
import sys
import os
import argparse
import imp # Used to import the module and fragment files dynamically
import shutil
import re
import glob

import build_types
import build_tools

# Cache the root directory of the build script
build_dir = os.path.realpath(os.path.dirname(__file__))

# Update the path so the local scripts can be found
sys.path.append( os.path.join(build_dir, 'generators') )
sys.path.append( os.path.join(build_dir, 'platforms') )


class TestOptions(build_types.BaseOptions):

    """Generation options specific to the generation of the unit test projects."""

    def __init__(self):

        build_types.BaseOptions.__init__(self)

        self.test_type = None
        self.twitchsdk_dir = None


    def print_options(self):

        build_types.BaseOptions.print_options(self)

        print("Test Type:             " + self.test_type)
        print("Twitch SDK Path:       " + self.twitchsdk_dir)


class UnitTestOutputObject(build_types.BaseOutputObject):

    """Knows how to generate the parts needed to create the unit test project."""

    def __init__(self, options):

        build_types.BaseOutputObject.__init__(self, options)


    def load_fragment_file(self, python_module):

        # Try and load the test fragment
        test_fragment = None
        if hasattr(python_module, 'load_unittest_fragment'):
            test_fragment = python_module.load_unittest_fragment(self.options)
        else:
            test_fragment = build_types.TestSourceFragment('placeholder', 'placeholder', os.path.dirname(os.path.realpath(__file__)))

        # Capture the public properties of the sdk fragment
        sdk_fragment = build_tools.load_twitchsdk_fragment(python_module, build_types.Primitives(), self.options)

        # Merge in the public include paths
        test_fragment.add_header_search_paths( sdk_fragment.get_header_search_paths(public_only=True) , is_public=True)

        return test_fragment


    def generate_glue_fragment(self, options):

        """Generates the main fragment for the unit test project."""

        fragment = build_types.TestSourceFragment('unittest', 'unittest', options.output_dir)

        # Add platform-specific preprocessor definitions
        options.platform_settings.add_platform_preprocessor_definitions(options, fragment)

        # Add main source files
        test_root = os.path.realpath( os.path.join(options.main_root_path, 'modules', 'core', 'tests', 'unit') )

        files = fragment.glob_source_files( os.path.join(test_root, 'source', '*.cpp') )
        fragment.add_source_group("Source Files/unittest", files)

        files = fragment.glob_source_files( os.path.join(test_root, 'source', 'fixtures', '*.cpp') )
        fragment.add_source_group("Source Files/unittest/fixtures", files)

        files = fragment.glob_header_files( os.path.join(test_root, 'include', '*.h') )
        fragment.add_source_group("Header Files/unittest", files)

        files = fragment.glob_header_files( os.path.join(test_root, 'include', 'fixtures', '*.h') )
        fragment.add_source_group("Header Files/unittest/fixtures", files)

        # Add Google Test framework source
        files = fragment.glob_source_files( os.path.join(options.gtest_path, 'src', 'gtest-all.cc') )
        fragment.add_source_group("Source Files/gtest", files)

        files = fragment.glob_header_files( os.path.join(options.gtest_path, 'include', 'gtest', '*.h') )
        fragment.add_source_group("Header Files/gtest", files)

        files = fragment.glob_header_files( os.path.join(options.gtest_path, 'include', 'gtest', 'internal', '*.h') )
        fragment.add_source_group("Header Files/gtest/internal", files)

        fragment.add_dependency_search_paths( os.path.join(options.gtest_path, 'include') )
        fragment.add_dependency_search_paths( os.path.join(options.gtest_path) ) # Since gtest-all.cc includes other source files
        fragment.add_header_search_paths( options.twitchsdk_dir )
        fragment.add_header_search_paths( os.path.join(test_root, 'include') )

        return fragment


def generate_unittest_project(options, test_fragments, output_object):

    """Generates the project file for the unit test project, assuming that all the module source fragments have been added."""

    print('Generating unittest executable project')

    # Create the main module
    mod = build_types.SourceModule('__main__')

    # We add all source from all fragments into the same project for since it allows gtest
    # to dynamically find all test cases.  Google Test does not work well with static libraries.
    for fragment in test_fragments:
        mod.add_fragment(fragment)

    # Create the main fragment
    glue_fragment = output_object.generate_glue_fragment(options)
    mod.add_fragment(glue_fragment)

    output_object.add_module(mod)

    build_tools.add_global_preprocessor_definitions(mod, output_object)

    # Add twitchsdk as an external project
    options.add_external_project(options.twitchsdk_dir)

    # Update precompiled header mappings
    mod.create_precompiled_header_mappings()

    # Generate the project
    generator = options.platform_settings.create_build_generator(options)
    generator.produce_main_generator_files(options.output_dir, output_object)
    generator.run_generator(options.output_dir, options)


def copy_test_data(fragments, options):

    """Copies unit test data to the output project directory."""

    print('Copying unit tests data...')

    # Create the test data directory
    test_data_oath = os.path.join(options.output_dir, 'data')
    if os.path.exists(test_data_oath):
        shutil.rmtree(test_data_oath, ignore_errors=True)
    if not os.path.exists(test_data_oath):
        os.makedirs(test_data_oath)

    # Copy test data
    for fragment in fragments:
        for entry in fragment.test_data_files:

            # Create the destination directory
            dest = test_data_oath
            if entry['destination'] != '':
                dest = os.path.join(test_data_oath, entry['destination'])
            if not os.path.exists(dest):
                os.makedirs(dest)

            # Copy files
            for file in entry['files']:
                filename = os.path.basename(file)
                dest_path = os.path.join(dest, filename)
                shutil.copyfile(file, dest_path)
                print('  ' + file + ' --> ' + dest_path)

    print('Done copying unit tests data')


# The possible supported test types
POSSIBLE_TEST_CHOICES = [
    'unit'
]


def parse_command_line(explicit_arguments, options):

    """Parses the command line for configuring the generation of the Twitch SDK tests."""

    global POSSIBLE_TEST_CHOICES

    parser = argparse.ArgumentParser(description='Generate project files for the Twitch SDK tests.')

    build_tools.add_common_argparse_parameters(parser)

    parser.add_argument(
        '--test',
        required=True,
        metavar='<test>',
        choices=POSSIBLE_TEST_CHOICES,
        help='Specifies the type of tests to generate.'
    )

    parser.add_argument(
        '--twitchsdk-dir',
        required=True,
        metavar='<output_dir>',
        help='Specifies the directory in which to find the Twitch SDK project.'
    )

    args = parser.parse_args(explicit_arguments)

    build_tools.parse_common_argparse_parameters(args, options)

    options.test_type = args.test
    options.twitchsdk_dir = build_tools.resolve_path( args.twitchsdk_dir )


def generate(options):

    # Standard options setup
    build_tools.setup_options(options, __file__, None)

    # Always build an executable
    options.output_object_type = 'executable'

    # Print the options
    options.print_options()

    # Create the executable container
    executable = UnitTestOutputObject(options)

    # Load all unit test modules and combine all the fragments into one project
    test_fragments = []
    for module_name in options.modules:
        module = build_tools.load_module(module_name, module_name, executable)

        test_fragments = test_fragments + module.fragments

    # Generate the executable project which includes all the module projects
    generate_unittest_project(options, test_fragments, executable)

    # Copy test data
    copy_test_data(test_fragments, options)

    # Notify of completion
    print('')
    print('Generated build files have been written to ' + options.output_dir)


# Process the command line arguments if run as the primary script
if __name__ == "__main__":

    options = TestOptions()

    # Determine what we need to do
    parse_command_line(None, options)

    generate(options)
