#!/usr/bin/python
#
# The script used to generate the Twitch SDK library projects.
#
# Requirements
# ==================
# * Python 2.7
# * gitpython
#
# The high-level function of these build scripts:
#
# Generate one project per module:
#   For each module:
#       Create directory ${output_dir}/${module_name}
#       Process all the fragments in the module
#       Runs the build generator on each module
# Generate the twitchsdk library project:
#   Generate twitchsdk.h and twitchsdk.cpp in ${output_dir}
#   Runs the build generator on the library
#
# TODO
# ============
# * Enable precompiled headers
# * Get dynamic libraries working in Xcode, https://cmake.org/gitweb?p=cmake.git;a=blob;f=Tests/iOSNavApp/CMakeLists.txt


# Import standard modules
import sys
import os
import argparse
import imp # Used to import the module and fragment files dynamically
import shutil
import re

import build_types
import build_tools

# Cache the root directory of the build script
build_dir = os.path.realpath(os.path.dirname(__file__))

# Update the path so the local scripts can be found
sys.path.append( os.path.join(build_dir, 'generators') )
sys.path.append( os.path.join(build_dir, 'platforms') )


class TwitchSdkOptions(build_types.BaseOptions):

    """Generation options specific to the generation of the Twitch SDK library projects."""

    def __init__(self):

        build_types.BaseOptions.__init__(self)


    def print_options(self):

        build_types.BaseOptions.print_options(self)


class TwitchSdkLibraryOutputObject(build_types.BaseOutputObject):

    """Knows how to generate the parts needed by the Twitch SDK library."""

    def __init__(self, primitives, options):

        build_types.BaseOutputObject.__init__(self, options, has_internal_access=True)

        self.primitives = primitives


    def load_fragment_file(self, python_module):

        return build_tools.load_twitchsdk_fragment(python_module, self.primitives, self.options)


    def generate_glue_fragment(self, options):

        """
        Generates the code and settings required for the project that will link
        all the modules together and provide the initialization and shutdown function.
        """

        fragment = build_types.SourceFragment('twitchsdk', 'twitchsdk', options.output_dir)

        header_file_path = os.path.join(options.output_dir, 'twitchsdk.h')
        source_file_path = os.path.join(options.output_dir, 'twitchsdk.cpp')

        self.generate_glue_code(source_file_path, header_file_path)

        fragment.add_source_files(source_file_path)
        fragment.add_header_files(header_file_path)

        fragment.add_source_group("Source Files/twitchsdk", source_file_path)
        fragment.add_source_group("Header Files/twitchsdk", header_file_path)

        return fragment


    def generate_glue_code(self, source_file_path, header_file_path):

        """
        Generates twitchsdk.cpp and twitchsdk.h.
        """

        print("Generating glue code...")

        # Generate the header file
        file = open(header_file_path, 'w')

        # Write the commons header stuff
        build_tools.write_header_file_common_preamble(file, self.modules, self.options)

        if not self.options.no_primitive_registration:
            file.write(" * Primitives:\n")
            for primitive in self.primitives.table:
                if primitive['implemented']:
                    if primitive['class']:
                        file.write(" *   " + primitive['name'] + ": " + primitive['class'] + " => " + primitive['header'] + "\n")
                else:
                    file.write(" *   " + primitive['name'] + ": MISSING\n")
        else:
            print("Primitives not registered in twitchsdk.cpp/h")


        build_tools.write_header_file_preamble_done(file)

        file.write("""
#include "twitchsdk/core/types/coretypes.h"
#include "twitchsdk/core/types/errortypes.h"

/**
 * Initializes the Twitch SDK library for use by apps.  The application should use the generated library as follows:
 *
 *  - Add the Twitch SDK library project to you header search paths.
 *  - #include "library.h" from your project source.
 *  - Call TTV_InitializeLibrary() which should return TTV_EC_SUCCESS.
 *  - Register any custom primitive components that the application wants to provide, such as sockets and HTTP services.
 *  - Create and initialize CoreAPI.
 *  - Create and initialize feature modules such as ChatAPI or SocialAPI which rely on CoreAPI.
 *  - Run your app.
 *  - Shutdown all feature modules.
 *  - Shutdown CoreAPI.
 *  - Unregister custom application provided primitives.
 *  - Call TTV_LibraryShutdown().
 *
 * If successful this function will return TTV_EC_SUCCESS.  If the library is already initialized it will return TTV_EC_ALREADY_INITIALIZED.
 * Other specific errors may be returned if platform primitives or services fail to initialize.
 */
TTV_ErrorCode TTV_InitializeLibrary();

/**
 * Shuts down the Twitch SDK library.  This should return TTV_EC_SUCCESS.  Before calling this be sure to
 *
 *  - Shut down all feature modules and CoreAPI.
 *  - Unregister custom pritive implementations and factories.
 *
 * If successful this function will return TTV_EC_SUCCESS.  If the library is not initialized it will return TTV_EC_NOT_INITIALIZED.
 * Other specific errors may be returned if platform primitives or services fail to shutdown.
 */
TTV_ErrorCode TTV_ShutdownLibrary();

/**
 * Returns a unique string which identifies this build of the Twitch SDK library.  It is generated using the commit hashes
 * of the various repositories that were used to build the library.
 */
 const char* TTV_GetVersionString();
""");
        file.close()


        # Generate the source file
        file = open(source_file_path, 'w')

        file.write("/********************************************************************************\n")
        file.write(" * \n")
        file.write(" * This file was auto-generated using Twitch build tools - Do not modify by hand\n")
        file.write(" * \n")
        file.write(" ********************************************************************************/\n")
        file.write("#include \"twitchsdk.h\"\n")

        if not self.options.no_primitive_registration:
            file.write("#include <memory>\n")
            file.write("#include \"twitchsdk/core/thread.h\"\n")
            file.write("#include \"twitchsdk/core/threadsync.h\"\n")
            file.write("#include \"twitchsdk/core/systemclock.h\"\n")
            file.write("#include \"twitchsdk/core/tracer.h\"\n")
            file.write("#include \"twitchsdk/core/mutex.h\"\n")
            file.write("#include \"twitchsdk/core/socket.h\"\n")
            file.write("#include \"twitchsdk/core/httprequest.h\"\n")
            file.write("#include \"twitchsdk/core/eventscheduler.h\"\n")
        file.write("#include \"twitchsdk/core/version.h\"\n")
        file.write("\n")


        if not self.options.no_primitive_registration:

            # Find the unique includes
            includes = []

            for primitive in self.primitives.table:
                include = primitive['header']
                if include and not include in includes:
                    includes.append(include)

            for include in includes:
                file.write("#include \"" + include + "\"\n")

            file.write("\n")

        file.write("namespace\n")
        file.write("{\n")
        file.write("    using namespace ttv;\n")
        file.write("\n")

        if not self.options.no_primitive_registration:
            for primitive in self.primitives.table:
                if primitive['implemented']:
                    if primitive['class']:
                        file.write("    std::shared_ptr<" + primitive['interface'] + "> " + primitive['variable_name'] + ";\n")
            file.write("\n")

            for primitive in self.primitives.table:
                if primitive['flag_code']:
                    for line in primitive['flag_code']:
                        file.write("    " + line + "\n")

        file.write("    bool gInitialized = false;\n")
        file.write("}\n")
        file.write("\n")
        file.write("TTV_ErrorCode TTV_InitializeLibrary()\n")
        file.write("{\n")
        file.write("    if (gInitialized)\n")
        file.write("    {\n")
        file.write("        return TTV_EC_ALREADY_INITIALIZED;\n")
        file.write("    }\n")
        file.write("\n")

        if not self.options.no_primitive_registration:
            file.write("    ttv::InitializeSocketLibrary();\n")
            file.write("\n")

        file.write("    ttv::SetVersionString(TTV_GetVersionString());\n")
        file.write("\n")

        if not self.options.no_primitive_registration:

            # Special initialization code
            for primitive in self.primitives.table:
                if primitive['init_code']:
                    file.write("    {\n")
                    for line in primitive['init_code']:
                        file.write("        " + line + "\n")
                    file.write("    }\n")
                    file.write("\n")

            # Registration code
            for primitive in self.primitives.table:
                if primitive['implemented']:
                    if primitive['class']:
                        file.write("    " + primitive['variable_name'] + " = " + "std::make_shared<" + primitive['class'] + ">();\n")
                        file.write("    " + primitive['registration_function'] + "(" + primitive['variable_name'] + ");\n")
                        file.write("\n")
                else:
                    file.write("    // NOTE: " + primitive['missing_message'] + "\n")
                    file.write("\n")

        file.write("    gInitialized = true;\n")
        file.write("\n")
        file.write("    return TTV_EC_SUCCESS;\n")
        file.write("}\n")
        file.write("\n")
        file.write("TTV_ErrorCode TTV_ShutdownLibrary()\n")
        file.write("{\n")
        file.write("    if (!gInitialized)\n")
        file.write("    {\n")
        file.write("        return TTV_EC_NOT_INITIALIZED;\n")
        file.write("    }\n")
        file.write("\n")

        if not self.options.no_primitive_registration:

            # Unregistration
            for primitive in self.primitives.table:
                if primitive['implemented']:
                    if primitive['class']:
                        file.write("    if (" + primitive['variable_name'] + " != nullptr)\n")
                        file.write("    {\n")
                        if primitive['unregistration_function']:
                            file.write("        " + primitive['unregistration_function'] + "(" + primitive['variable_name'] + ");\n")
                        else:
                            file.write("        " + primitive['registration_function'] + "(nullptr);\n")

                        file.write("        " + primitive['variable_name'] + ".reset();\n")
                        file.write("    }\n")
                        file.write("\n")
                else:
                    file.write("    // NOTE: " + primitive['missing_message'] + "\n")
                    file.write("\n")

            # Special shutdown code
            reverse_list = list(self.primitives.table)
            reverse_list.reverse()
            for primitive in reverse_list:
                if primitive['shutdown_code']:
                    file.write("    {\n")
                    for line in primitive['shutdown_code']:
                        file.write("        " + line + "\n")
                    file.write("    }\n")
                    file.write("\n")

            file.write("    ttv::ShutdownSocketLibrary();\n")
            file.write("\n")

        file.write("    gInitialized = false;\n")
        file.write("\n")
        file.write("    return TTV_EC_SUCCESS;\n")
        file.write("}\n")
        file.write("\n")

        file.write("const char* TTV_GetVersionString()\n")
        file.write("{\n")
        file.write("    return \"" + self.options.version_string + "\";\n")
        file.write("}\n")
        file.write("\n")

        file.close()

        print("Done generating glue code")


def generate_twitchsdk_project(primitives, options, output_object):

    """Generates the project file for the Twitch SDK library, assuming that all the module source fragments have been added."""

    #primitives.debug_print()

    print('Generating twitchsdk library project')

    # Create the main fragment
    fragment = output_object.generate_glue_fragment(options)

    # Symbol exports if dynamic library
    if options.output_object_type == 'dynamic_library':
        fragment.add_symbol_export("TTV_InitializeLibrary")
        fragment.add_symbol_export("TTV_ShutdownLibrary")
        fragment.add_symbol_export("TTV_GetVersionString")

    # Create the main module
    library_module = build_types.SourceModule('__main__')
    library_module.project_name = 'twitchsdk'
    library_module.add_fragment(fragment)

    # Monolithic projects contain all source in the main project
    if options.monolithic_project:
        # Remove the previous modules
        modules = output_object.get_modules()
        output_object.clear_modules()

        # Merge all modules into the main module
        for module in modules:
            for fragment in module.get_fragments():
                library_module.add_fragment(fragment)

    # Append the main module
    output_object.add_module(library_module)

    build_tools.add_global_preprocessor_definitions(library_module, output_object)

    # Create the symbol exports file
    if options.output_object_type == 'dynamic_library':
        exports_path = os.path.join(options.output_dir, 'twitchsdk')
        exports_path = output_object.generate_exports_file(exports_path)

        if not exports_path is None:
            fragment.add_source_files(exports_path)
            fragment.add_source_group("Source Files/twitchsdk", exports_path)

    # Update precompiled header mappings
    library_module.create_precompiled_header_mappings()

    # Generate the project
    generator = options.platform_settings.create_build_generator(options)
    generator.produce_main_generator_files(options.output_dir, output_object)
    generator.run_generator(options.output_dir, options)


def parse_command_line(explicit_arguments, options):

    """Parses the command line for configuring the generation of the Twitch SDK."""

    parser = argparse.ArgumentParser(description='Generate project files for the Twitch SDK.')

    build_tools.add_common_argparse_parameters(parser)

    parser.add_argument(
        '--static',
        required=False,
        action='store_true',
        help='Specifies the final library to be generated should be a static library.'
    )

    parser.add_argument(
        '--dynamic',
        required=False,
        action='store_true',
        help='Specifies the final library to be generated should be a dynamic library.'
    )

    parser.add_argument(
        '--no-primitive-registration',
        required=False,
        action='store_true',
        help='Specifies if primitive components should not be registered.'
    )

    args = parser.parse_args(explicit_arguments)

    build_tools.parse_common_argparse_parameters(args, options)

    if args.static:
        options.output_object_type = 'static_library'

    if args.dynamic:
        options.output_object_type = 'dynamic_library'

    if args.no_primitive_registration:
        options.no_primitive_registration = True;

def generate(options):

    # Standard options setup
    build_tools.setup_options(options, __file__, None)

    # Initialize the primitives table which is used during code generation
    primitives = build_types.Primitives()

    # Determine implicit platforms by checking for settings of the form TTV_USE_<platform>_<feature>
    for key, value in options.settings.iteritems():
        result = re.match(r'^TTV_USE_([^_]+)_.+', key, re.I)
        if result:
            platform = result.group(1).lower()
            if not platform in options.platforms:
                print('Implicit platform specified via switch ' + key + ': ' + platform)
                options.platforms.append(platform)

    # Print the options
    options.print_options()

    # Create the library container
    library = TwitchSdkLibraryOutputObject(primitives, options)

    # Adjust the monolithic flag based on platform needs
    options.monolithic_project = options.monolithic_project or options.platform_settings.requires_monolithic_project(options)

    # Generate the projects for the specified modules
    for module_name in options.modules:
        module = build_tools.load_module(module_name, module_name, library)

        library.add_module(module)

        # Now generate the project file
        if not options.monolithic_project:
            build_tools.generate_module_project(module, library)

    # Generate the library project which includes all the module projects
    generate_twitchsdk_project(primitives, options, library)

    # Notify of completion
    print('')
    print('Generated build files have been written to ' + options.output_dir)


# Process the command line arguments if run as the primary script
if __name__ == "__main__":

    options = TwitchSdkOptions()

    # Determine what we need to do
    parse_command_line(None, options)

    generate(options)
