import abc
import build_generator
import os
import subprocess
import platform


class AndroidNdkBuildGenerator(build_generator.BuildGenerator):

    def produce_module_generator_files(self, source_module, output_directory, output_object):

        """Generates Android.mk for the given module."""

        source_module.output_directory = output_directory
        android_mk_path = os.path.join(output_directory, 'Android.mk')

        source_module.generator_data = {
            'generator': 'android_mk',
            'android_mk_path': android_mk_path
        }

        # TODO: Handle other possible inter-module dependencies if they ever some up
        core_module = output_object.get_module('core')
        if core_module is None:
            raise Exception('Could not find core module')

        print('Generating ' + android_mk_path + '...')

        file = open(android_mk_path, 'w')

        file.write("""#################################################################################
# This file was auto-generated using Twitch build tools - Do not modify by hand
#################################################################################

# Clear unneeded variables with the exception of LOCAL_PATH
include $(CLEAR_VARS)

# Save the previous LOCAL_PATH so we can restore it later
PREVIOUS_LOCAL_PATH := $(LOCAL_PATH)

# Define the location of source code, my-dir is a macro to get the current directory
LOCAL_PATH := $(call my-dir)

# Define the name of the output
""")
        file.write("LOCAL_MODULE := " + source_module.name + "\n")
        file.write("LOCAL_MODULE_FILENAME := lib" + source_module.name + "\n")
        file.write("\n")

        self.write_toolchain(file)

        # Preprocessor definitions
        self.write_preprocessor_definitions(file, source_module)
        file.write("\n")

        # Include paths
        self.write_include_paths(file, [core_module, source_module])
        file.write("\n")

        # Source files
        self.write_source_files(file, source_module)
        file.write("\n")

        # TODO: Generate precompiled headers

        # Build as a static library
        file.write("include $(BUILD_STATIC_LIBRARY)\n")

        file.write("""
# Restore the previous LOCAL_PATH
LOCAL_PATH := $(PREVIOUS_LOCAL_PATH)
""")

        file.close()

        print('Done generating Android.mk file')


    def produce_main_generator_files(self, output_directory, output_object):

        """Generates Android.mk and Application.mk for the library."""

        # Generate Application.mk
        application_mk_path = os.path.join(output_directory, 'Application.mk')

        print('Generating ' + application_mk_path + '...')

        file = open(application_mk_path, 'w')

        file.write("""#################################################################################
# This file was auto-generated using Twitch build tools - Do not modify by hand
#################################################################################

NDK_TOOLCHAIN_VERSION := clang

APP_PLATFORM    := android-9
APP_STL         := c++_static
APP_CPPFLAGS    := -fexceptions -frtti

# Configure debug or release build
ifdef NDK_DEBUG
    $(info Building debug)
    APP_OPTIM   := debug
else
    $(info Building release)
    APP_OPTIM   := release
endif

""")

        # Instruction sets
        file.write("# Instruction sets\n")
        if output_object.options.target is None or output_object.options.target == '':
            file.write("APP_ABI := all\n")

        else:
            file.write("APP_ABI := \n")
            targets = output_object.options.target.split(',')
            for target in targets:
                file.write("APP_ABI += " + target + "\n")
        file.write("\n")

        # Add the module projects
        file.write("# Specify modules to build\n")
        for m in output_object.modules:
            if m.name != 'library':
                file.write("APP_MODULES += " + m.name + "\n")
        file.write("APP_MODULES += twitchlibrary\n")

        file.close()

        # Generate Android.mk
        android_mk_path = os.path.join(output_directory, 'Android.mk')

        print('Generating ' + android_mk_path + '...')

        file = open(android_mk_path, 'w')

        file.write("""#################################################################################
# This file was auto-generated using Twitch build tools - Do not modify by hand
#################################################################################

# Define the location of source code, my-dir is a macro to get the current directory
LOCAL_PATH := $(call my-dir)

# Clear unneeded variables with the exception of LOCAL_PATH
include $(CLEAR_VARS)

""")
        # Add the module projects
        for m in output_object.modules:
            if 'android_mk_path' in m.generator_data:
                module_directory = os.path.dirname( m.generator_data['android_mk_path'] )
                module_directory = module_directory.replace('\\', '/')
                file.write("include " + module_directory + "/Android.mk\n")
                file.write("include $(CLEAR_VARS)\n")
                file.write("\n")

        self.write_toolchain(file)

        file.write("""
# Define the name of the output
""")
        file.write("LOCAL_MODULE := twitchlibrary\n")
        file.write("LOCAL_MODULE_FILENAME := lib" + output_object.options.project_name + "\n")

        file.write("\n")

        file.write("""
# Link against atomic for armeabi and mips only
ifeq ($(TARGET_ARCH_ABI), armeabi)
    LOCAL_LDLIBS += -latomic
endif
ifeq ($(TARGET_ARCH_ABI),mips)
    LOCAL_LDLIBS += -latomic
endif

LOCAL_LDLIBS += -llog

""")

        # Find the twitchsdk library module
        main_module = output_object.get_module('__main__')

        # Preprocessor definitions
        self.write_preprocessor_definitions(file, main_module)
        file.write("\n")

        # Include paths
        self.write_include_paths(file, output_object.modules)
        file.write("\n")

        # Source files
        self.write_source_files(file, main_module)
        file.write("\n")

        # Link against libraries
        self.write_link_libraries(file, output_object)
        file.write("\n")

        # Library type
        self.write_library_type(file, output_object)
        file.write("\n")

        file.close()


    def run_generator(self, directory, options):
        # This is a NOOP since the we directly generate the final make files
        pass


    def write_source_files(self, file, module):
        for fragment in module.fragments:
            for path in fragment.source_files:
                path = path.replace('\\', '/')
                file.write("LOCAL_SRC_FILES += " + path + "\n")

    def write_include_paths(self, file, modules):
        for module in modules:
            paths = module.get_header_search_paths()
            for path in paths:
                file.write("LOCAL_C_INCLUDES += " + path + "\n")

    def write_preprocessor_definitions(self, file, module):
        for fragment in module.fragments:
            for key, value in fragment.preprocessor_defines.iteritems():
                if not value is None:
                    file.write("LOCAL_CPPFLAGS += -D" + key + "=" + str(value) + "\n")
                else:
                    file.write("LOCAL_CPPFLAGS += -D" + key + "=1\n")

    def write_link_libraries(self, file, output_object):
        # Link the library project against the module projects
        for m in output_object.modules:
            if m.name != 'library' and 'android_mk_path' in m.generator_data:
                module_directory = os.path.dirname( m.generator_data['android_mk_path'] )
                module_directory = module_directory.replace('\\', '/')
                file.write("LOCAL_STATIC_LIBRARIES += lib" + m.name + "\n")
        file.write("\n")

        # TODO: Link against external dependencies
        libs = output_object.get_link_libraries()
        for path in libs:
            name = os.path.basename(path)
            path = path.replace('\\', '/')
            file.write("LOCAL_STATIC_LIBRARIES += " + path + "\n")
        file.write("\n")


    def write_library_type(self, file, output_object):
        if output_object.options.output_object_type == 'dynamic_library':
            file.write("include $(BUILD_SHARED_LIBRARY)\n")
        elif output_object.options.output_object_type == 'static_library':
            file.write("include $(BUILD_STATIC_LIBRARY)\n")


    def write_toolchain(self, file):
        file.write("""
NDK_TOOLCHAIN_VERSION := clang
LOCAL_CLANG := true
LOCAL_CFLAGS := -std=c++14

# Enable exceptions
LOCAL_CPP_FEATURES += exceptions

# Configure debug or release build
ifeq ($(APP_OPTIM),debug)
    #LOCAL_CFLAGS += -Werror
    LOCAL_CPPFLAGS += -DDEBUG
else
    LOCAL_CPPFLAGS += -DNDEBUG
endif

""")


# Set up inheritance
build_generator.BuildGenerator.register(AndroidNdkBuildGenerator)
