import abc
import os

class BuildGenerator(object):

    """The interface for generators capable of creating platform and compiler specific project files"""

    __metaclass__ = abc.ABCMeta

    @abc.abstractmethod
    def produce_module_generator_files(self, source_module, output_directory, output_object):
        """Produces the files needed to generate the project for the library which links against module projects."""
        pass

    @abc.abstractmethod
    def produce_main_generator_files(self, output_directory, output_object):
        """Produces the files needed to generate the main project which links against module projects."""
        pass

    @abc.abstractmethod
    def run_generator(self, directory, options):
        """Called after generating the files needed to create the final projects."""
        pass

    def output_hierarchical_structure(self):
        """Whether the generator should output its products with each module in its own
        folder, or whether they should output in a flat structure.
        Subclasses may override this."""
        return True

    def is_exe(self, fpath):
        return (not fpath is None) and os.path.isfile(fpath) and os.access(fpath, os.X_OK)

    def find_executable(self, program):

        fpath, fname = os.path.split(program)

        if fpath:
            if self.is_exe(program):
                return program
        else:
            for path in os.environ["PATH"].split(os.pathsep):
                path = path.strip('"')
                exe_file = os.path.join(path, program)
                if self.is_exe(exe_file):
                    return exe_file

        return None
