import abc
import build_generator
import os
import platform
import re
import subprocess
import sys

build_dir = os.path.realpath(os.path.join(os.path.dirname(__file__), '..'))
sys.path.append(build_dir)

import build_tools


class CMakeBuildGenerator(build_generator.BuildGenerator):

    """
    A generator which writes CMakeLists.txt files and runs CMake in order to generate project files.
    CMake will be searched for as follows:
      * Checks for the environment variables called TTV_CMAKE_PATH which is the bin directory of cmake.exe
      * Checks the system path
      * Checks some hardcoded paths based on the platform
    """

    def __init__(self):
        build_generator.BuildGenerator.__init__(self)
        self.min_cmake_version = '3.8.0'


    def get_source_group_string(self, str):
        return str.replace('/', '\\\\\\\\')


    def produce_module_generator_files(self, source_module, output_directory, output_object):

        """Produces the files required to generate the project for the module"""

        source_module.output_directory = output_directory
        cmake_lists_path = os.path.join(output_directory, 'CMakeLists.txt')

        # Load and cache other previous CMakelists.txt settings
        if os.path.isfile(cmake_lists_path):
            output_object.options.platform_settings.read_cmake_previous_cmakelists_file(cmake_lists_path)

        source_module.generator_data = {
            'generator': 'cmake',
            'cmake_lists_path': cmake_lists_path
        }

        # TODO: Handle other possible inter-module dependencies if they ever some up
        core_module = output_object.get_module('core')
        if core_module is None:
            raise Exception('Could not find core module')

        # Generate the CMakeLists.txt

        print('Generating ' + cmake_lists_path + '...')

        file = open(cmake_lists_path, 'w')

        # Project setup
        if hasattr(output_object.options.platform_settings, 'min_cmake_version'):
            self.min_cmake_version = output_object.options.platform_settings.min_cmake_version

        file.write("cmake_minimum_required (VERSION " + self.min_cmake_version + " FATAL_ERROR)\n")
        file.write("set(CMAKE_SUPPRESS_REGENERATION true)\n")
        file.write("set (CMAKE_CONFIGURATION_TYPES Debug Release)\n")
        file.write("set_property (GLOBAL PROPERTY OS_FOLDERS ON)\n")
        file.write("\n")
        self.write_compiler_flags(file, output_object.options)
        self.write_custom(file, output_object.options, output_object, source_module.project_name);
        file.write("\n")

        # Include paths
        self.write_include_paths(file, [core_module, source_module], public_only=False)
        file.write("\n")

        # Source files
        self.write_source_files(file, source_module)
        file.write("\n")

        # Header files
        self.write_header_files(file, source_module)
        file.write("\n")

        file.write("set (all_files ${source_files} ${header_files})\n")
        file.write("\n")

        # Source groups
        self.write_source_groups(file, source_module)
        file.write("\n")

        # Generate precompiled headers
        self.write_precompiled_header_logic(file, source_module, output_object)

        # Library type
        self.write_output_object_type(file, 'static_library', source_module.project_name)
        file.write("\n")

        # Linker flags
        self.write_linker_flags(file, 'static_library', source_module.project_name, output_object.options)
        file.write("\n")

        # Preprocessor definitions
        self.write_preprocessor_definitions(file, source_module.project_name, source_module)

        file.close()

        print('Done generating CMakeLists.txt file')


    def produce_main_generator_files(self, output_directory, output_object):

        """Produces the generator files for the main project."""

        # Generate the CMakeLists.txt
        cmake_lists_path = os.path.join(output_directory, 'CMakeLists.txt')

        # Load and cache other previous CMakelists.txt settings
        if os.path.isfile(cmake_lists_path):
            output_object.options.platform_settings.read_cmake_previous_cmakelists_file(cmake_lists_path)

        print('Generating ' + cmake_lists_path + '...')

        # Find the main module
        main_module = output_object.get_module('__main__')

        file = open(cmake_lists_path, 'w')

        if hasattr(output_object.options.platform_settings, 'min_cmake_version'):
            self.min_cmake_version = output_object.options.platform_settings.min_cmake_version

        file.write("cmake_minimum_required (VERSION " + self.min_cmake_version + " FATAL_ERROR)\n")
        file.write("set(CMAKE_SUPPRESS_REGENERATION true)\n")
        file.write("project (\"" + output_object.options.project_name + "\")\n")
        file.write("set (CMAKE_CONFIGURATION_TYPES Debug Release)\n")
        file.write("set_property (GLOBAL PROPERTY OS_FOLDERS ON)\n")
        file.write("\n")
        self.write_compiler_flags(file, output_object.options)
        self.write_custom(file, output_object.options, output_object, output_object.options.project_name);
        file.write("\n")

        # Add the module projects
        self.write_module_projects(file, output_object)
        file.write("\n")

        # Add external projects
        self.write_external_projects(file, output_object)
        file.write("\n")

        # Include paths
        public_only = not output_object.has_internal_access
        modules = output_object.modules[::]
        modules.remove(main_module)
        self.write_include_paths(file, modules, public_only=public_only)
        self.write_include_paths(file, [main_module], public_only=False)
        file.write("\n")

        # Source files
        self.write_source_files(file, main_module)
        file.write("\n")

        # Header files
        self.write_header_files(file, main_module)
        file.write("\n")

        file.write("set (all_files ${source_files} ${header_files})\n")

        # Source groups
        self.write_source_groups(file, main_module)
        file.write("\n")

        # Generate precompiled headers
        self.write_precompiled_header_logic(file, main_module, output_object)

        # Output type
        self.write_output_object_type(file, output_object.options.output_object_type, output_object.options.project_name)
        file.write("\n")

        # Linker flags
        self.write_linker_flags(file, output_object.options.output_object_type, output_object.options.project_name, output_object.options)

        # Link against libraries
        self.write_link_libraries(file, output_object)

        # Preprocessor definitions
        self.write_preprocessor_definitions(file, output_object.options.project_name, main_module)
        file.write("\n")

        file.close()


    def write_output_object_type(self, file, type, name):

        if type == 'static_library':
            file.write("add_library (\"" + name + "\" \"STATIC\" ${all_files})\n")

        elif type == 'dynamic_library':
            file.write("add_library (\"" + name + "\" \"SHARED\" ${all_files})\n")

        elif type == 'executable':
            file.write("add_executable (\"" + name + "\" ${all_files})\n")

        else:
            raise Exception('Unhandled object type: ' + type)


    def write_module_projects(self, file, output_object):
        for m in output_object.modules:
            if 'cmake_lists_path' in m.generator_data:
                module_directory = os.path.dirname( m.generator_data['cmake_lists_path'] )
                module_directory = module_directory.replace('\\', '/')
                self.write_subproject(file, output_object, module_directory)


    def write_external_projects(self, file, output_object):
        for project_path in output_object.options.external_projects:
            self.write_subproject(file, output_object, project_path, project_path)


    def write_subproject(self, file, output_object, subproject_directory, binary_dir=None):
        if not binary_dir is None:
            file.write("add_subdirectory (\"" + subproject_directory + "\" \"" + binary_dir + "\")\n")

        else:
            file.write("add_subdirectory (\"" + subproject_directory + "\")\n")

        file.write("link_directories (\"" + subproject_directory + "\")\n")

    def write_source_files(self, file, module):
        file.write("set (source_files \n")
        for fragment in module.fragments:
            for path in fragment.source_files:
                file.write("     \"" + path + "\"\n")
        file.write(")\n")

    def write_header_files(self, file, module):
        file.write("set (header_files \n")
        for fragment in module.fragments:
            for path in fragment.header_files:
                file.write("     \"" + path + "\"\n")
        file.write(")\n")

    def write_include_paths(self, file, modules, public_only):
        file.write("set (include_paths \n")
        for module in modules:
            paths = module.get_header_search_paths(public_only=public_only)
            for path in paths:
                file.write("     \"" + path + "\"\n")
        file.write(")\n")
        file.write("include_directories (${include_paths})\n")

        file.write("set (include_dependencies \n")
        for module in modules:
            paths = module.get_dependency_search_paths()
            for path in paths:
                file.write("     \"" + path + "\"\n")
        file.write(")\n")
        file.write("include_directories (SYSTEM ${include_dependencies})\n")

    def write_source_groups(self, file, module):
        for fragment in module.fragments:
            for name, paths in fragment.source_groups.iteritems():
                group = self.get_source_group_string(name)
                if group != "":
                    for path in paths:
                        file.write("source_group(\"" + group + "\" FILES \"" + path + "\")\n")

    def append_compiler_flag(self, file, flag, debug, release):

        if debug and release:
            file.write("set (CMAKE_CXX_FLAGS \"${CMAKE_CXX_FLAGS} " + flag + "\")\n")

        elif debug:
            file.write("set (CMAKE_CXX_FLAGS_DEBUG \"${CMAKE_CXX_FLAGS_DEBUG} " + flag + "\")\n")

        elif release:
            file.write("set (CMAKE_CXX_FLAGS_RELEASE \"${CMAKE_CXX_FLAGS_RELEASE} " + flag + "\")\n")

    def append_linker_flag(self, file, target_name, flag, debug, release):

        if debug and release:
            file.write("set_target_properties (\"" + target_name + "\" PROPERTIES LINK_FLAGS \"${LINK_FLAGS} " + flag + "\")\n")

        elif debug:
            file.write("set_target_properties (\"" + target_name + "\" PROPERTIES LINK_FLAGS_DEBUG \"${LINK_FLAGS_DEBUG} " + flag + "\")\n")

        elif release:
            file.write("set_target_properties (\"" + target_name + "\" PROPERTIES LINK_FLAGS_RELEASE \"${LINK_FLAGS_RELEASE} " + flag + "\")\n")

    def append_preprocessor_definition(self, file, key, value, debug, release):

        if isinstance(value, basestring):
            value = "\"" + value + "\""
        else:
            value = str(value)

        if debug and release:
            file.write("set (COMPILE_DEFINITIONS ${COMPILE_DEFINITIONS} " + key + "=" + str(value) + ")\n")

        elif debug:
            file.write("set (COMPILE_DEFINITIONS_DEBUG ${COMPILE_DEFINITIONS_DEBUG} " + key + "=" + str(value) + ")\n")

        elif release:
            file.write("set (COMPILE_DEFINITIONS_RELEASE ${COMPILE_DEFINITIONS_RELEASE} " + key + "=" + str(value) + ")\n")


    def write_custom(self, file, options, output_object, project_name):

        props = options.platform_settings.get_cmake_custom_properties(options, output_object)

        file.write("# Custom properties\n")
        if len(props) == 0:
            file.write("#   <None>\n")
        else:
            for name, val in props.items():
                file.write('set (' + name + ' "' + val + '")\n')
        file.write("\n")

        # Write custom CMake stuff
        options.platform_settings.write_cmake_custom(file, options, project_name)

    # The CMake code written in `append_compiler_flag` to set CMAKE_CXX_FLAGS will add duplicate compiler flags.
    # We thus need the below function to remove these extraneous flags.
    def write_remove_duplicate_compiler_flags(self, file, flag):
        file.write("# Remove duplicates from " + flag + "\n")
        file.write("separate_arguments(" + flag + ")\n")
        file.write("list(REMOVE_DUPLICATES " + flag + ")\n")
        file.write("string(REPLACE \";\" \" \" " + flag + " \"${" + flag + "}\")\n")

    def write_compiler_flags(self, file, options):

        flags = options.platform_settings.get_c_flags(options)

        file.write("# Common C Compiler Flags\n")
        if not 'common' in flags or len(flags['common']) == 0:
            file.write("#   <None>\n")
        else:
            for flag in flags['common']:
                self.append_compiler_flag(file, flag, debug=True, release=True)
            file.write("\n")
            self.write_remove_duplicate_compiler_flags(file, "CMAKE_CXX_FLAGS")
        file.write("\n")

        file.write("# Debug C Compiler Flags\n")
        if not 'debug' in flags or len(flags['debug']) == 0:
            file.write("#   <None>\n")
        else:
            for flag in flags['debug']:
                self.append_compiler_flag(file, flag, debug=True, release=False)
            file.write("\n")
            self.write_remove_duplicate_compiler_flags(file, "CMAKE_CXX_FLAGS_DEBUG")
        file.write("\n")

        file.write("# Release C Compiler Flags\n")
        if not 'release' in flags or len(flags['release']) == 0:
            file.write("#   <None>\n")
        else:
            for flag in flags['release']:
                self.append_compiler_flag(file, flag, debug=False, release=True)
            file.write("\n")
            self.write_remove_duplicate_compiler_flags(file, "CMAKE_CXX_FLAGS_RELEASE")
        file.write("\n")

    def write_linker_flags(self, file, type, target_name, options):

        if type == 'static_library':
            flags = options.platform_settings.get_static_library_linker_flags(options)

        elif type == 'dynamic_library':
            flags = options.platform_settings.get_dynamic_library_linker_flags(options)

        elif type == 'executable':
            flags = options.platform_settings.get_executable_linker_flags(options)

        else:
            raise Exception('Unhandled object type: ' + type)

        file.write("# Common Linker Flags\n")
        if not 'common' in flags or len(flags['common']) == 0:
            file.write("#   <None>\n")
        else:
            for flag in flags['common']:
                self.append_linker_flag(file, target_name, flag, debug=True, release=True)
        file.write("\n")

        file.write("# Debug Linker Flags\n")
        if not 'debug' in flags or len(flags['debug']) == 0:
            file.write("#   <None>\n")
        else:
            for flag in flags['debug']:
                self.append_linker_flag(file, target_name, flag, debug=True, release=False)
        file.write("\n")

        file.write("# Release Linker Flags\n")
        if not 'release' in flags or len(flags['release']) == 0:
            file.write("#   <None>\n")
        else:
            for flag in flags['release']:
                self.append_linker_flag(file, target_name, flag, debug=False, release=True)
        file.write("\n")

    def write_preprocessor_definitions(self, file, target_name, module):

        definitions = module.get_preprocessor_definitions();

        file.write("# Common Preprocessor Defines\n")
        if not 'common' in definitions or len(definitions['common']) == 0:
            file.write("#   <None>\n")
        else:
            for key, value in definitions['common'].iteritems():
                self.append_preprocessor_definition(file, key, value, debug=True, release=True)

        file.write("\n")

        file.write("# Debug Preprocessor Defines\n")
        if not 'debug' in definitions or len(definitions['debug']) == 0:
            file.write("#   <None>\n")
        else:
            for key, value in definitions['debug'].iteritems():
                self.append_preprocessor_definition(file, key, value, debug=True, release=False)

        file.write("\n")

        file.write("# Release Preprocessor Defines\n")
        if not 'release' in definitions or len(definitions['release']) == 0:
            file.write("#   <None>\n")
        else:
            for key, value in definitions['release'].iteritems():
                self.append_preprocessor_definition(file, key, value, debug=False, release=True)
        file.write("\n")

        file.write("target_compile_definitions(\"" + target_name + "\" PRIVATE ${COMPILE_DEFINITIONS} $<$<CONFIG:Debug>:${COMPILE_DEFINITIONS_DEBUG}> $<$<CONFIG:Release>:${COMPILE_DEFINITIONS_RELEASE}>)\n")



    def write_link_libraries(self, file, output_object):

        options = output_object.options

        # Link the project against the module projects
        for m in output_object.modules:
            if 'cmake_lists_path' in m.generator_data:
                module_directory = os.path.dirname( m.generator_data['cmake_lists_path'] )
                file.write("target_link_libraries (\"" + options.project_name + "\" \"" + m.project_name + "\")\n")
        file.write("\n")

        # Link against other output_object source
        for subproject_directory in options.external_projects:
            # Determine the project name so we can link against it
            cmakelists_path = os.path.join(subproject_directory, 'CMakeLists.txt')
            if os.path.isfile(cmakelists_path):
                library_name = self.get_project_name_from_cmakelists(cmakelists_path)
                if not library_name is None:
                    file.write("target_link_libraries (\"" + options.project_name + "\" \"" + library_name + "\")\n")

        # Sort the libraries into categories
        libraries = options.platform_settings.sort_cmake_libraries( output_object.get_link_libraries() )

        # Link against external dependencies
        libs = libraries['external']
        file.write("# External library dependencies\n")
        if len(libs) == 0:
            file.write("#   <None>\n")
        else:
            for path in libs:
                name = os.path.basename(path)
                file.write("add_library (\"" + name + "\" STATIC IMPORTED)\n")
                file.write("set_target_properties (\"" + name + "\" PROPERTIES IMPORTED_LOCATION \"" + path + "\")\n")
                file.write("target_link_libraries (\"" + options.project_name + "\" \"" + name + "\")\n")
            file.write("\n")

        # Platform-specific system libraries
        libs = libraries['system']
        file.write("# Platform-specific system libraries\n")
        if len(libs) == 0:
            file.write("#   <None>\n")
        else:
            file.write('target_link_libraries (\"' + options.project_name + '\"\n');
            for lib in libs:
                file.write('    \"' + lib + '\"\n')
            file.write(')\n');
        file.write("\n")


    def write_precompiled_header_logic(self, file, module, output_object):

        output_object.options.platform_settings.write_cmake_precompiled_headers(file, module, output_object)


    def get_project_name_from_cmakelists(self, cmakelists_path):

        with open(cmakelists_path) as file:
            lines = file.read().splitlines()

        for line in lines:
            line = line.strip()

            result = re.match("\\s*project\\s+\\(\\s*['\"](.*)['\"]\\s*\\)\\s*", line, re.I)
            if result:
                return result.group(1)

        return None


    def run_generator(self, directory, options):

        if not options.cmake_options.no_generate:

            # Build the CMake command line
            executable = build_tools.find_cmake(options.cmake_options.custom_path)

            print('Found CMake at: ' + executable)

            shell_args = [executable]

            # Specify the compiler
            generator = options.platform_settings.get_cmake_generator_name(options)
            shell_args.extend(['-G', generator])

            # Set the architecture
            shell_args.extend(['-DPROJECT_ARCH:STRING=\"' + options.architecture + '\"'])

            # Add any custom parameters
            shell_args.extend(options.platform_settings.get_cmake_custom_params(options))

            print('Running CMake:')
            print(" ".join(str(x) for x in shell_args))

            # Change directory to the CMakeLists.txt
            os.chdir(directory)

            # Run CMake to generate the project file
            subprocess.call(shell_args)

            # Do processing of the project file
            options.platform_settings.post_process_generated_files(directory, options)

        else:
            print('Not running cmake because of --cmake-no-generate switch')

# Set up inheritance
build_generator.BuildGenerator.register(CMakeBuildGenerator)
