import build_generator
import os.path
import glob

class CocoaPodsBuildGenerator(build_generator.BuildGenerator):

    def output_hierarchical_structure(self):
        # The podspecs produced by this generator are single files that don't
        # need to be their own respective folders.
        return False

    def produce_module_generator_files(self, source_module, output_directory, output_object):
        source_module.output_directory = output_directory
        podspec_path = os.path.join(output_directory, source_module.project_name + '.podspec')

        print('Generating ' + podspec_path + '...')

        file = open(podspec_path, 'w')

        file.write('Pod::Spec.new do |twitchmodule|\n')

        for fragment in source_module.fragments:
            self.write_fragment_subspec(file, fragment, source_module.output_directory)

        dependency_directories = []
        for search_path in output_object.options.fragment_search_paths:
            dependency_directories.append(os.path.join(search_path, 'dependencies'))

        dependency_search_paths = []
        for search_path in source_module.get_header_search_paths():
            for dependency_directory in dependency_directories:
                if search_path.startswith(dependency_directory):
                    name = ''
                    for component in search_path[len(dependency_directory):].split('/'):
                        if len(component) > 0:
                            name = component
                            break
                    self.write_dependency_subspec(file, name, search_path, output_directory)
                    break

        file.write('  twitchmodule.name                  = \'{}\'\n'.format(source_module.project_name))
        file.write('  twitchmodule.summary               = \'{}\'\n'.format(source_module.project_name))

        self.write_misc_data(file)
        self.write_external_libraries(file, source_module)
        self.write_platform(file, output_object.options)
        self.write_xcode_config(file, self.get_c_flags(output_object), source_module.get_preprocessor_definitions())

        if source_module.project_name != "core":
            file.write('  twitchmodule.dependency \'core\'       , \'0.1.0\'\n')

        file.write('end')

        pass

    def write_external_libraries(self, file, source_module):
        # For some reason we have to add c++ here to get cocoapods to work
        libraries = ['c++']
        frameworks = []
        framework_prefix = '-framework '
        library_prefix = '-l'
        for link_target in source_module.get_link_libraries():
            if link_target.startswith(framework_prefix):
                frameworks.append(link_target[len(framework_prefix):])
            elif link_target.startswith(library_prefix):
                libraries.append(link_target[len(library_prefix):])

        is_first_pass = True
        if len(libraries):
            file.write('  twitchmodule.libraries = ')
            for library in libraries:
                if is_first_pass:
                    is_first_pass = False
                else:
                    file.write(',\n                           ')
                file.write('\'{}\''.format(library))
        file.write('\n')

        is_first_pass = True
        if len(frameworks):
            file.write('  twitchmodule.frameworks = ')
            for framework in frameworks:
                if is_first_pass:
                    is_first_pass = False
                else:
                    file.write(',\n                            ')
                file.write('\'{}\''.format(framework))
        file.write('\n')


    def get_c_flags(self, output_object):
        c_flags = output_object.options.platform_settings.get_c_flags(output_object.options)['common']

        # For some reason, this flag causes issues with compilation when using cocoapods, so we remove it.
        # It's already specified in some of the other configuration anyway.
        c_flags.remove('-std=c++14')

        # Also, the code generated by cocoapods itself causes the auto-import warning, so we should disable
        # it or we won't be able to compile.
        c_flags.append('-Wno-auto-import')

        return c_flags

    def write_fragment_subspec(self, file, fragment, output_directory):
        file.write('  twitchmodule.subspec \'{}\' do |fragment|\n'.format(fragment.name))
        file.write('    fragment.header_mappings_dir = \'{}\'\n'.format(os.path.relpath(os.path.join(fragment.root_path, 'include'), output_directory)))
        if len(fragment.source_files) > 0:
            file.write('    fragment.source_files        = ')
            self.write_relative_paths(file, fragment.source_files + fragment.header_files, output_directory, '                                   ')
            file.write('\n')

        if len(fragment.header_files) > 0:
            file.write('    fragment.public_header_files = ')
            self.write_relative_paths(file, fragment.header_files, output_directory, '                                   ')
            file.write('\n')

        file.write('  end\n\n')

    def write_dependency_subspec(self, file, name, include_path, output_directory):
        header_paths = []
        for (dirpath, dirnames, filenames) in os.walk(include_path):
            for filename in filenames:
                (root, ext) = os.path.splitext(filename)
                if (ext == '.h') or (ext == '.hpp'):
                    header_paths.append(os.path.join(dirpath, filename))

        if len(header_paths) > 0:
            file.write('  twitchmodule.subspec \'{}\' do |twitchdependency|\n'.format(name))
            file.write('    twitchdependency.source_files        = \'\'\n')
            file.write('    twitchdependency.header_mappings_dir = \'{}\'\n'.format(os.path.relpath(include_path, output_directory)))
            file.write('    twitchdependency.public_header_files = ')

            is_first_pass = True
            for header_path in header_paths:
                if is_first_pass:
                    is_first_pass = False
                else:
                    file.write(',\n                                           ')
                file.write('\'{}\''.format(os.path.relpath(header_path, output_directory)))
            file.write('\n')
            file.write('  end\n\n')

    def write_relative_paths(self, file, paths, base_path, padding):
        is_first_pass = True
        for path in paths:
            if is_first_pass:
                is_first_pass = False
            else:
                file.write(',\n{}'.format(padding))
            file.write('\'{}\''.format(os.path.relpath(path, base_path)))

    def produce_main_generator_files(self, output_directory, output_object):
        main_module = output_object.get_module('__main__')
        main_module.output_directory = output_directory

        podspec_path = os.path.join(output_directory, output_object.options.project_name + '.podspec')

        print('Generating ' + podspec_path + '...')

        file = open(podspec_path, 'w')
        file.write('Pod::Spec.new do |twitchmodule|\n')

        file.write('  twitchmodule.name                  = \'{}\'\n'.format(output_object.options.project_name))
        file.write('  twitchmodule.summary               = \'{}\'\n'.format(output_object.options.project_name))

        self.write_misc_data(file)
        self.write_platform(file, output_object.options)
        self.write_xcode_config(file, self.get_c_flags(output_object), main_module.get_preprocessor_definitions())

        # The main module just has the one fragment.
        main_fragment = main_module.fragments[0]

        if len(main_fragment.source_files) > 0:
            file.write('  twitchmodule.source_files        = ')
            self.write_relative_paths(file, main_fragment.source_files, output_directory, '                                       ')
            file.write('\n')

        if len(main_fragment.header_files) > 0:
            file.write('  twitchmodule.public_header_files = ')
            self.write_relative_paths(file, main_fragment.header_files, output_directory, '                                       ')
            file.write('\n')


        for module in output_object.get_modules():
            if module != main_module:
                file.write('  twitchmodule.dependency \'{}\'       , \'0.1.0\'\n'.format(module.project_name))

        file.write('end')

    def run_generator(self, directory, options):
        # We don't actually generate a project with the cocoapods directly.
        pass

    def write_xcode_config(self, file, c_flags, preprocessor_defs):
        file.write('  twitchmodule.pod_target_xcconfig   = {\n')
        file.write('                                 \'CLANG_CXX_LANGUAGE_STANDARD\'  => \'c++14\',\n')
        file.write('                                 \'CLANG_CXX_LIBRARY\'            => \'libc++\',\n')

        file.write('                                 \'GCC_PREPROCESSOR_DEFINITIONS\' => \'$(TW_GCC_PREPROCESSOR_DEFINITIONS_$(TW_CONFIGURATION)) \'')

        for key, value in preprocessor_defs.iteritems():
            if value is None:
                value = '1'

            file.write(' + \n                                                                   \'{}={} \''.format(key, value))
        file.write(',\n')

        if len(c_flags) > 0:
            file.write('                                 \'OTHER_CFLAGS\'                 => \'$(inherited) \'')
            for flag in c_flags:
                file.write(' + \n                                                                   \'{} \''.format(flag))
            file.write(',\n')

        file.write('                                 \'TW_CONFIGURATION\'                        => \'$(TW_CONFIGURATION_$(TW_CONFIGURATION__$(CONFIGURATION)))\',\n')
        file.write('                                 \'TW_CONFIGURATION_\'                       => \'Release\', # The default value for $(TW_CONFIGURATION)\n')
        file.write('                                 \'TW_CONFIGURATION_Debug\'                  => \'Debug\',\n')
        file.write('                                 \'TW_CONFIGURATION__Debug\'                 => \'Debug\',\n')
        file.write('                                 \'TW_GCC_PREPROCESSOR_DEFINITIONS_Debug\'   => \'\',\n')
        file.write('                                 \'TW_GCC_PREPROCESSOR_DEFINITIONS_Release\' => \'NDEBUG=1\'\n')

        file.write('                               }\n')

    def write_misc_data(self, file):
        file.write('  twitchmodule.authors               = { \'None\' => \'none@example.com\' }\n\n')
        file.write('  twitchmodule.homepage              = \'https://example.com\'\n')
        file.write('  twitchmodule.license               = { :type => \'None\', :file => \'LICENSE.txt\' }\n')
        file.write('  twitchmodule.source                = { :git => \'https://example.com/example.git\' }\n')
        file.write('  twitchmodule.version               = \'0.1.0\'\n')

    def write_platform(self, file, options):
        if options.target == 'ios':
            file.write('  twitchmodule.platform              = :ios, \'8.0\'\n')
        else:
            file.write('  twitchmodule.platform              = :osx, \'10.12\'\n')
