import os
import build_types
import cmake
import cocoapods


class DarwinPlatformSettings:

    def __init__(self):

        self.name = 'darwin'
        self.shared_lib_extension = '.dylib'
        self.static_lib_extension = '.so'

    def requires_monolithic_project(self, options):
        return False

    def link_against_openssl(self, options, fragment):
        # TODO: Link against OpenSSL if ever needed
        #fragment.add_dependency_search_paths( os.path.join(fragment.root_path, '..', 'modules', 'core', 'dependencies', 'openssl', 'lib', options.primary_platform, options.architecture) )
        #fragment.add_link_libraries( os.path.join(fragment.root_path, '..', 'modules', 'core', 'dependencies', 'openssl', 'lib', options.primary_platform, options.architecture, 'libcrypto.lib') )
        #fragment.add_link_libraries( os.path.join(fragment.root_path, '..', 'modules', 'core', 'dependencies', 'openssl', 'lib', options.primary_platform, options.architecture, 'libssl.lib') )
        pass

    def add_platform_preprocessor_definitions(self, options, fragment):
        pass

    def get_c_flags(self, options):
        """Returns platform-specific C compiler flags."""
        flags = {
            'common': [
                '-std=c++14',
                '-stdlib=libc++',
                '-g',
                '-fmessage-length=0',
                '-fobjc-arc',

                # Enabled Warnings
                '-pedantic',
                '-Wall',
                '-Werror',
                '-Weverything',
                # Disabled Warnings
                '-Wno-unknown-warning-option',
                '-Wno-missing-noreturn',
                '-Wno-exit-time-destructors',
                '-Wno-undef',
                '-Wno-c++98-compat-pedantic',
                '-Wno-padded',
                '-Wno-global-constructors',
                '-Wno-covered-switch-default',
                '-Wno-switch-enum',
                '-Wno-weak-vtables',
                '-Wno-documentation',
                '-Wno-documentation-unknown-command',
                '-Wno-deprecated-declarations',
                '-Wno-reserved-id-macro',
                '-Wno-shift-sign-overflow',
                '-Wno-unused-macros',
                '-Wno-unused-member-function',
                '-Wno-suggest-override',
                '-Wno-suggest-destructor-override'
            ],
            'debug': [],
            'release': []
        }

        #flags['common'].append('-g')

        flags['common'].append('-g2')

        return flags

    def get_static_library_linker_flags(self, options):
        flags = {
            'common': []
        }

        return flags

    def get_dynamic_library_linker_flags(self, options):
        flags = {
            'common': []
        }

        return flags

    def get_executable_linker_flags(self, options):
        flags = {
            'common': []
        }

        return flags

    def write_symbol_exports_file(self, path_prefix, symbols):
        path = path_prefix + '.exp'
        file = open(path, 'w')
        for symbol in symbols:
            file.write('_' + symbol + '\n')
        file.write('\n')
        file.close()
        return path

    def create_build_generator(self, options):
        generator_name = options.generator
        if not generator_name:
            generator_name = "cmake"

        if generator_name == "cmake":
            return cmake.CMakeBuildGenerator()
        elif generator_name == "cocoapods":
            return cocoapods.CocoaPodsBuildGenerator()
        else:
            raise Exception('Illegal generator type "{}" specified for {} platform'.format(generator_name, self.name))


    # CMake-specific functions #######################################################################

    def read_cmake_previous_cmakelists_file(self, cmakelists_path):
        """Reads and parses the previous CMake lists file to retain any properties."""
        pass

    def get_cmake_custom_properties(self, options, output_object):
        props = {
            'CMAKE_XCODE_ATTRIBUTE_GCC_VERSION': 'com.apple.compilers.llvm.clang.1_0',
            'CMAKE_XCODE_ATTRIBUTE_CLANG_CXX_LANGUAGE_STANDARD': 'c++14',
            'CMAKE_XCODE_ATTRIBUTE_CLANG_CXX_LIBRARY': 'libc++'
        }

        if options.target == 'ios':
            props['CMAKE_MACOSX_BUNDLE'] = 'YES'
            #props['CMAKE_XCODE_ATTRIBUTE_CODE_SIGNING_REQUIRED'] = 'NO'
            props['XCODE_ATTRIBUTE_CODE_SIGN_IDENTITY'] = 'iOS Developer'
            props['APP_TYPE'] = 'MACOSX_BUNDLE'
            props['CMAKE_OSX_SYSROOT'] = 'iphoneos'

            # Set the CMAKE_OSX_ARCHITECTURES flag so that ONLY_ACTIVE_ARCH is set to NO by default for iOS
            # http://stackoverflow.com/a/22689917/571778
            props['CMAKE_OSX_ARCHITECTURES'] = '$(ARCHS_STANDARD)'

            if options.has_setting('TTV_IOS_DEPLOYMENT_TARGET'):
                props['CMAKE_XCODE_ATTRIBUTE_IPHONEOS_DEPLOYMENT_TARGET'] = options.settings['TTV_IOS_DEPLOYMENT_TARGET']
        else:
            if options.has_setting('TTV_OSX_DEPLOYMENT_TARGET'):
                props['CMAKE_OSX_DEPLOYMENT_TARGET'] = options.settings['TTV_OSX_DEPLOYMENT_TARGET']

        return props

    def write_cmake_custom(self, file, options, project_name):
        pass

    def write_cmake_precompiled_headers(self, file, module, output_object):
        # TODO: Implement precompiled headers in Xcode
        pass

    def sort_cmake_libraries(self, list):
        """Given a flat list of dependencies this sorts them into categories so they can be added properly by CMake"""
        result = {
            'external': [],
            'system': []
        }

        for x in list:
            if x.startswith('-'):
                result['system'].append(x)
            else:
                result['external'].append(x)

        return result

    def get_cmake_custom_params(self, config):
        """Retrieves any custom parameters to pass to CMake"""
        return []

    def get_cmake_generator_name(self, options):

        if options.compiler == 'xcode':
            return "Xcode"

        else:
            raise Exception('Unhandled platform options, unable to determine CMake generator')

    def post_process_generated_files(self, directory, options):

        pbxproj_path = os.path.join(directory, options.project_name + '.xcodeproj', 'project.pbxproj')

        if os.path.isfile(pbxproj_path):

            # Read in lines from the pbxproj file
            file = open(pbxproj_path, 'r')
            lines = file.readlines()
            file.close()

            # CMake generates extraneous attributes in the pbxproj that is incompatible with CocoaPods/XcodeProj
            # https://github.com/CocoaPods/CocoaPods/issues/6325
            # In PBXProject section, need to remove `buildSettings = {};` and `buildStyles = ( ... );`
            in_pbxproject = False
            in_pbxproject_buildSettings = False
            in_pbxproject_buildStyles = False

            file = open(pbxproj_path, 'w')
            for line in lines:

                if "isa = PBXProject;" in line:
                    in_pbxproject = True

                if in_pbxproject:
                    if "buildSettings = {" in line:
                        in_pbxproject_buildSettings = True
                        continue
                    if in_pbxproject_buildSettings:
                        if "};" in line:
                            in_pbxproject_buildSettings = False
                        continue
                    if "buildStyles = (" in line:
                        in_pbxproject_buildStyles = True
                        continue
                    if in_pbxproject_buildStyles:
                        if ");" in line:
                            in_pbxproject_buildSettings = False
                            in_pbxproject = False
                        continue

                # Change paths in the pbxproj file to be relative to the output directory
                if options.relative_paths:
                    file.write(line.replace(directory, "$(SRCROOT)"))
                else:
                    file.write(line)
            file.close()

def get_platform_settings():

    return DarwinPlatformSettings()

