import os
import build_types
import cmake
import re
import uuid


class OrbisPlatformSettings:

    def __init__(self):

        self.name = 'orbis'
        self.shared_lib_extension = '.so'
        self.static_lib_extension = '.a'
        self.cmakelists_property_cache = {}

    def requires_monolithic_project(self, options):
        return False

    def link_against_openssl(self, options, fragment):
        pass

    def add_platform_preprocessor_definitions(self, options, fragment):
        fragment.add_preprocessor_definition('TTV_INCLUDE_CPPTRANSITIONS')

    def get_c_flags(self, options):
        """Returns platform-specific C compiler flags."""
        flags = {
            'common': [
                '-std=c++14',
                '-stdlib=libc++',
                '-g',

                '-fexceptions',
                '-frtti'
            ],
            'debug': [],
            'release': [
                '-funroll-loops',
                '-flto'
            ]
        }
        
        return flags

    def get_static_library_linker_flags(self, options):
        """Returns platform-specific linker flags for linking a static library."""
        flags = {
            'common': [],
            'debug': [],
            'release': []
        }

        return flags

    def get_dynamic_library_linker_flags(self, options):
        """Returns platform-specific linker flags for linking a dynamic library."""
        flags = {
            'common': [],
            'debug': [],
            'release': []
        }

        return flags

    def get_executable_linker_flags(self, options):
        """Returns platform-specific linker flags for linking an executable."""
        flags = {
            'common': [],
            'debug': [],
            'release': []
        }

        return flags

    def write_symbol_exports_file(self, path_prefix, symbols):
        path = path_prefix + '.def'
        file = open(path, 'w')
        file.write("VERSION 0.1\n")
        file.write("EXPORTS\n")
        for symbol in symbols:
            file.write('\t' + symbol + '\n')
        file.write('\n')    
        file.close()
        return path 

    def create_build_generator(self, options):
        return cmake.CMakeBuildGenerator()


    # CMake-specific functions #######################################################################

    def read_cmake_previous_cmakelists_file(self, cmakelists_path):
        """Reads and parses the previous CMake lists file to retain any properties."""

        # Look for the project GUID
        with open(cmakelists_path) as file:
            lines = file.read().splitlines()

        for line in lines:
            line = line.strip()

            result = re.match("^\\s*set\\((\\w+)_GUID_CMAKE\s+\\\"(.+)\\\"\\s+.+$", line, re.I)
            if result:
                guid = result.group(2)
                self.cmakelists_property_cache['project_guid'] = guid
            elif 'project_guid' in self.cmakelists_property_cache:
                del self.cmakelists_property_cache['project_guid']

    def get_cmake_custom_properties(self, options, output_object):
        return {
        }

    def write_cmake_custom(self, file, options, project_name):
        # Set the Visual Studio project GUID
        if 'project_guid' in self.cmakelists_property_cache:
            guid = self.cmakelists_property_cache['project_guid'].upper()
            print('Reusing project GUID for ' + project_name + ': ' + guid)
        else:
            guid = str(uuid.uuid4()).upper()
        file.write("set(" + project_name + "_GUID_CMAKE \"" + guid + "\" CACHE INTERNAL \"Project GUID\")\n")

    def write_cmake_precompiled_headers(self, file, module, output_object):
        pass

    def sort_cmake_libraries(self, list):
        """Given a flat list of dependencies this sorts them into categories so they can be added properly by CMake"""
        result = {
            'external': [],
            'system': []
        }

        return result

    def get_cmake_custom_params(self, config):
        """Retrieves any custom parameters to pass to CMake"""
        return []

    def get_cmake_generator_name(self, options):

        architecture = options.architecture

        return "Visual Studio 14 2015"

def get_platform_settings():

    return OrbisPlatformSettings()
