import os
import build_types
import cmake
import re
import uuid


class Win32PlatformSettings:

    def __init__(self):

        self.name = 'win32'
        self.shared_lib_extension = '.dll'
        self.static_lib_extension = '.lib'
        self.cmakelists_property_cache = {}

    def requires_monolithic_project(self, options):
        # UWP apps that export CX types need to be built with all C++/CX code in the DLL project
        # or the types will not be exported
        return options.target == 'uwp'

    def link_against_openssl(self, options, fragment):
        fragment.add_dependency_search_paths( os.path.join(fragment.root_path, '..', 'dependencies', 'openssl', 'lib', options.compiler, options.architecture) )
        fragment.add_link_libraries( os.path.join(fragment.root_path, '..', 'dependencies', 'openssl', 'lib', options.compiler, options.architecture, 'libcrypto.lib') )
        fragment.add_link_libraries( os.path.join(fragment.root_path, '..', 'dependencies', 'openssl', 'lib', options.compiler, options.architecture, 'libssl.lib') )

    def add_platform_preprocessor_definitions(self, options, fragment):
        fragment.add_preprocessor_definition('_SCL_SECURE_NO_WARNINGS')
        fragment.add_preprocessor_definition('_CRT_SECURE_NO_WARNINGS')
        fragment.add_preprocessor_definition('WIN32_LEAN_AND_MEAN')
        fragment.add_preprocessor_definition('STRICT')
        fragment.add_preprocessor_definition('NOMINMAX')
        fragment.add_preprocessor_definition('_VARIADIC_MAX', 10)

        # Handle UWP apps
        if options.target == 'uwp':
            fragment.add_preprocessor_definition('_WINRT_DLL')
            fragment.add_preprocessor_definition('__WRL_NO_DEFAULT_LIB__')


    def get_c_flags(self, options):

        """Returns platform-specific C compiler flags."""
        flags = {
            'common': [],
            'debug': [],
            'release': []
        }

        # Allow large obj files
        flags['common'].append("/bigobj")

        # SDL checks
        flags['common'].append('/sdl')

        # Multiprocessor build
        # https://blog.kitware.com/cmake-building-with-all-your-cores/
        flags['common'].append('/MP')

        # Debug information format
        flags['common'].append("/Zi")

        # Disable warning C4068: unknown pragma
        flags['common'].append("/wd4068")

        if options.target == 'uwp':
            flags['common'].append("/ZW")
            #flags['common'].append("/ZW:nostdlib")
            flags['common'].append("/EHsc")

        # Multithreaded vs Multithreaded DLL
        # TODO: Add an option to explicitly configure Multithreaded vs Multithreaded DLL
        if options.output_object_type == 'dynamic_library' or options.target == 'uwp':
            flags['debug'].append("/MDd")
            flags['release'].append("/MD")
        else:
            flags['debug'].append("/MTd")
            flags['release'].append("/MT")

        # Optimization
        flags['debug'].append("/Od")

        # Run-time checks
        # https://docs.microsoft.com/en-us/previous-versions/8wtf2dfz(v=vs.140)
        flags['debug'].append("/RTC1")

        # Inline function expansion
        flags['debug'].append("/Ob0")
        flags['release'].append("/Ob2")
        
        flags['release'].append("/Ot") # Favor Speed
        flags['release'].append("/GT") # Enable Fiber-Safe Optimizations
        flags['release'].append("/GL") # Whole Program Optimization
        flags['release'].append("/GF") # Eliminate Duplicate Strings

        return flags

    def get_static_library_linker_flags(self, options):
        """Returns platform-specific linker flags for linking a static library."""
        flags = {
            'common': [
            ],
            'debug': [
                '/NODEFAULTLIB:\\"libcpmt.lib\\"'
            ],
            'release': [
            ]
        }

        return flags

    def get_dynamic_library_linker_flags(self, options):
        """Returns platform-specific linker flags for linking a dynamic library."""
        flags = {
            'common': [
            ],
            'debug': [
                '/NODEFAULTLIB:\\"libcpmt.lib\\"'
            ],
            'release': [
            ]
        }

        if options.target == 'uwp':
            # Generate Windows Metadata
            flags['common'].append('/WINMD')

            # The root namespace for generated bindings is Twitch so the file needs to have that name
            flags['common'].append('/WINMDFILE:$(OutDir)Twitch.winmd')

        return flags

    def get_executable_linker_flags(self, options):
        """Returns platform-specific linker flags for linking an executable."""
        flags = {
            'common': [
            ],
            'debug': [
                '/NODEFAULTLIB:\\"libcpmt.lib\\"'
            ],
            'release': [
                '/LTCG' # Whole Program Optimization
            ]
        }

        return flags

    def write_symbol_exports_file(self, path_prefix, symbols):
        path = path_prefix + '.def'
        file = open(path, 'w')
        file.write("VERSION 0.1\n")
        file.write("EXPORTS\n")
        for symbol in symbols:
            file.write('\t' + symbol + '\n')
        file.write('\n')
        file.close()
        return path

    def create_build_generator(self, options):
        return cmake.CMakeBuildGenerator()


    # CMake-specific functions #######################################################################

    def read_cmake_previous_cmakelists_file(self, cmakelists_path):
        """Reads and parses the previous CMake lists file to retain any properties."""

        # Look for the project GUID
        with open(cmakelists_path) as file:
            lines = file.read().splitlines()

        for line in lines:
            line = line.strip()

            result = re.match("^\\s*set\\((\\w+)_GUID_CMAKE\s+\\\"(.+)\\\"\\s+.+$", line, re.I)
            if result:
                guid = result.group(2)
                self.cmakelists_property_cache['project_guid'] = guid
            elif 'project_guid' in self.cmakelists_property_cache:
                del self.cmakelists_property_cache['project_guid']

    def get_cmake_custom_properties(self, options, output_object):
        return {
        }

    def write_cmake_custom(self, file, options, project_name):
        # Set the Visual Studio project GUID
        if 'project_guid' in self.cmakelists_property_cache:
            guid = self.cmakelists_property_cache['project_guid'].upper()
            print('Reusing project GUID for ' + project_name + ': ' + guid)
        else:
            guid = str(uuid.uuid4()).upper()
        file.write("set(" + project_name + "_GUID_CMAKE \"" + guid + "\" CACHE INTERNAL \"Project GUID\")\n")

    def write_cmake_precompiled_headers(self, file, module, output_object):

        # Multiple precompiled headers don't work properly within the same project in Visual Studio
        if output_object.options.monolithic_project:
            return

        pch_create_map = module.get_precompiled_header_create_map()
        pch_use_map = module.get_precompiled_header_use_map()

        # Create
        if len(pch_create_map) > 0:
            file.write("# Create precompiled header\n")

            for pch_cpp_path in pch_create_map:

                pch_h_path = pch_create_map[pch_cpp_path].replace('\\', '/')
                pch_cpp_path = pch_cpp_path.replace('\\', '/')

                create_flags = '"/Yc\\\"' + pch_h_path + '\\\""'
                file.write('set_source_files_properties (\"' + pch_cpp_path + '\" PROPERTIES COMPILE_FLAGS ' + create_flags + ')\n')
            file.write("\n")

        # Use
        if len(pch_use_map) > 0:
            file.write("# Use precompiled headers\n")

            for cpp_path in pch_use_map:
                pch_h_path = pch_use_map[cpp_path].replace('\\', '/')
                cpp_path = cpp_path.replace('\\', '/')

                use_flags = '"/Yu\\\"' + pch_h_path + '\\\""'
                file.write('set_source_files_properties (\"' + cpp_path + '\" PROPERTIES COMPILE_FLAGS ' + use_flags + ')\n')
            file.write("\n")


    def sort_cmake_libraries(self, list):
        """Given a flat list of dependencies this sorts them into categories so they can be added properly by CMake"""
        result = {
            'external': [],
            'system': []
        }

        for x in list:
            if x.startswith('-'):
                result['system'].append(x)
            else:
                result['external'].append(x)

        return result

    def get_cmake_custom_params(self, config):
        """Retrieves any custom parameters to pass to CMake"""
        if config.target == 'uwp':
            return [
                '-DCMAKE_SYSTEM_NAME=WindowsStore',
                '-DCMAKE_SYSTEM_VERSION=10.0'
            ]
        return []

    def get_cmake_generator_name(self, options):

        architecture = options.architecture
        if architecture == 'x64':
            architecture = 'x86_64'

        if options.compiler == "vs2010":
            if architecture == 'x86_64':
                return "Visual Studio 10 Win64"
            else:
                return "Visual Studio 10"

        elif options.compiler == "vs2012":
            if architecture == 'x86_64':
                return "Visual Studio 11 Win64"
            else:
                return "Visual Studio 11"

        elif options.compiler == "vs2013":
            if architecture == 'x86_64':
                return "Visual Studio 12 Win64"
            else:
                return "Visual Studio 12"

        elif options.compiler == "vs2015":
            if architecture == 'x86_64':
                return "Visual Studio 14 Win64"
            else:
                return "Visual Studio 14"

        elif options.compiler == "vs2017":
            if architecture == 'x86_64':
                return "Visual Studio 15 Win64"
            else:
                return "Visual Studio 15"

        else:
            raise Exception('Unhandled platform options, unable to determine CMake generator')

    def post_process_generated_files(self, directory, options):
        pass

def get_platform_settings():

    return Win32PlatformSettings()
