import os
import abc
import shutil
import re
from pprint import pprint
import json
import filecmp


def read_file_lines(path):
    with open(path) as file:
        lines = file.read().splitlines()
    return map(lambda line : line.rstrip(), lines)


def write_file_lines(path, lines, only_if_changed=True):

    def do_write(path):
        with open(path, 'w') as file:
            for line in lines:
                file.write(line + "\n")

    # Determine if we need to write the file
    write = not only_if_changed
    if only_if_changed:
        # Doesn't exist yet
        if not os.path.isfile(path):
            write = True
        # Write to a temporary file and compare them
        else:
            temp_path = path + '_temp'
            #print("Comparing " + path + ' and ' + temp_path)
            do_write(temp_path)
            # Compare files
            write = not filecmp.cmp(path, temp_path, shallow=False)
            # Delete temp file
            if os.path.isfile(temp_path):
                os.remove(temp_path)

    if write:
        do_write(path)


def replace_all(lines, key, value):
    result = []
    for line in lines:
        line = line.replace(key, value)
        result.append(line)
    return result


def get_line_indentation(line):

    """ Determines the amount of indentation on the given source code line. """

    trimmed = line.lstrip()
    return line[0:len(line) - len(trimmed)]


def fix_line_indentation(line, indent):

    # Convert leading tabs to the given indent
    trimmed = line.lstrip()
    num_whitespace_chars = len(line) - len(trimmed)
    whitespace = line[0:num_whitespace_chars]
    whitespace = whitespace.replace('\t', indent)
    return whitespace + trimmed


def find_line_range_marker(tag_name, lines, must_exist=True):

    """
    Searches the given lines for the given tag and returns the index.
    If not found then None is returned.
    """

    tag = '<<' + tag_name + '>>'
    index = -1;

    for i in range(0, len(lines)):

        line = lines[i].upper()

        if line.find(tag) >= 0:
            return i;

    if must_exist:
        raise Exception("Tag not found: " + tag_name)

    return None


def find_line_range_markers(tag_name, lines, must_exist=True):

    """Searches the given lines for the given start and end tags and returns the indices."""

    start_tag = '<<' + tag_name + '_START>>'
    end_tag = '<<' + tag_name + '_END>>'

    start_index = -1;
    end_index = -1;

    for i in range(0, len(lines)):

        line = lines[i].upper()

        if line.find(start_tag) >= 0:
            start_index = i;

        elif line.find(end_tag) >= 0:
            end_index = i;

    if start_index < 0 or end_index < 0:
        if must_exist:
            raise Exception("Substitution range not found: " + tag_name)
        return None
    else:
        return [start_index, end_index]


def replace_tag_line(tag_name, all_lines, replacement_lines, indent, must_exist=True):

    """ Replaces the tag line with the given lines. """

    index = find_line_range_marker(tag_name, all_lines, must_exist)
    if index is None:
        return all_lines

    indent = get_line_indentation(all_lines[index])

    pre = all_lines[:index]
    post = all_lines[index+1:]

    pre = map(lambda line : fix_line_indentation(line, indent), pre)
    post = map(lambda line : fix_line_indentation(line, indent), post)

    replacement_lines = map(lambda line : indent + line, replacement_lines)
    return pre + replacement_lines + post


def replace_lines_range(tag_name, all_lines, replacement_lines, indent, must_exist=True):

    """Replaces the lines in between the given start and end tags."""

    markers = find_line_range_markers(tag_name, all_lines, must_exist)
    if not markers:
        return all_lines

    indent = get_line_indentation(all_lines[ markers[0] ])

    pre = all_lines[:markers[0]+1]
    post = all_lines[markers[1]:]

    pre = map(lambda line : fix_line_indentation(line, indent), pre)
    post = map(lambda line : fix_line_indentation(line, indent), post)

    replacement_lines = map(lambda line : indent + line, replacement_lines)
    return pre + replacement_lines + post


def fix_path(path):

    """Standardizes path formats to use / as the delimiter."""

    path = os.path.realpath(path)
    return path.replace('\\', '/')


def get_lines_from_text(text, trim=False, allow_blanks=False):

    lines = text.splitlines()
    if trim:
        lines = [x.strip() for x in lines]
    if not allow_blanks:
        lines = [x for x in lines if len(x) > 0]

    return lines
