import abc
import imp
import os
import re
from pprint import pprint


def get_native_type_name_components(type_name):

    """ Given a fully qualified native typename split it into its components. """

    index = type_name.rfind('::')
    namespace = type_name[0:index]
    if namespace.startswith('::'):
        namespace = namespace[2:]
    name = type_name[index+2:]

    namespace_tokens = filter(lambda x: x != '', namespace.split('::'))

    return {
        'name': name,
        'namespace': namespace,
        'namespace_tokens': namespace_tokens
    }


def is_output_parameter(native_type_name):

    """ Determines if the given type represents an output parameter. """

    tokens = native_type_name.split()
    return tokens[-1] == '&'


def is_pointer_type(native_type_name):
    tokens = native_type_name.split()
    if tokens[-1] == '*':
        return True
    elif len(tokens) > 1 and tokens[-2] == '*':
        return True
    return False


def is_array_member(member_info, owning_type_info):

    """ Returns the name of the member that holds the length, or a number describing the fixed length. """

    # Constant sized array
    if member_info.fixed_array_size is not None:
        return member_info.fixed_array_size

    # Otherwise, array members are pointers
    tokens = member_info.type_name.split()
    if tokens[-1] != '*':
        return None

    # Array members and with 'Array' and there is a corresponding element that ends with 'ArrayLength'
    if not member_info.name.endswith('Array'):
        return None

    length_member_name = member_info.name + 'Length'
    if not owning_type_info is None:
        if owning_type_info.find_member(length_member_name):
            return length_member_name

    return None


def is_bitfield_member(type_info, native_type_name, member_name):
    # Currently, bitfield members have names that end with 'Bitfield'
    return member_name.endswith('Bitfield')


def strip_rightmost_array_modifier(native_type_name):

    """ Removes the right-most * from the type if it exists. """

    tokens = native_type_name.split()[::-1] # Reversed token list

    # Rightmost *
    if '*' in tokens:
        index = tokens.index('*')
        has_const = (index < len(tokens) - 1) and tokens[index + 1] == 'const'
        tokens.reverse()
        del tokens[len(tokens) - index - 1]
        # Remove the redundant const now that the * is gone
        if has_const:
            del tokens[len(tokens) - index - 1]
        return ' '.join(tokens)

    # Remove [.*]
    index = native_type_name.find('[')
    if index < 0:
        return native_type_name

    result = re.match('.+\[([0-9].)\].*', native_type_name)
    return native_type_name[:result.start(1)-1] + native_type_name[result.end(1)+1:]


def strip_native_type_modifiers(native_type_name, remove_const=False, remove_volatile=False, remove_star=False, remove_ref=False):

    remove_all = not remove_const and not remove_volatile and not remove_star and not remove_ref

    if remove_all or remove_const: native_type_name = native_type_name.replace('const', '')
    if remove_all or remove_volatile: native_type_name = native_type_name.replace('volatile', '')
    if remove_all or remove_star: native_type_name = native_type_name.replace('*', '')
    if remove_all or remove_ref: native_type_name = native_type_name.replace('&', '')

    native_type_name = native_type_name.replace('  ', ' ')
    native_type_name = native_type_name.strip()

    return native_type_name;


def get_native_method_line(type_mapping, method_type_info, is_declaration, is_virtual=False, is_override=False, access_modifier='public'):

    """ Given a common.source_tools.CppMethod, this returns the native C++ lines used to declare the function. """

    line = ''

    # public/private/internal
    if access_modifier is not None and is_declaration:
        line = line + access_modifier + ': '

    # virtual
    if is_virtual:
        line = line + 'virtual '

    # Output the return value type
    if method_type_info.return_type is None:
        line = line + 'void '
    else:
        line = line + method_type_info.return_type.type_name + ' '

    # Owning class
    if not is_declaration and type_mapping:
        type_name_components = get_native_type_name_components(type_mapping.target_type_name)
        line = line + type_mapping.target_type_name + '::'

    # Method name
    line = line + method_type_info.name + '('

    # Arguments
    args = []
    for arg in method_type_info.arguments:
        token = arg.raw_type_name + ' ' + arg.name
        args.append(token)

    # Close the method
    line = line + ', '.join(args) + ')'

    if is_override:
        line = line + ' override'

    if is_declaration:
        line = line + ';'

    return line


def get_native_param_type_name(arg_type_info):
    """ Determines the native output type for the given method argument. """
    type_name = ''
    if arg_type_info.template_info:
        type_name = arg_type_info.template_info.template_name + '<'
        for index, arg in enumerate(arg_type_info.template_info.template_args):
            native_type_name = strip_native_type_modifiers(arg)
            # Fix up strings
            if native_type_name == '::std::basic_string':
                native_type_name = '::std::string'

            if index > 0:
                type_name = type_name + ', '
            type_name = type_name + native_type_name

        type_name = type_name + '>'
    else:
        type_name = strip_native_type_modifiers(arg_type_info.type_name, remove_ref=True)

    return type_name


def open_namespace(namespace_tokens, indentation='    '):
    lines = []
    indent = ''
    for ns in namespace_tokens:
        lines.append(indent + 'namespace ' + ns)
        lines.append(indent + '{')
        indent = indent + indentation
    return lines


def close_namespace(namespace_tokens, indentation='    '):
    lines = []
    for i in range(0, len(namespace_tokens)):
        indent = indentation * (len(namespace_tokens) - i - 1)
        lines.append(indent + '}')
    return lines


def get_using_namespace_lines(namespace_list):
    # Get uniques
    namespace_list = list( set(namespace_list) )
    return map(lambda x: 'using namespace ' + x + ';', namespace_list)
