import argparse
import os
import os.path
import platform
from pprint import pprint
import pygccxml
import re
import subprocess
import sys

# TODO: Document setup

# Get LLVM and Clang source 3.8.0 from http://llvm.org/releases/download.html
# Extract llvm-3.8.0.src.tar.xz to ```llvm```
# Extract cfe-3.8.0.src.tar.xz to ```llvm/tools``` in a directory called ```clang```
# ```mkdir build```
# ```cd build```
# ```cmake -G "Visual Studio 14 Win64" ../llvm```
# Wait for the build to finishm it takes a while
# Open ```LLVM.sln``` and build it
# In the LLVM build directory run
#   ```cmake --build . --target install```
# Get CastXML from repo - https://github.com/CastXML/CastXML

#
#   cmake -G "Visual Studio 14 Win64" -DCMAKE_BUILD_TYPE=Release
#   Open LLVM.sln and build it
#
#   cmake -G "Visual Studio 14 Win64" -DCMAKE_BUILD_TYPE=Release
#   Open LLVM.sln and build it
# Get CastXML from repo - https://github.com/CastXML/CastXML
#   cmake "-DLLVM_DIR=C:\Drew\Tools\LLVM\output\share\llvm\cmake"
#   Open CastXML.sln and build it


# "C:\Program Files (x86)\Microsoft Visual Studio 14.0\VC\vcvarsall.bat"



# You need to fix the pygccxml source in one place in version 1.7.5 that I tested on Windows.
# https://sourceforge.net/p/pygccxml/bugs/37/
# Update # pygccxml\parser\source_reader.py as follows:
#
# def __produce_full_file(self, file_path):
#     if os.name in ['nt', 'posix']:
#         file_path = file_path.replace(r'\/', os.path.sep)
#     file_path = os.path.abspath(file_path)                 <--------- This line added
#     if os.path.isabs(file_path):
#         return file_path
#     try:
#         abs_file_path = os.path.realpath(
#             os.path.join(
#                 self.__config.working_directory,
#                 file_path))
#         if os.path.exists(abs_file_path):
#             return os.path.normpath(abs_file_path)
#         return file_path
#     except Exception:
#         return file_path


class CppLocation:
    def __init__(self, location=None):
        if location:
            self.header = location.file_name
            self.line = location.line
            self.header = self.header.replace('\\', '/')

        else:
            self.header = None
            self.line = None


class CppTemplateInstantiation:
    def __init__(self, type_name):
        self.type_name = type_name
        if not pygccxml.declarations.templates.is_instantiation(type_name):
            raise Exception('Not a template instantiation: ' + type_name)
        template_name, template_args = pygccxml.declarations.templates.split(type_name)
        self.template_name = template_name
        self.template_args = template_args

    def print_template(self, indent=''):
        print(indent + 'Template instantiation:')
        print(indent + '  template_name: ' + self.template_name)
        print(indent + '  args:')
        for a in self.template_args:
            print(indent + '    ' + a)


class CppDeclaration:
    def __init__(self, type_name, location=None):
        self.type_name = type_name
        self.raw_type_name = type_name # The unmodified type name
        self.location = CppLocation(location)
        if pygccxml.declarations.templates.is_instantiation(type_name):
            self.template_info = CppTemplateInstantiation(type_name)
        else:
            self.template_info = None


class CppVariable(CppDeclaration):

    def __init__(self, name, type_name, location=None):
        CppDeclaration.__init__(self, type_name, location)
        self.name = name
        self.fixed_array_size = None

        # Determine if a fixed length array
        index = type_name.find('[')
        if index > 0:
            result = re.match('.+\[([0-9].)\].*', type_name)
            if result:
                self.fixed_array_size = int(type_name[result.start(1):result.end(1)])
                self.type_name = type_name[:result.start(1)-1] + type_name[result.end(1)+1:] + ' *'

    def print_variable(self, indent=''):
        print(indent + 'Variable:')
        print(indent + '  name: ' + self.name)
        print(indent + '  type_name: ' + self.type_name)
        print(indent + '  fixed_array_size: ' + str(self.fixed_array_size))
        if self.template_info:
            self.template_info.print_template(indent + '  ')


class CppArgument(CppVariable):

    def __init__(self, name, type_name, out_param=False, location=None):
        CppVariable.__init__(self, name, type_name, location)
        self.out_param = out_param # bool

    def print_argument(self, indent=''):
        print('Argument:')
        print(indent + '  name: ' + self.name)
        print(indent + '  type_name: ' + self.type_name)
        print(indent + '  out_param: ' + str(self.out_param))
        if self.template_info:
            self.template_info.print_template(indent + '  ')


class CppClass(CppDeclaration):

    def __init__(self, type_name, class_type, location=None):
        CppDeclaration.__init__(self, type_name, location)
        self.members = [] # CppVariable
        self.methods = [] # CppMethod
        self.base_type_names = [] # The base class/interface type names
        self.class_type = class_type

    def find_member(self, name):
        for m in self.members:
            if m.name == name:
                return m
        return None

    def print_class(self, indent=''):
        print(indent + 'Class:')
        print(indent + '  type_name: ' + self.type_name)
        print(indent + '  members:')
        print(indent + '  bases:')
        for b in self.base_type_names:
            print(indent + '    :' + b)
        print(indent + '  methods:')
        for m in self.methods:
            m.print_method(indent + '  ')
        if self.template_info:
            self.template_info.print_template(indent + '  ')
        for m in self.members:
            m.print_variable(indent + '  ')
        if self.template_info:
            self.template_info.print_template(indent + '  ')


class CppEnum(CppDeclaration):

    def __init__(self, type_name, location=None):
        CppDeclaration.__init__(self, type_name, location)
        self.values = [] # CppEnumValue

    def print_enum(self, indent=''):
        print('Enum:')
        print(indent + '  type_name: ' + self.type_name)
        print(indent + '  members:')
        for v in self.values:
            v.print_value(indent + '  ')
        if self.template_info:
            self.template_info.print_template(indent + '  ')


class CppTypedef(CppDeclaration):

    def __init__(self, type_name, aliased_type, location=None):
        CppDeclaration.__init__(self, type_name, location)
        self.aliased_type = aliased_type

    def print_typedef(self, indent=''):
        print('Typedef:')
        print(indent + '  type_name: ' + self.type_name)
        print(indent + '  aliased_type: ' + self.aliased_type)
        if self.template_info:
            self.template_info.print_template(indent + '  ')


class CppMethod(CppDeclaration):

    def __init__(self, type_name, name, location=None):
        CppDeclaration.__init__(self, type_name, location)
        self.name = name
        self.arguments = [] # CppArgument
        self.return_type = None # CppVariable

        # HACK: We don't mark methods as template instantiations, the helper function falsely marks functions
        # that have templated arguments as itself being templated
        self.template_info = None


    def print_method(self, indent=''):
        print('Method:')
        print(indent + '  name: ' + self.name)
        print(indent + '  return_type: ')
        if not self.return_type is None:
            self.return_type.print_variable(indent + '  ')
        else:
            print(indent + '  <none>')
        print(indent + '  arguments:')
        for m in self.arguments:
            m.print_variable(indent + '  ')


class CppEnumValue:

    def __init__(self, name, value):
        self.name = name
        self.value = value

    def print_value(self, indent=''):
        print(indent + self.name + ' = ' + str(self.value))


def extract_class(decl, extract_methods, extract_variables, extract_base_classes):

    print('Extracting class: ' + decl.decl_string + '...')

    result = CppClass(decl.decl_string, decl.class_type, decl.location)

    if extract_variables:
        for value in decl.variables(allow_empty=True):
            member = CppVariable(value.name, value.decl_type.decl_string)
            result.members.append(member)

    if extract_methods:
        for method in decl.public_members:
            # Ignore ctor and dtor
            if isinstance(method, pygccxml.declarations.calldef_members.constructor_t):
                continue
            elif isinstance(method, pygccxml.declarations.calldef_members.destructor_t):
                continue
            elif isinstance(method, pygccxml.declarations.calldef_members.member_function_t):
                member = extract_method(method)
                result.methods.append(member)

    # Extract information about base classes
    if extract_base_classes:
        for base in decl.bases: # pygccxml.declarations.class_declaration.hierarchy_info_t[]
            base_class = base.related_class # pygccxml.declarations.class_declaration.class_t
            result.base_type_names.append(base_class.decl_string)
            #print('Base class: ' + base_class.decl_string)

    print('Done extracting: ' + decl.decl_string)

    return result


def extract_enum(decl):

    print('Extracting enum: ' + decl.decl_string + '...')

    result = CppEnum(decl.decl_string, decl.location)

    for value in decl.values:
        result.values.append( CppEnumValue(value[0], value[1]) )
        #print(value)

    print('Done extracting: ' + decl.decl_string)

    return result


def extract_method(decl):

    print('Extracting method: ' + decl.name + '...')

    result = CppMethod(decl.decl_string, decl.name, decl.location)

    # Return type
    if decl.return_type != None and decl.return_type.decl_string != 'void':
        result.return_type = CppArgument('', decl.return_type.decl_string, False)
    else:
        result.return_type = None

    #print('  return type: ' + decl.return_type.decl_string)

    # Arguments
    for a in decl.arguments:

        out_param = False

        # Determine direction
        # NOTE: This is a bit of a hack:  If & but not const then it's an out value
        tokens = a.decl_type.decl_string.split(' ')
        if len(tokens) > 1:
            if tokens[-2] != 'const' and tokens[-1] == '&':
                out_param = True

        arg = CppArgument(a.name, a.decl_type.decl_string, out_param)
        result.arguments.append(arg)

        direction = 'in'
        if arg.out_param:
            direction = 'out'

        #print('  arg: ' + arg.name + ' ' + arg.type_name + ' [' + direction + ']')

    print('Done extracting: ' + decl.name)

    return result


def extract_typedef(decl):

    result = None

    if decl.decl_string != decl.decl_type.decl_string:
        #print('Extracting typedef: ' + decl.decl_string + '...')

        result = CppTypedef(decl.decl_string, decl.decl_type.decl_string, decl.location)
        #print(result.type_name + ' --> ' + result.aliased_type)

        #print('Done extracting: ' + decl.decl_string)

    return result


def find_namespace(global_namespace, namespace_name):

    tokens = namespace_name.split('::')

    ns = global_namespace
    for t in tokens:
        if t != '':
            ns = ns.namespace(t)

    return ns


def get_short_type_name(type_name):

    tokens = type_name.split('::')
    short_name = tokens[-1]
    return tokens[-1]


def find_type(global_namespace, type_name, expected_pygccxml_type):

    short_name = get_short_type_name(type_name)
    namespace_name = type_name[0:len(type_name) - len(short_name)]

    criteria = pygccxml.declarations.declaration_matcher(name=type_name)
    results = pygccxml.declarations.matcher.find(criteria, global_namespace)

    # for x in results:
    #     print(vars(x))

    decl = None
    for x in results:
        if isinstance(x, expected_pygccxml_type):
            decl = x
            break

    if decl is None:
        raise Exception("Type not found: " + type_name)

    return decl


def is_executable_in_path(exe_name):
    def is_exe(fpath):
        return os.path.isfile(fpath) and os.access(fpath, os.X_OK)

    fpath, fname = os.path.split(exe_name)
    if fpath:
        if is_exe(exe_name):
            return exe_name
    else:
        for path in os.environ["PATH"].split(os.pathsep):
            path = path.strip('"')
            exe_file = os.path.join(path, exe_name)
            if is_exe(exe_file):
                return exe_file

    return None


def ensure_cl_in_path(exe_name):
    if platform.system() == 'Windows' and not exe_name.endswith('.exe'):
        exe_name = exe_name + '.exe'

    if not is_executable_in_path(exe_name):
        if platform.system() == 'Windows':
            vcvars_paths = [
                "C:/Program Files (x86)/Microsoft Visual Studio 14.0/VC/vcvarsall.bat",
                "C:/Program Files (x86)/Microsoft Visual Studio 12.0/VC/vcvarsall.bat",
                "C:/Program Files (x86)/Microsoft Visual Studio 11.0/VC/vcvarsall.bat"
                "C:/Program Files (x86)/Microsoft Visual Studio/2017/Professional/VC/Auxiliary/Build/vcvarsall.bat",
                "C:/Program Files (x86)/Microsoft Visual Studio/2017/Enterprise/VC/Auxiliary/Build/vcvarsall.bat",
            ]

            sys.path.append('C:/Program Files (x86)/Microsoft Visual Studio 14.0/VC/bin')

            #for path in vcvars_paths:
            #    if os.path.exists(path):
            #        raise Exception("You need to make sure " + exe_name + " is in the path.  You have Visual Studio installed so you just need to run '" + path + "'")
            #        return
            #raise Exception("You need to make sure " + exe_name + " is in the path.  It doesn't seem like you have Visual Studio installed.")

        else:
            # TODO: Determine if you need
            pass


class CastXmlConfig:
    def __init__(self):
        self.existing_generated_xml = None, # A previously generated XML file with all the needed type information
        self.source_files = [] # The .h and .cpp files to scan looking for type definitions
        self.include_paths = [] # The include paths to add to the scan so that compilation succeeds for the source files
        self.namespace_names = [] # The namespaces to scan for types
        self.enumeration_names = []
        self.class_names = []
        self.constant_names = []

def run_castxml(config):

    """
    Runs CastXML with the given CastXmlConfig arguments
    """

    class Result:
        def __init__(self):
            self.enumerations = [] # CppEnum
            self.classes = [] # CppClass
            self.typedefs = [] # CppTypedef
            self.all = {} # The mapping of type name to the Cpp* instance

        def print_all(self):
            for x in self.typedefs:
                x.print_typedef()
            for x in self.enumerations:
                x.print_enum()
            for x in self.classes:
                x.print_class()

    result = Result()

    # Make sure cl is in the path
    ensure_cl_in_path('cl');

    # Find the CastXML install location
    generator_path, generator_name = pygccxml.utils.find_xml_generator()

    print('CastXML path: ' + generator_path)

    # Configure the xml generator
    xml_generator_config = pygccxml.parser.xml_generator_configuration_t(
        xml_generator_path = generator_path,
        xml_generator = generator_name,
        include_paths = config.include_paths,
        compiler_path = 'C:/Program Files (x86)/Microsoft Visual Studio 14.0/VC/bin/cl.exe'
        )

    # Parse the source

    decls = pygccxml.parser.parse(files=config.source_files, config=xml_generator_config, compilation_mode=pygccxml.parser.COMPILATION_MODE.ALL_AT_ONCE)

    global_namespace = pygccxml.declarations.get_global_namespace(decls)

    # Track all the typedefs we might be using
    if config.namespace_names:
        for namespace_name in config.namespace_names:
            ns = find_namespace(global_namespace, namespace_name)
            for decl in ns.declarations:
                if isinstance(decl, pygccxml.declarations.typedef.typedef_t):
                    # Only pull typedefs out of headers we care about
                    #header_path = decl.location.file_name.replace('\\', '/')
                    #if header_path in source_files:
                    value = extract_typedef(decl)
                    if value != None:
                        result.typedefs.append(value)
                        result.all[value.type_name] = value

    # Extract the types we're looking for
    if config.enumeration_names:
        for type_name in config.enumeration_names:
            decl = find_type(global_namespace, type_name, pygccxml.declarations.enumeration.enumeration_t)
            value = extract_enum(decl)
            if value != None:
                result.enumerations.append(value)
                result.all[type_name] = value

    if config.class_names:
        for type_name in config.class_names:
            decl = find_type(global_namespace, type_name, pygccxml.declarations.class_declaration.class_t)
            value = extract_class(decl, extract_methods=True, extract_variables=True, extract_base_classes=True)
            if value != None:
                result.classes.append(value)
                result.all[type_name] = value

    # Extract base classes that weren't specified but are implicitly needed
    # This is for situations where one module depends on another module that is generated elsewhere
    check = []
    check.extend(result.classes)

    for x in check:
        for base_class_type_name in x.base_type_names:
            if base_class_type_name not in result.all:
                decl = find_type(global_namespace, base_class_type_name, pygccxml.declarations.class_declaration.class_t)
                value = extract_class(decl, extract_methods=True, extract_variables=False, extract_base_classes=True)
                if value != None:
                    result.all[base_class_type_name] = value

    return result


# Run the test scenario if the main script
if __name__ == "__main__":

    params = CastXmlConfig()
    params.source_files = [
        'X:/modules/core/core_common/include/twitchsdk/core/types/coretypes.h',
        'X:/modules/core/core_common/include/twitchsdk/core/types/errortypes.h',
        'X:/modules/core/core_common/include/twitchsdk/core/types/tracingtypes.h',
        'X:/modules/core/core_common/include/twitchsdk/core/channel/ichannellistener.h',
        'X:/modules/core/core_common/include/twitchsdk/core/corelistener.h',
        'X:/modules/core/core_common/include/twitchsdk/core/coreapi.h'
    ]
    params.include_paths = [
        'X:/modules/core/core_common/include'
    ]
    params.namespace_names = [
        '::ttv'
    ]
    params.enumeration_names = [
        '::ttv::VodType',
        '::ttv::VodStatus',
        '::ttv::PubSubState'
    ]
    params.constant_names = [
    ]
    params.class_names = [
        '::ttv::ChannelInfo',
        '::ttv::ProfileImage',
        '::ttv::StreamInfo',
        '::ttv::UserInfo',
        '::ttv::WatchPartyUpdate',
        '::ttv::CoreAPI',
        '::ttv::IChannelListener',
        '::ttv::ICoreAPIListener'
    ]

    types = run_castxml(params)
