import abc
import imp
import os
from pprint import pprint

import common.configuration


class TypeGeneratorBase(object):

    """
    The language independent base class for all type generators.
    """

    def __init__(self, binding_generator, type_mapping, type_info):
        self.config = binding_generator.config # common.configuration.ConfigSettings
        self.binding_generator = binding_generator # language_binding_generator.LanguageBindingGenerator
        self.type_mapping = type_mapping
        self.type_info = type_info
        self.all_methods = [] # All public methods, including those in base classes
        self.virtual_method_names = [] # The names of methods that need to be marked as virtual
        self.export_method_names = None
        self.member_type_names = {} # Map of member name to the target type name
        self.override_members = [] # List of all member type_infos that need to be implemented because they are declared in an abstract base class
        self.array_member_lengths = {} # Mapping of array member name to the length member name

        class_config = self.config.find_class_configuration(self.type_info.type_name)
        export_methods = None
        if class_config:
            self.export_method_names = class_config.export_methods


    @abc.abstractmethod
    def get_template_names(self):
        """Returns the list of template file names to be used to generate the target source."""
        return []

    @abc.abstractmethod
    def infer_output_file_path_for_type(self, type_mapping, template_name):
        pass


    def determine_virtual_methods(self):
        """Determine which methods need to be virtual when implemented."""
        for method in self.all_methods:
            for base_mapping in self.type_mapping.target_base_class_mappings:
                base_type_info = self.binding_generator.extracted_type_info.all[base_mapping.native_type_name]
                if len(filter(lambda x: x.name == method.name, base_type_info.methods)) > 0:
                    self.virtual_method_names.append(method.name)


    def fixup_base_classes(self):

        # Update base classes and fixup public methods
        def process(type_info):
            for native_base_class_type_name in type_info.base_type_names:
                base_type_mapping = self.config.find_type_mapping(native_base_class_type_name)
                if base_type_mapping:
                    self.type_mapping.target_base_class_mappings.append(base_type_mapping)

                else:
                    if native_base_class_type_name in self.binding_generator.extracted_type_info.all:
                        base_type_info = self.binding_generator.extracted_type_info.all[native_base_class_type_name]
                        for method_info in base_type_info.methods:
                            self.all_methods.append(method_info)
                            #print(method_info.name)

                        process(base_type_info)

        process(self.type_info)
        self.all_methods.extend(self.type_info.methods)

        # Track members that need to be overidden
        for base_type_mapping in self.type_mapping.target_base_class_mappings:
            if base_type_mapping.is_abstract:
                base_type_info = self.binding_generator.extracted_type_info.all[base_type_mapping.native_type_name]
                for member_info in base_type_info.members:
                    if len(filter(lambda x: x.name == member_info.name, self.override_members)) == 0:
                        self.override_members.append(member_info)


    def determine_member_types(self):

        """Determines and caches the target binding types for the members."""

        for member_type_info in self.type_info.members:

            # Skip array length members since we use array objects
            if member_type_info.name.endswith('ArrayLength'):
                continue

            # Track array members
            array_length = self.binding_generator.is_array_member(member_type_info, self.type_info)
            if array_length is not None:
                self.array_member_lengths[member_type_info.name] = array_length

        for member_type_info in self.type_info.members:

            # Skip array length members since we use array objects
            if member_type_info.name.endswith('ArrayLength'):
                continue

            self.member_type_names[member_type_info.name] = self.determine_target_type_name_from_variable(member_type_info, self.type_info)

        for member_type_info in self.override_members:
            # Skip array length members since we use array objects
            if member_type_info.name.endswith('ArrayLength'):
                continue

            self.member_type_names[member_type_info.name] = self.determine_target_type_name_from_variable(member_type_info, self.type_info)


    def infer_output_file_path(self, template_name):
        return self.infer_output_file_path_for_type(self.type_mapping, template_name)


    def get_target_type_name_components(self, target_type_name):
        # CX shares the same type name format as native
        return common.native_tools.get_native_type_name_components(target_type_name)


    def fill_template_common(self, lines):

        output_type_name = self.type_mapping.target_type_name
        type_name_components = self.get_target_type_name_components(output_type_name)

        lines = common.file_utils.replace_all(lines, "<<SHORT_TYPE_NAME>>", type_name_components['name'])
        lines = common.file_utils.replace_all(lines, "<<FULL_TYPE_NAME>>", output_type_name)
        lines = common.file_utils.replace_all(lines, "<<NATIVE_TYPE_NAME>>", self.type_mapping.native_type_name)

        return lines


    def fill_template_native_base_classes(self, lines):

        base_classes = ''

        if len(self.type_mapping.target_base_class_mappings) > 0:
            base_classes = base_classes + ' : '
            for index, base in enumerate(self.type_mapping.target_base_class_mappings):
                if index > 0:
                    base_classes = base_classes + ', '
                base_classes = base_classes + 'public ' + base.native_type_name

        return common.file_utils.replace_all(lines, "<<BASE_CLASSES>>", base_classes)


    def fill_template_namespace(self, lines):

        """Creates namespace scopes for the type and indents lines in between."""

        markers = common.file_utils.find_line_range_markers('NAMESPACE', lines, must_exist=False)
        if markers:
            type_name_components = self.get_target_type_name_components(self.type_mapping.target_type_name)
            tokens = type_name_components['namespace_tokens']
            if len(tokens) > 0:
                index = markers[0]
                for i in range(0, len(tokens)):
                    lines.insert((2*i) + index, (self.config.indent * i) + 'namespace ' + tokens[i])
                    lines.insert((2*i) + index + 1, (self.config.indent * i) + '{')

                markers = common.file_utils.find_line_range_markers('NAMESPACE', lines)

                # Indent lines in the namespace
                for i in range(markers[0], markers[1]):
                    lines[i] = (self.config.indent * len(tokens)) + lines[i]

                del lines[ markers[1] ]
                del lines[ markers[0] ]

                index = markers[1]
                for i in range(0, len(tokens)):
                    lines.insert(index + i, (self.config.indent * (len(tokens) - i - 1) + '}'))

        return lines


    def is_primitive_type(self, target_type_name):
        primitive_types = [
            'void',
            'bool',
            'char',
            'char16',
            'int16',
            'uint16',
            'int',
            'uint',
            'int32',
            'int64',
            'uint32',
            'uint64',
            'float32',
            'float64'
        ]
        return target_type_name in primitive_types


    def fixup_header_include_path(self, header_path):
        if header_path.startswith('include/'):
            header_path = header_path[len('include/'):]
        return header_path


    def gather_native_types(self, type_info, result, add_bases=True, add_enums=True, add_classes=True):

        """
        Recursively walks the given type and all the types it references to gather the full list of types that are referenced.
        native_types: The final list of types

        result = {
            native_types = []
        }
        """

        visited = {}

        def process_referenced_type(type_name):
            type_name = self.binding_generator.strip_native_type_modifiers(type_name)
            if type_name in self.binding_generator.extracted_type_info.all:
                arg_type_info = self.binding_generator.extracted_type_info.all[type_name]
                recurse(type_info=arg_type_info, result=result, is_base=False)

        def recurse(type_info, result, is_base):

            #pprint(type_info.type_name + ' ' + str(type_info) + ' ' + str(type_info.template_info))

            if type_info.type_name in visited:
                return

            # A variable is not a type
            if not isinstance(type_info, common.source_tools.CppVariable):
                visited[type_info.type_name] = True

            # A collection
            if type_info.template_info:
                for arg in type_info.template_info.template_args:
                    native_type_name = self.binding_generator.strip_native_type_modifiers(arg)
                    if not native_type_name in result['native_types']:
                        result['native_types'].append(native_type_name)

                    # See if the template argument type has a mapping an recurse
                    process_referenced_type(native_type_name)

            # Check the current type and possibly recurse
            else:
                add_this_type = True
                if isinstance(type_info, common.source_tools.CppMethod):
                    add_this_type = False
                elif isinstance(type_info, common.source_tools.CppClass):
                    if is_base:
                        add_this_type = add_bases
                    else:
                        add_this_type = add_classes
                elif isinstance(type_info, common.source_tools.CppVariable):
                    add_this_type = add_classes
                elif isinstance(type_info, common.source_tools.CppEnum):
                    add_this_type = add_enums

                if add_this_type:
                    native_type_name = self.binding_generator.strip_native_type_modifiers(type_info.type_name)
                    #print('native_type_name: ' + native_type_name + ' ' + str(type_info))
                    if not native_type_name in result['native_types']:
                        result['native_types'].append(native_type_name)

                # Recurse if necessary

                # Method
                if isinstance(type_info, common.source_tools.CppMethod):
                    for arg_type_info in type_info.arguments:
                        recurse(type_info=arg_type_info, result=result, is_base=False)

                    # Return type
                    if type_info.return_type:
                        recurse(type_info=type_info.return_type, result=result, is_base=False)

                # Class, struct or interface
                elif isinstance(type_info, common.source_tools.CppClass):
                    for member_type_info in type_info.members:
                        recurse(type_info=member_type_info, result=result, is_base=False)
                    for method_type_info in type_info.methods:
                        if self.export_method_names is None or method_type_info.name in self.export_method_names:
                            recurse(type_info=method_type_info, result=result, is_base=False)

                    # Check base classes
                    type_mapping = self.config.find_type_mapping(type_info.type_name)
                    if type_mapping:
                        for base_mapping in type_mapping.target_base_class_mappings:
                            base_native_type_info = self.binding_generator.extracted_type_info.all[base_mapping.native_type_name]
                            recurse(type_info=base_native_type_info, result=result, is_base=True)

                # Variable
                elif isinstance(type_info, common.source_tools.CppVariable):
                    # Lookup the referenced type
                    process_referenced_type(type_info.type_name)

        recurse(type_info=type_info, result=result, is_base=False)


    def get_native_class_type(self, type_mapping):

        """Given a type mapping, determine the native C++ type is represents."""

        if isinstance(type_mapping, common.configuration.ClassConfigTypeMapping):
            class_type = 'class'
        elif isinstance(type_mapping, common.configuration.StructConfigTypeMapping):
            class_type = 'struct'
        elif isinstance(type_mapping, common.configuration.InterfaceConfigTypeMapping):
            class_type = 'class'
        elif isinstance(type_mapping, common.configuration.EnumerationConfigTypeMapping):
            class_type = 'enum'
        elif isinstance(type_mapping, common.configuration.InterfaceAdapterConfigTypeMapping):
            class_type = 'class'
        else:
            raise Exception('Unhandled mapping type')

        return class_type


    def native_types_to_type_mappings(self, native_type_names):

        """Finds all the type mappings for the given list of native type names."""

        result = []

        for native_type_name in native_type_names:

            # Look for a type mapping
            native_type_name = self.binding_generator.strip_native_type_modifiers(native_type_name)
            type_mapping = self.config.find_type_mapping(native_type_name)
            if not type_mapping:
                continue

            # Make sure we don't already have the mapping
            if len( filter(lambda x: x.target_type_name == type_mapping.target_type_name, result) ) == 0:
                result.append(type_mapping)

        return result
