import abc
import imp
import os
import re
from pprint import pprint

import common.file_utils
import common.configuration
import common.source_tools
import common.struct_utils
import cx.cx_type_generator_base


class CxStructTypeGenerator(cx.cx_type_generator_base.CxTypeGeneratorBase):

    def __init__(self, binding_generator, type_mapping, type_info):
        cx.cx_type_generator_base.CxTypeGeneratorBase.__init__(self, binding_generator, type_mapping, type_info)

        self.fixup_base_classes()
        self.determine_virtual_methods()
        self.determine_member_types()


    def get_template_names(self):
        return [ 'struct.cx.h.template', 'struct.cx.cpp.template' ]


    def fill_template(self, lines):

        lines = self.fill_template_common(lines)

        # Class type and sealed
        lines = self.fill_template_binding_class_type_sealed(lines)

        # Base class inheritance
        lines = self.fill_template_binding_base_classes(lines)

        # Create namespace scopes if needed
        lines = self.fill_template_namespace(lines)

        return lines


    def update(self, template_name, lines):

        # Header file
        if template_name == 'struct.cx.h.template':
            lines = self.update_header_file(lines)

        # Source file
        elif template_name == 'struct.cx.cpp.template':
            lines = self.update_source_file(lines)

        return lines


    def update_header_file(self, lines):

        # Binding includes
        includes_result = self.gather_header_binding_includes()
        includes = includes_result['includes']
        exclude_mappings = includes_result['included_mappings']

        # Native includes
        header_path = self.type_mapping.context.get_include_path_relative_header(self.type_info.location.header)
        includes.append('#include "' + header_path + '"')

        includes.sort()
        lines = common.file_utils.replace_lines_range('INCLUDES', lines, includes, indent=self.config.indent, must_exist=False)

        # Forward declarations
        forward_delaration_lines = self.get_forward_declaration_lines(exclude_type_mappings=exclude_mappings)
        lines = common.file_utils.replace_lines_range('FORWARD_DECLARATIONS', lines, forward_delaration_lines, indent=self.config.indent, must_exist=True)

        # Fill in fields
        replacement_lines = self.get_header_field_lines()
        lines = common.file_utils.replace_lines_range('FIELDS', lines, replacement_lines, indent=self.config.indent)

        # Conversion functions
        conversion_lines = self.get_header_native_conversion_lines(indentation=self.config.indent);
        lines = common.file_utils.replace_lines_range('CONVERSIONS', lines, conversion_lines, indent=self.config.indent)

        return lines


    def get_source_includes(self):

        # Special includes
        explicit_target_type_mappings = []
        if self.type_mapping.is_abstract:
            # We assume we have a UnionAsBaseClassTransformation for now
            trans = self.type_mapping.transformations[0]
            if not isinstance(trans, common.configuration.UnionAsBaseClassTransformation):
                raise Exception('Unhandled transformation')
            explicit_target_type_mappings = trans.derived_class_type_mappings

        # Binding includes
        includes = self.gather_source_binding_includes(explicit_target_type_mappings=explicit_target_type_mappings)['includes']
        includes.append('#include "twitchsdk/core/cx_coreutil.h"')
        includes.sort()

        # Precompiled header
        if self.config.precompiled_header_path:
            includes.insert(0, '#include "' + self.config.precompiled_header_path + '"')

        return includes


    def update_source_file(self, lines):

        includes = self.get_source_includes()
        lines = common.file_utils.replace_lines_range('INCLUDES', lines, includes, indent=self.config.indent, must_exist=False)

        # Using namespace
        using_lines = self.get_source_using_namespace_lines()
        lines = common.file_utils.replace_lines_range('USING_NAMESPACE', lines, using_lines, indent=self.config.indent, must_exist=False)

        # Marshalling
        impl_lines = self.get_source_to_native_conversion_lines()
        lines = common.file_utils.replace_lines_range('CONVERSIONS', lines, impl_lines, indent=self.config.indent, must_exist=False)

        impl_lines = self.get_source_to_binding_conversion_lines()
        lines = common.file_utils.replace_lines_range('CONVERSIONS', lines, impl_lines, indent=self.config.indent, must_exist=False)

        return lines


    def get_source_using_namespace_lines(self):
        return common.native_tools.get_using_namespace_lines([
            '::ttv::binding::cx',
            self.type_mapping.context.default_native_namespace
        ])


    def get_header_native_conversion_lines(self, indentation):
        """Returns the native forward declaration of the functions which will marshall to and from the target language type."""

        # Open namespace
        namespace_tokens = self.type_mapping.context.default_native_namespace.strip(':').split('::')
        indent = ''

        lines = common.native_tools.open_namespace(namespace_tokens, indentation)
        indent = indentation * len(namespace_tokens)

        # Declare the functions
        #lines.append(indent + '//void ToNative(<<FULL_TYPE_NAME>>^ val, <<NATIVE_TYPE_NAME>>* result);')
        lines.append(indent + 'void ToBinding(const ' + self.type_mapping.native_type_name + '* val, ' + self.type_mapping.target_type_name + '^* result);')
        lines.append(indent + 'inline void ToBinding(const ' + self.type_mapping.native_type_name + '** val, ' + self.type_mapping.target_type_name + '^* result) { ToBinding(*val, result); }')

        # Close namespace
        lines.extend( common.native_tools.close_namespace(namespace_tokens, indentation) )

        return lines


    def get_source_to_native_conversion_lines(self):
        # TODO: Implement this if needed
        lines = []
        # lines.append('/*')
        # lines.append('void ' + self.type_mapping.context.default_native_namespace + '::ToNative(<<FULL_TYPE_NAME>>^ val, <<NATIVE_TYPE_NAME>>* result)')
        # lines.append('{')

        # # Marshal each field
        # for member_type_info in self.type_info.members:
        #     if member_type_info.name in self.member_type_names:
        #         binding_arg_name = '_binding_' + member_type_info.name
        #         lines.append( self.config.indent + self.member_type_names[member_type_info.name] + ' ' + binding_arg_name + ' = val->' + member_type_info.name + ';' )
        #         lines.append( self.config.indent + 'ToNative(&' + binding_arg_name + ', &result->' + member_type_info.name + ');' )

        # lines.append('}')
        # lines.append('*/')
        # lines.append('')

        return lines


    def get_source_to_binding_conversion_lines(self):
        lines = []
        lines.append('void ' + self.type_mapping.context.default_native_namespace + '::ToBinding(const <<NATIVE_TYPE_NAME>>* val, <<FULL_TYPE_NAME>>^* result)')
        lines.append('{')
        lines.append(self.config.indent + 'if (val != nullptr)')
        lines.append(self.config.indent + '{')

        if self.type_mapping.is_abstract:
            body_lines = self.get_source_to_binding_union_conversion_lines()
        else:
            body_lines = self.get_source_to_binding_concrete_conversion_lines()

        body_lines = map(lambda x: self.config.indent + self.config.indent + x, body_lines)
        lines.extend(body_lines)

        lines.append(self.config.indent + '}')
        lines.append(self.config.indent + 'else')
        lines.append(self.config.indent + '{')
        lines.append(self.config.indent + self.config.indent + '*result = nullptr;')
        lines.append(self.config.indent + '}')
        lines.append('}')

        lines = self.fill_template_common(lines)

        return lines


    def get_source_to_binding_conversion_for_member(self, member_type_info):
        lines = []
        array_length = ''
        if member_type_info.name in self.array_member_lengths:
            length = self.array_member_lengths[member_type_info.name]

            if isinstance(length, basestring):
                array_length = 'val->' + str(length) + ', '
            elif isinstance(length, (int, long)):
                array_length = 'static_cast<size_t>(' + str(length) + '), '

        native_arg_name = '_native_' + member_type_info.name
        resolved_type_name = self.binding_generator.resolve_typedefs(member_type_info.type_name);

        if resolved_type_name.find('const') < 0:
            resolved_type_name = 'const ' + resolved_type_name

        lines.append( resolved_type_name + ' ' + native_arg_name + ' = static_cast<' + resolved_type_name + '>(val->' + member_type_info.name + ');' )

        binding_arg_name = '_binding_' + member_type_info.name
        lines.append( self.member_type_names[member_type_info.name] + ' ' + binding_arg_name + ';' )
        lines.append( 'ToBinding(&' + native_arg_name + ', ' + array_length + '&' + binding_arg_name + ');' )
        lines.append( '(*result)->' + member_type_info.name + ' = ' + binding_arg_name + ';' )

        return lines


    def get_source_to_binding_union_conversion_lines(self):
        """Unions which have been converted to base classes of other structs."""
        lines = []

        # NOTE: For now we assume that there's a UnionAsBaseClassTransformation instance
        trans = self.type_mapping.transformations[0]
        if not isinstance(trans, common.configuration.UnionAsBaseClassTransformation):
            raise Exception('Unhandled transformation')

        enum_type_info = trans.enum_type_info
        union_type_info = trans.union_type_info

        # NOTE: We assume the container has a field called 'type' which is an enum describing the type to select
        # NOTE: We assume that the enum is in the same order as the union members were listed
        lines.append('switch (val->type)')
        lines.append('{')

        for index, value in enumerate(enum_type_info.values):
            member_type_info = union_type_info.members[index]
            derived_class_mapping = trans.derived_class_type_mappings[index]
            lines.append(self.config.indent + 'case ' + value.name + ':')
            lines.append(self.config.indent + '{')
            lines.append(self.config.indent + self.config.indent + 'const auto& n = val->' + trans.union_member_name + '.' + member_type_info.name + ';')
            lines.append(self.config.indent + self.config.indent + derived_class_mapping.target_type_name + '^ b = ref new ' + derived_class_mapping.target_type_name + '();')
            lines.append(self.config.indent + self.config.indent + 'ToBinding(&n, &b);')
            lines.append(self.config.indent + self.config.indent + '*result = b;')
            lines.append(self.config.indent + self.config.indent + 'break;')
            lines.append(self.config.indent + '}')

        lines.append('}')

        # Write out extra members
        for member_type_info in self.type_info.members:
            if member_type_info.name in self.member_type_names:
                lines.extend( self.get_source_to_binding_conversion_for_member(member_type_info) )

        return lines


    def get_source_to_binding_concrete_conversion_lines(self):
        """Normal structures."""
        lines = []

        lines.append('*result = ref new <<FULL_TYPE_NAME>>();')

        # Marshal each field
        for member_type_info in self.type_info.members:
            if member_type_info.name in self.member_type_names:
                lines.extend( self.get_source_to_binding_conversion_for_member(member_type_info) )

        return lines
