import abc
import imp
import os
from pprint import pprint

import common.file_utils
import common.configuration
import common.source_tools
import common.native_tools
import common.type_generator_base


class CxTypeGeneratorBase(common.type_generator_base.TypeGeneratorBase):

    def __init__(self, binding_generator, type_mapping, type_info):
        common.type_generator_base.TypeGeneratorBase.__init__(self, binding_generator, type_mapping, type_info)


    def infer_output_file_path_for_type(self, type_mapping, template_name):

        if template_name.endswith('.h.template'):
            dir_name = 'include'
            extension = '.h'
        else:
            dir_name = 'source'
            extension = '.cpp'

        index = type_mapping.target_type_name.rfind('::')
        if index >= 0:
            namespace = type_mapping.target_type_name[0:index]
            name = type_mapping.target_type_name[index+2:]
        else:
            namespace = ''
            name = type_mapping.target_type_name

        namespace = namespace.strip('::')
        namespace = namespace.lower().replace('::', '/')

        return dir_name + '/' + namespace + '/' + name + extension;


    def get_target_type_name_components(self, target_type_name):
        # CX shares the same type name format as native
        return common.native_tools.get_native_type_name_components(target_type_name)


    def fill_template_common(self, lines):

        output_type_name = self.type_mapping.target_type_name
        type_name_components = self.get_target_type_name_components(output_type_name)

        lines = common.file_utils.replace_all(lines, "<<SHORT_TYPE_NAME>>", type_name_components['name'])
        lines = common.file_utils.replace_all(lines, "<<FULL_TYPE_NAME>>", output_type_name)
        lines = common.file_utils.replace_all(lines, "<<NATIVE_TYPE_NAME>>", self.type_mapping.native_type_name)

        return lines


    def fill_template_binding_class_type_sealed(self, lines):

        if not self.type_mapping.is_abstract:
            lines = common.file_utils.replace_all(lines, "<<SEALED>>", ' sealed')
        else:
            lines = common.file_utils.replace_all(lines, "<<SEALED>>", '')

        class_type = self.get_class_type(self.type_mapping, is_native=False)
        return common.file_utils.replace_all(lines, "<<CLASS_TYPE>>", class_type)


    def fill_template_binding_base_classes(self, lines):

        base_classes = ''

        if len(self.type_mapping.target_base_class_mappings) > 0:
            base_classes = base_classes + ': '
            for index, base in enumerate(self.type_mapping.target_base_class_mappings):
                if index > 0:
                    base_classes = base_classes + ', '
                base_classes = base_classes + 'public ' + base.target_type_name

        return common.file_utils.replace_all(lines, "<<BASE_CLASSES>>", base_classes)


    def fill_template_namespace(self, lines):

        """Creates namespace scopes for the type and indents lines in between."""

        markers = common.file_utils.find_line_range_markers('NAMESPACE', lines, must_exist=False)
        if markers:
            type_name_components = self.get_target_type_name_components(self.type_mapping.target_type_name)
            tokens = type_name_components['namespace_tokens']
            if len(tokens) > 0:
                index = markers[0]
                for i in range(0, len(tokens)):
                    lines.insert((2*i) + index, (self.config.indent * i) + 'namespace ' + tokens[i])
                    lines.insert((2*i) + index + 1, (self.config.indent * i) + '{')

                markers = common.file_utils.find_line_range_markers('NAMESPACE', lines)

                # Indent lines in the namespace
                for i in range(markers[0], markers[1]):
                    lines[i] = (self.config.indent * len(tokens)) + lines[i]

                del lines[ markers[1] ]
                del lines[ markers[0] ]

                index = markers[1]
                for i in range(0, len(tokens)):
                    lines.insert(index + i, (self.config.indent * (len(tokens) - i - 1) + '}'))

        return lines


    def is_primitive_type(self, target_type_name):
        primitive_types = [
            'void',
            'bool',
            'char',
            'char16',
            'int16',
            'uint16',
            'int',
            'uint',
            'int32',
            'uint32',
            'uint64',
            'float32',
            'float64'
        ]
        return target_type_name in primitive_types


    def get_binding_dependency_headers(self, type_mappings):

        """Finds all the binding header files that are needed to include to satisfy the dependencies of the given native type names."""

        result = []

        for type_mapping in type_mappings:

            header_path = self.infer_output_file_path_for_type(type_mapping, '*.cx.h.template')
            header_path = self.fixup_header_include_path(header_path)

            if not header_path in result:
                result.append(header_path)

        return result


    def gather_header_binding_includes(self, explicit_target_type_mappings=[]):
        return self.gather_binding_includes(exclude_self=True, add_bases=True, add_enums=True, add_classes=True, explicit_target_type_mappings=explicit_target_type_mappings)

    def gather_source_binding_includes(self, explicit_target_type_mappings=[]):
        return self.gather_binding_includes(exclude_self=False, add_bases=False, add_enums=False, add_classes=False, explicit_target_type_mappings=explicit_target_type_mappings)


    def gather_binding_includes(self, exclude_self, add_bases, add_enums, add_classes, explicit_target_type_mappings=[]):

        """
        Returns the paths of all the generated header files that this type needs in its own header file.
        result = [
            '#include "a/b/c.h"',
            '#include <collection.h>'
        ]
        """

        native_type_names = {
            "native_types": []
        }

        # Determine unique native types used
        self.gather_native_types(self.type_info, native_type_names, add_bases=add_bases, add_enums=add_enums, add_classes=add_classes)

        type_names = native_type_names["native_types"]

        # We don't want our own header file
        if exclude_self and self.type_info.type_name in type_names:
            type_names.remove(self.type_info.type_name)
        else:
            if self.type_info.type_name not in type_names:
                type_names.append(self.type_info.type_name)

        # Create the type mappings to include
        type_mappings = self.native_types_to_type_mappings(type_names)
        type_mappings.extend(explicit_target_type_mappings)

        # Find all the binding types and their headers
        result = self.get_binding_dependency_headers(type_mappings)

        # Wrap in #include
        result = map(lambda x: '#include "' + x + '"', result)

        # Platform-specific
        result.append('#include <collection.h>')

        return {
            'includes': result,
            'included_mappings': type_mappings
        }


    def get_class_type(self, type_mapping, is_native):
        if is_native:
            class_type = self.get_native_class_type(type_mapping)
        elif isinstance(type_mapping, common.configuration.ClassConfigTypeMapping):
            class_type = 'ref class'
        elif isinstance(type_mapping, common.configuration.StructConfigTypeMapping):
            if type_mapping.is_abstract:
                class_type = 'interface class'
            else:
                class_type = 'ref struct'
        elif isinstance(type_mapping, common.configuration.InterfaceConfigTypeMapping):
            class_type = 'interface class'
        elif isinstance(type_mapping, common.configuration.EnumerationConfigTypeMapping):
            class_type = 'enum class'
        elif isinstance(type_mapping, common.configuration.InterfaceAdapterConfigTypeMapping):
            class_type = 'ref class'
        elif isinstance(type_mapping, common.configuration.UnionConfigTypeMapping):
            class_type = 'ref struct'
        else:
            raise Exception('Unhandled mapping type: ' + str(type_mapping))

        return class_type


    def determine_target_type_name_from_variable(self, variable_type_info, owning_type_info=None):

        # Determine if it's an array
        is_array = False
        native_type_name = variable_type_info.type_name
        if variable_type_info.name in self.array_member_lengths:
            is_array = True
            native_type_name = self.binding_generator.strip_rightmost_array_modifier(native_type_name)

        def lookup_binding_type(native_type_name):
            # Look in the extracted type info
            native_type_info = None
            if native_type_name in self.binding_generator.extracted_type_info.all:
                native_type_info = self.binding_generator.extracted_type_info.all[native_type_name]

            # See what type this maps to
            target_type_name = self.binding_generator.find_mapped_type(native_type_name)

            is_enum = native_type_info and isinstance(native_type_info, common.source_tools.CppEnum)

            if not self.is_primitive_type(target_type_name) and not is_enum:
                target_type_name = target_type_name + '^'

            return target_type_name

        # If it's a templated type extract the param types
        if variable_type_info.template_info:
            target_type_name = self.binding_generator.find_mapped_type(variable_type_info.template_info.template_name)

            # Just use a pointer to the object
            if target_type_name == '^':
                if len(variable_type_info.template_info.template_args) != 1:
                    raise Exception("Found a pointer template type that doesn't have only one param")
                arg_binding_type = lookup_binding_type(variable_type_info.template_info.template_args[0])
                target_type_name = arg_binding_type

            # Translate the template
            else:
                #print(variable_type_info.template_info.template_name + ' --->> ' + target_type_name)
                target_type_name = target_type_name + '<'
                for index, template_arg in enumerate(variable_type_info.template_info.template_args):
                    arg_binding_type = lookup_binding_type(template_arg)
                    if index > 0:
                        target_type_name = target_type_name + ', '
                    target_type_name = target_type_name + arg_binding_type

                target_type_name = target_type_name + '>^'

        # Not a templated type
        else:
            target_type_name = lookup_binding_type(native_type_name)

        # Special case strings
        target_char_type = self.binding_generator.find_mapped_type('::char')
        if target_type_name == target_char_type and is_array:
            target_type_name = self.binding_generator.find_mapped_type('::std::string') + '^'
            is_array = False

        # If an array then wrap in platform array
        if is_array:
            target_type_name = 'Platform::Array<' + target_type_name + '>^'

        return target_type_name


    def get_forward_declaration_lines(self, explicit_native_type_mappings=[], explicit_target_type_mappings=[], exclude_type_mappings=[]):

        """Gathers the types referenced in the root type and generates the lines needed to forward declare them."""

        native_type_info = {
            "native_types": []
        }

        # Recursively find all native types in use
        self.gather_native_types(self.type_info, native_type_info)

        # Convert all native types to binding types
        type_mappings = self.native_types_to_type_mappings(native_type_info["native_types"])

        lines = []
        namespace_stack = []

        def append_line(line):
            lines.append( (self.config.indent * len(namespace_stack)) + line )

        def open_namespace(name):
            append_line('namespace ' + name)
            append_line('{')
            namespace_stack.append(name)

        def close_namespace():
            del namespace_stack[-1]
            append_line('}')
            if len(namespace_stack) == 0:
                append_line('')

        def add_forward_declaration_from_mapping(dependency_type_mapping, is_target):

            # Exclude if requested
            if len(filter(lambda x: x.target_type_name == dependency_type_mapping.target_type_name, exclude_type_mappings)) > 0:
                return

            # Get the typename parts
            type_name = None
            is_native = not is_target or isinstance(dependency_type_mapping, common.configuration.InterfaceAdapterConfigTypeMapping)

            if is_target:
                type_name = dependency_type_mapping.target_type_name
            else:
                type_name = dependency_type_mapping.native_type_name

            if is_native:
                type_name_components = common.native_tools.get_native_type_name_components(type_name)
            else:
                type_name_components = self.get_target_type_name_components(type_name)

            # Manage the current namespace
            namespace_tokens = type_name_components['namespace_tokens']

            # Close namespaces that don't match
            index = 0
            while index < len(namespace_stack) and index < len(namespace_tokens):
                if namespace_stack[index] != namespace_tokens[index]:
                    for i in range(index, len(namespace_stack)):
                        close_namespace()
                    break
                else:
                    index = index + 1

            # Open namespaces
            while index < len(namespace_tokens):
                open_namespace(namespace_tokens[index])
                index = index + 1

            class_type = self.get_class_type(dependency_type_mapping, is_native)
            append_line(class_type + ' ' + type_name_components['name'] + ';')

        # Primitives don't need forward declaration
        #type_mappings = filter(lambda x: not self.is_primitive_type(x), type_mappings)
        type_mappings = sorted(type_mappings, key=lambda x: x.target_type_name)

        # Add explicit types
        for x in explicit_target_type_mappings:
            if x not in type_mappings:
                type_mappings.append(x)

        # Map native types to binding types and forward declare those
        for dependency_type_mapping in type_mappings:
            # You can't forward declare enums
            if not isinstance(dependency_type_mapping, common.configuration.EnumerationConfigTypeMapping):
                add_forward_declaration_from_mapping(dependency_type_mapping, is_target=True)

        # Forward declare native types
        for x in explicit_native_type_mappings:
            add_forward_declaration_from_mapping(self.type_mapping, is_target=False)

        # Close remaining namespaces
        while len(namespace_stack) > 0:
            close_namespace()

        return lines


    def get_header_field_lines(self, explicit_fields=[]):

        lines = []

        def process_member(member_type_info, is_virtual):
            # Skip array length members since we use array objects
            if member_type_info.name.endswith('ArrayLength'):
                return

            target_type_name = self.member_type_names[member_type_info.name]
            #print(member.type_name + ' --> ' + native_type_name + ' --> ' + target_type_name)

            # Determine if overriding a base class or implementing an interface
            virtual = ''
            if is_virtual:
                virtual = 'virtual '

            line = virtual + "property " + target_type_name + ' ' + member_type_info.name + ";"
            lines.append(line)

        # Base class overrides
        for member_type_info in self.override_members:
            process_member(member_type_info, True)

        # Extracted
        for member_type_info in self.type_info.members:
            process_member(member_type_info, False)

        # Explicit fields
        for x in explicit_fields:
            lines.append(x)

        # Append custom fields
        custom_headers = self.type_mapping.load_custom_value_as_file_path('header.fields')
        if custom_headers:
            for x in custom_headers:
                lines.append(x)

        return lines


    def get_binding_method_line(self, method_type_info, is_declaration, is_virtual=False, is_override=False, access_modifier='public'):

        """Forward declares the given class method in the header."""

        line = ''

        # public/private/internal
        if access_modifier is not None and is_declaration:
            line = line + access_modifier + ': '

        # virtual
        if is_virtual:
            line = line + 'virtual '

        # Output the return value type
        if method_type_info.return_type is None:
            line = line + 'void '
        else:
            line = line + self.determine_target_type_name_from_variable(method_type_info.return_type, None) + ' '

        # Owning class
        if not is_declaration and self.type_mapping:
            type_name_components = self.get_target_type_name_components(self.type_mapping.target_type_name)
            line = line + self.type_mapping.target_type_name + '::'

        # Method name
        line = line + method_type_info.name + '('

        # Arguments
        args = []
        for arg in method_type_info.arguments:
            target_type_name = self.determine_target_type_name_from_variable(arg, None)
            token = target_type_name
            if arg.out_param:
                token = token + '*'
            token = token + ' ' + arg.name
            args.append(token)

        # Close the method
        line = line + ', '.join(args) + ')'

        if is_override:
            line = line + ' override'

        if is_declaration:
            line = line + ';'

        return line


    def get_header_binding_method_lines(self, methods, virtual_method_names=[], override_method_names=[], access_modifier='public'): # CppClass

        lines = []

        # See if there is an explicit list of methods to export
        class_config = self.config.find_class_configuration(self.type_info.type_name)

        # Generate each method line
        for method_type_info in methods:

            if class_config is None or method_type_info.name in class_config.export_methods:
                is_virtual = method_type_info.name in virtual_method_names
                is_override = method_type_info.name in override_method_names
                line = self.get_binding_method_line(method_type_info, is_declaration=True, is_virtual=is_virtual, is_override=is_override, access_modifier=access_modifier)

                lines.append(line)

        return lines
