# Usage
#
#   generate_class_info_func.py
#     --class-path=<class path>
#     --class=<class to export>
#     --hpp=<output .h file>
#     --cpp=<output .cpp file>
#     --pch=<pch include path>
#     --namespace=<C++ namespace>

import argparse
import sys
import os
import re
import platform
from pprint import pprint
from subprocess import Popen, PIPE


def determine_header_include_path(hpp_path):
    include_path = hpp_path
    path_tokens = hpp_path.split('/')
    index = path_tokens.index('include')

    if index >= 0:
        path_tokens = path_tokens[index+1:]
        return '/'.join(path_tokens)
    else:
        return None


def copyright():
    return """/********************************************************************************************
* Twitch Broadcasting SDK
*
* This software is supplied under the terms of a license agreement with Twitch Interactive, Inc. and
* may not be copied or used except in accordance with the terms of that agreement
* Copyright (c) 2012-2017 Twitch Interactive, Inc.
*********************************************************************************************/

"""


def generate(java_class_name, class_path, hpp_path, cpp_path, pch_path, namespace):

    namespace = namespace or ''

    print('java_class_name: ' + str(java_class_name))
    print('class_path: ' + str(class_path))
    print('hpp_path: ' + str(hpp_path))
    print('cpp_path: ' + str(cpp_path))
    print('pch_path: ' + str(pch_path))
    print('namespace: ' + str(namespace))

    java_short_class_name = java_class_name.split(".")[-1]
    java_short_class_name = java_short_class_name.replace("$", "_");
    java_class_relative_path = None

    if hpp_path:
        hpp_path = hpp_path.replace('\\', '/')

    # Read the output from javap
    if platform.system() == 'Windows':
        cmd = os.path.expandvars("%JAVA_HOME%/bin/javap.exe")
    else:
        cmd = os.path.expandvars("javap")

    process = Popen([cmd, "-classpath", class_path, "-s", java_class_name], stdout=PIPE)
    (output, err) = process.communicate()
    exit_code = process.wait()

    if exit_code != 0:
        print("Failed to run javap on type: " + java_class_name)
        return

    # Parse the output
    output = output.decode("utf-8")
    lines = output.split("\n")

    #pprint (lines)

    ctors = []
    methods = []
    static_methods = []
    fields = []
    static_fields = []

    for i in range(len(lines)):
        line = lines[i].strip()
        print(line)
        if not java_class_relative_path:
            class_declarations = ["public class ", "public abstract class ", "public final class ", "public interface ", "public final interface "]
            for class_declaration in class_declarations:
                if line.startswith(class_declaration):
                    (java_full_class_name, _, _) = line[len(class_declaration):].partition(" ")
                    java_class_relative_path = java_full_class_name.replace(".", "/")

                    # strip away any templates from the class name
                    java_class_relative_path = re.sub('<.*?>', '', java_class_relative_path)
                    break
            continue;

        match = re.search('\s*(Signature:|descriptor:)\s*(\S*)', line)

        if not match:
            continue

        # Found a signature
        sig = match.group(2)

        line = lines[i-1].strip()

        # Find the name and args
        match = re.search('(\S+)(\(.*\))(.*);', line)

        # Not a method, must be a field
        is_field = False
        if not match:
            is_field = True
            match = re.search('(\S+)(\s*);', line)
            if not match:
                print("Unhandled type: " + line)

        name = match.group(1)
        args = match.group(2)

        entry = {
            'name': name,
            'sig': sig,
            'args': args,
            'desc': line
        }

        # Parse the attributes
        attributes = line[:match.start(1)-1].strip()
        attributes = attributes.split(" ")

        if is_field:
            if 'static' in attributes:
                static_fields.append(entry)

            else:
                fields.append(entry)

        else:
            if 'static' in attributes:
                static_methods.append(entry)

            elif name == java_class_name:
                ctors.append(entry)

            else:
                methods.append(entry)


    if not java_class_relative_path:
        raise Exception("Could not find class declaration")

    namespace_tokens = namespace.split('::')

    if namespace != '':
        namespace = namespace + '::'

    # Generate the header code
    hpp_content = copyright() + """#pragma once

#include "twitchsdk/core/java_classinfo.h"

"""

    indentation_level = 0
    indentation = '    '
    for ns in namespace_tokens:
        hpp_content += indentation * indentation_level + "namespace " + ns + "\n"
        hpp_content += indentation * indentation_level + "{\n"
        indentation_level += 1

    hpp_content += indentation * indentation_level + "::ttv::binding::java::JavaClassInfo& GetJavaClassInfo_" + java_short_class_name + "(JNIEnv* jEnv);\n"

    while indentation_level > 0:
        indentation_level -= 1
        hpp_content += indentation * indentation_level + "}\n"


    # Generate the source file
    cpp_content = copyright()

    # Precompiled header
    if pch_path:
        cpp_content += '#include "' + pch_path + '"' + "\n"

    # Determine the include path to the header
    if hpp_path:
        include_path = determine_header_include_path(hpp_path)

        if include_path:
            cpp_content += '#include "' + include_path + '"' + "\n"

    cpp_content += "\n"
    cpp_content += ("::ttv::binding::java::JavaClassInfo& " + namespace + "GetJavaClassInfo_" + java_short_class_name + "(JNIEnv* jEnv)")
    cpp_content += """
{
    static ::ttv::binding::java::JavaClassInfo info;
    static bool initialized = false;

    if (!initialized)
    {
        initialized = true;

"""

    cpp_content += indentation * 2 + 'LookupJavaClass(jEnv, info, "' + java_class_relative_path + '");' + "\n"

    if len(ctors) > 0:
        cpp_content += "\n"

        # For now, prefer the ctor that takes a value, if it exists.  Otherwise use the default
        for entry in ctors:
            cpp_content += indentation * 2 + 'LookupJavaMethod(jEnv, info, "<init>", "' + entry['sig'] + '"); // ' + entry['desc'] + "\n"

    if len(static_methods) > 0:
        cpp_content += "\n"

        for entry in static_methods:
            cpp_content += indentation * 2 + 'LookupJavaStaticMethod(jEnv, info, "' + entry['name'] + '", "' + entry['sig'] + '"); // ' + entry['desc'] + "\n"

    if len(methods) > 0:
        cpp_content += "\n"

        for entry in methods:
            cpp_content += indentation * 2 + 'LookupJavaMethod(jEnv, info, "' + entry['name'] + '", "' + entry['sig'] + '"); // ' + entry['desc'] + "\n"

    if len(fields) > 0:
        cpp_content += "\n"

        for entry in fields:
            cpp_content += indentation * 2 + 'LookupJavaField(jEnv, info, "' + entry['name'] + '", "' + entry['sig'] + '"); // ' + entry['desc'] + "\n"

    if len(static_fields) > 0:
        cpp_content += "\n"

        for entry in fields:
            cpp_content += indentation * 2 + 'LookupJavaStaticField(jEnv, info, "' + entry['name'] + '", "' + entry['sig'] + '"); // ' + entry['desc'] + "\n"


    cpp_content += indentation + """}

    return info;
}
"""

    if hpp_path:
        file = open(hpp_path, 'w')
        file.write(hpp_content)
        file.close()

    if cpp_path:
        file = open(cpp_path, 'w')
        file.write(cpp_content)
        file.close()
    else:
        print(cpp_content)


# Process the command line arguments if run as the primary script
if __name__ == "__main__":
    parser = argparse.ArgumentParser(description = 'Tool for generating java class info function declarations.')
    parser.add_argument(
        '--class-path',
        required=True,
        metavar='<java class path>',
        action='append',
        help='Specifies the Java class path.'
    )
    parser.add_argument(
        '--class',
        required=True,
        metavar='<java class name>',
        help='The java class name.'
    )
    parser.add_argument(
        '--hpp',
        required=False,
        metavar='<output .h path>',
        help='The output .h file path.'
    )
    parser.add_argument(
        '--cpp',
        required=False,
        metavar='<output .cpp path>',
        help='The output .cpp file path.'
    )
    parser.add_argument(
        '--namespace',
        required=False,
        metavar='<code namespace>',
        help='The namespace to place the code in.'
    )
    parser.add_argument(
        '--pch',
        required=False,
        metavar='<pch include>',
        help='The path of the pch to include in the generated .cpp file.'
    )

    args = parser.parse_args()

    java_class_name = vars(args)['class']
    class_path = args.class_path
    hpp_path = args.hpp
    cpp_path = args.cpp
    pch_path = args.pch
    namespace = args.namespace

    # Transform the list of class paths into a string
    if isinstance(class_path, list):
        if platform.system() == 'Windows':
            class_path = ';'.join(class_path)
        else:
            class_path = ':'.join(class_path)

    generate(java_class_name, class_path, hpp_path, cpp_path, pch_path, namespace)
