import subprocess
import platform
import os
import sys
import glob
import re
import argparse


class JavaTools:

    """
    Tools for generating java bindings and updating native code.  It will look for javac in the PATH and also in the bin dir
    under the environment variable JAVA_HOME if set.
    """

    def find_executable(self, program):

        """Finds the named java executable"""

        def is_exe(fpath):
            return os.path.isfile(fpath) and os.access(fpath, os.X_OK)

        result = None

        if platform.system() == 'Windows':
            program = program + '.exe'

        fpath, fname = os.path.split(program)
        if fpath:
            if is_exe(program):
                result = program
        else:
            for path in os.environ["PATH"].split(os.pathsep):
                path = path.strip('"')
                exe_file = os.path.join(path, program)
                if is_exe(exe_file):
                    result = exe_file
                    break

        if not result:
            java_home = os.environ["JAVA_HOME"]
            if java_home:
                path = os.path.join(java_home, 'bin', program)
                if is_exe(path):
                    result = path

        if not result:
            raise Exception('Could not find ' + program)

        return result


    def compile_source_files(self, java_source_files, class_paths, output_root_dir):

        """
        Compiles the given list of java source files.
        """

        if not os.path.exists(output_root_dir):
            os.makedirs(output_root_dir)

        exe_path = self.find_executable('javac')

        shell_args = [exe_path]

        # Output directory
        shell_args.extend(['-d', output_root_dir])

        for class_path in class_paths:
            shell_args.extend(['-classpath', class_path])

        # .java files to compile
        for path in java_source_files:
            shell_args.extend([path])

        print('Running Javac:')
        print(" ".join(str(x) for x in shell_args))

        subprocess.call(shell_args)


    def compile_recursively(self, java_source_root_dir, class_paths, output_root_dir):

        """
        Recursively descends the directory heirarchy compiling all .java files and outputs them
        in the proper location in the output_root_dir.
        """

        java_source_root_dir = os.path.realpath(java_source_root_dir).replace('\\', '/')

        # Gather up all the .java files
        all_source_files = []

        for root, dirnames, filenames in os.walk(java_source_root_dir):
            for filename in filenames:
                if filename.endswith('.java'):
                    all_source_files.append(os.path.join(root, filename).replace('\\', '/'))

        # Compile all at once so references are resolved
        self.compile_source_files(java_source_files=all_source_files, class_paths=class_paths, output_root_dir=output_root_dir)


    def generate_header(self, class_name, java_class_paths, header_path):

        """Generates native header for the JNI entrypoints"""

        exe_path = self.find_executable('javah')

        shell_args = [exe_path]

        # Class path
        if java_class_paths:
            paths = ";".join(str(x) for x in java_class_paths)
            shell_args.extend(['-classpath', java_class_paths])

        # Output header
        shell_args.extend(['-o', header_path])

        # Java class
        shell_args.extend([class_name])

        self.execute(shell_args)


    def generate_exports(self, native_source_dirs, exports_path):

        """Generates the platform-independent definitions file for java function symbols to be exported from the dynamic library"""

        symbols = []

        # Find all the symbols
        for dir in native_source_dirs:
            header_files = glob.glob( os.path.join(dir, '*.h') )

            for header_file in header_files:
                with open(header_file) as file:
                    lines = file.read().splitlines()

                for line in lines:
                    line = line.strip()

                    match = re.search('JNIEXPORT\s+\S+\s+JNICALL\s+(.+)', line)

                    if not match:
                        continue

                    symbol = match.group(1)
                    symbols.append(symbol)

        # Write the symbols file
        file = open(exports_path, 'w')
        for symbol in symbols:
            file.write(symbol + '\n')
        file.close()


    def execute(self, shell_args):
        print('Running ' + shell_args[0] + '...')
        print(" ".join(str(x) for x in shell_args))
        subprocess.call(shell_args)


# Parse the command line if the main script
if __name__ == "__main__":

    parser = argparse.ArgumentParser(description='Tools for generating Java bindings.')

    parser.add_argument(
        '--compile',
        required=False,
        action='store_true',
        help='Specifies to compile java code.'
    )

    parser.add_argument(
        '--generate-header',
        action='store_true',
        required=False,
        help='Specifies to generate a native header from a .class file.'
    )

    parser.add_argument(
        '--generate-exports',
        action='store_true',
        required=False,
        help='Specifies to generate the exports.txt file from native java entrypoints.'
    )

    parser.add_argument(
        '--source-dir',
        required=False,
        metavar='<java source dir>',
        action='append',
        help='Specifies the Java source directory to compile.'
    )

    parser.add_argument(
        '--class-path',
        required=False,
        metavar='<java class path>',
        action='append',
        help='Specifies the Java class path.'
    )

    parser.add_argument(
        '--class',
        required=False,
        metavar='<java class name>',
        help='The java class name.'
    )

    parser.add_argument(
        '--output-dir',
        required=False,
        metavar='<output path>',
        help='Specifies the output directory for .class files.'
    )

    parser.add_argument(
        '--output',
        required=False,
        metavar='<output path>',
        help='Specifies the output file.'
    )

    args = parser.parse_args()

    # Compile java files
    if args.compile:
        # java_tools.py --compile --source-dir="C:/Drew/Repositories/sdk/sdk_cmake/core/core_java/bindings/tv/twitch" --output-dir="c:/Drew/junk"
        tools = JavaTools()
        java_source_dirs = args.source_dir
        output_dir = args.output_dir
        tools.compile(java_source_dirs, output_dir)

    # Generate the native header file for the native methods for a java class
    elif args.generate_header:
        # java_tools.py --generate-header --class="tv.twitch.CoreAPI" --class-path="c:/Drew/junk" --output="../include/twitchsdk/core/java_coreapi.h"
        tools = JavaTools()
        class_name = vars(args)['class']
        java_class_paths = args.class_path

        # Transform the list of class paths into a string
        if isinstance(java_class_paths, list):
            if platform.system() == 'Windows':
                java_class_paths = ';'.join(java_class_paths)
            else:
                java_class_paths = ':'.join(java_class_paths)
                        
        output_header = args.output
        tools.generate_header(class_name, java_class_paths, output_header)

    # Generate symbol exports file
    elif args.generate_exports:
        # java_tools.py --generate-exports --source-dir="../include/twitchsdk/core" --output="../source/exports.txt"
        tools = JavaTools()
        native_source_paths = args.source_dir
        output_exports = args.output
        tools.generate_exports(native_source_paths, output_exports)
