# The goal is for this file to run on specific .graphql files to make it easier to add new GQL queries in the future
# Note that this will not work for GQL Union objects since we currently do not have a way to JSON serialize ttv::Variant
# Bug where it doesn't work properly for fragments in the top-level payload (see TODO at the end of parse_graphql_file)
# Currently prints out the result but can output it to a file by uncommenting the very end of the script
# SDK-419

import json
import os
import re

# Maps of name of type -> class that represents the object
# Filled in parse_schema()
input_object_schemas = {}
object_schemas = {}
enum_schemas = {}
union_schemas = {}
mutation_schemas = {}
query_schemas = {}
gql_kind_map = {} # map of name of type ("ChatRoomMessage") to kind ("OBJECT")

# Map from type name to recursive type unique identifier
type_name_map = dict()
# Map of the concatenation of GQL names + type name to the suffix that is used to disambiguate the recursive type
type_name_suffix_map = dict()
# All of the GQL names we have encountered thus far while recursing through the GQL Query
# These names are used to uniquely identify recursive types
recursive_name_stack = []

gql_to_c_basic_types = {'Int':'int32_t', 'String':'std::string', 'ID':'std::string', 'Cursor':'std::string', 'Time':'ttv::Timestamp', 'Boolean':'bool', 'Float':'float'}
INPUT_PARAM_STRUCT_NAME = 'InputParams'
PAYLOAD_STRUCT_NAME = 'PayloadType'
AUTH_TOKEN_FIELD_NAME = 'authToken'

# CLASSES
class GraphQLLexer:
    """ move_to_next_token() needs to be called before fetching tokens """
    def __init__(self, gql_content):
        self.gql_content = gql_content.strip()
        self.current_start_index = 0
        self.current_end_index = 0
        self.next_start_index = 0
        self.next_end_index = 0
        self.special_tokens = ['(', ')', '{', '}', ':', '$', ',']

    def current_token(self):
        return self.gql_content[self.current_start_index : self.current_end_index]

    def peek_next_token(self):
        return self.gql_content[self.next_start_index : self.next_end_index]

    def move_to_next_token(self, debug = False):
        """ Returns False if there is no more token to move to, else returns True """
        if self.next_start_index == 0:
            # First time calling this function, determine the first token
            while self.gql_content[self.current_start_index] == '#':
                self.current_start_index = self.gql_content.find('\n', self.current_start_index) + 1
                while self.current_start_index < len(self.gql_content) and self.gql_content[self.current_start_index].isspace():
                    self.current_start_index += 1
                self.current_end_index = self.current_start_index

            while self.current_end_index <= len(self.gql_content) and self.gql_content[self.current_start_index : self.current_end_index] not in self.special_tokens and not self.gql_content[self.current_end_index].isspace():
                self.current_end_index += 1
        else:
            self.current_start_index = self.next_start_index
            self.current_end_index = self.next_end_index

        if self.current_start_index == len(self.gql_content):
            # End of file
            return False

        # Determine the next token
        self.next_start_index = self.current_end_index
        while self.next_start_index < len(self.gql_content) and self.gql_content[self.next_start_index].isspace():
            self.next_start_index += 1

        if self.next_start_index < len(self.gql_content):
            while self.gql_content[self.next_start_index] == '#':
                self.next_start_index = self.gql_content.find('\n', self.next_start_index) + 1
                while self.next_start_index < len(self.gql_content) and self.gql_content[self.next_start_index].isspace():
                    self.next_start_index += 1

        self.next_end_index = self.next_start_index

        if self.next_start_index == len(self.gql_content):
            # No more tokens after the current token
            self.next_end_index = self.next_start_index
        elif self.gql_content[self.next_start_index] in self.special_tokens:
            self.next_end_index = self.next_start_index + 1 # assumes special tokens can only be 1 char in length
        else:
            while self.next_end_index <= len(self.gql_content) and self.gql_content[self.next_end_index] not in self.special_tokens and not self.gql_content[self.next_end_index].isspace():
                self.next_end_index += 1

        if debug:
            print('{}, {}').format(self.current_token(), self.peek_next_token())

        return True

# Classes populated from the GQL Schema:
class Field:
    def __init__(self, name, type, args, is_list, list_nullable, nullable):
        self.name = name # e.g. "code", not for payloads
        self.type = type # e.g. "BanUserFromChatRoomErrorCode"
        self.args = args # not for payloads, list of Field
        self.is_list = is_list
        self.list_nullable = list_nullable # only used when is_list is True
        self.nullable = nullable

class QuerySchema:
    def __init__(self, name, args, payload):
        self.name = name
        self.args = args # list of Field
        self.payload = payload # Field

class Enum:
    def __init__(self, name, values):
        self.name = name
        self.values = values # list of strings

class Union:
    def __init__(self, name, possible_types):
        self.name = name
        self.possible_types = possible_types # list of Field

class InputObjectInfo:
    def __init__(self, name, input_fields):
        self.name = name
        self.input_fields = input_fields # list of Field

class ObjectInfo:
    def __init__(self, name, fields):
        self.name = name
        self.fields = fields # map from name of field to Field

# Classes populated from .graphql files:
class PayloadField:
    def __init__(self, name):
        self.name = name
        self.child_fields = [] # list of other PayloadFields that can be queried on this field
        self.fragment_fields = [] # list of strings of fragment names that are on the field

        # list of other PayloadFields that are queried depending on the true type of the current payload field
        # The name field of the elements in the list represents the type of the payload field instead of the name
        self.child_union_fields = []

class GQLQuery:
    def __init__(self, is_mutation, query_name, args, query_function, payload, fragment_names, source_file_name, gen_full):
        self.is_mutation = is_mutation
        self.query_name = query_name
        self.args = args # dictionary of name of input -> type of input
        self.query_function = query_function
        self.payload = payload # list of PayloadFields
        self.fragment_names = fragment_names # list of strings containing the filenames of the graphql fragment files
        self.source_file_name = source_file_name # name of the .graphql file this was created from
        self.gen_full = gen_full # True if we're generating the whole info struct, False if we're only generating the function to get the raw string

class GQLFragment:
    def __init__(self, name, type, payload, source_file_name, fragment_names):
        self.name = name
        self.type = type
        self.payload = payload # list of PayloadFields
        self.source_file_name = source_file_name
        self.fragment_names = fragment_names # list of strings containing the filenames of the graphql fragment files

# Classes to be converted to C++ code:
class GQLQueryHeader:
    def __init__(self, query_info, namespaces):
        self.query_info = query_info
        self.headers = [] # list of full paths to header files
        self.structs = [] # list of tuple (name of struct, list of tuples (type, name of variable))
        self.union_schemas = [] # list of tuple (name of schema, list of union structs)
        self.enums = {} # maps from name of enum -> list of strings of enum values
        self.namespaces = namespaces

class GQLFragmentHeader:
    def __init__(self, fragment_info, namespaces):
        self.fragment_info = fragment_info
        self.headers = []
        self.structs = []
        self.enums = {}
        self.namespaces = namespaces

def lowercase(str):
    return '{}{}'.format(str[0].lower(), str[1:])

# PRINT FUNCTIONS
def print_field_info_helper(field_info):
    if field_info.name:
        type_string = '{}'.format(field_info.name)
        if field_info.args:
            type_string += '('
            for arg in field_info.args[:-1]:
                type_string += '{}, '.format(print_field_info_helper(arg))
            type_string += '{})'.format(print_field_info_helper(field_info.args[-1]))
        type_string += ': '
    else:
        type_string = ''

    if field_info.is_list:
        type_string += '['
    type_string += field_info.type
    if not field_info.nullable:
        type_string += '!'
    if field_info.is_list:
        type_string += ']'
        if not field_info.list_nullable:
            type_string += '!'

    return type_string

def print_field_info(field_info, indent):
    print '{}{}'.format(indent, print_field_info_helper(field_info))

def print_payload_fields(payload_field, indent, indentation_level = 0):
    print '{}{}'.format(indent * indentation_level, payload_field.name)
    for field in payload_field.child_fields:
        print_payload_fields(field, indent, indentation_level + 1)

def print_queries(queries):
    for key, value in queries.iteritems():
        print(key)
        print('args:')
        for arg in value.args:
            print_field_info(arg, '  ')
        print('payload:')
        print_field_info(value.payload, '  ')
        print('')

# PARSING FUNCTIONS
def is_valid_gql_name(name):
    return re.match('^[_A-Za-z][_0-9A-Za-z]*$', name) is not None

def parse_field(json_field, is_named):
    """ Returns a Field object """
    args = []

    if is_named:
        name = json_field['name']

        if json_field.get('args'):
            args = [parse_field(arg, True) for arg in json_field['args']]

        type_info = json_field['type']
    else:
        name = None
        type_info = json_field

    if type_info['kind'] == 'NON_NULL':
        nullable = False
        type_info = type_info['ofType']
    else:
        nullable = True

    list_nullable = None
    if type_info['kind'] == 'LIST':
        is_list = True
        type_info = type_info['ofType']
        list_nullable = nullable
        if type_info['kind'] == 'NON_NULL':
            nullable = False
            type_info = type_info['ofType']
        else:
            nullable = True
    else:
        is_list = False

    type = type_info['name']

    if is_named:
        return Field(name, type, args, is_list, list_nullable, nullable)
    else:
        return Field(None, type, args, is_list, list_nullable, nullable)

def parse_schema(schema_file_path):
    with open(schema_file_path) as file:
        schema = json.load(file)['__schema']

    types = schema['types']
    queryType = schema['queryType']['name']
    mutationType = schema['mutationType']['name']

    for type in types:
        kind = type['kind']
        name = type['name']
        gql_kind_map[name] = kind
        if kind == 'OBJECT':
            if name == queryType or name == mutationType:
                for query in type['fields']:
                    query_name = query['name']

                    query_args = []
                    for jsonArg in query['args']:
                        arg = parse_field(jsonArg, True)
                        if arg is not None:
                            query_args.append(arg)
                    query_payload_type = parse_field(query['type'], False)

                    if name == queryType:
                        query_schemas[query_name] = QuerySchema(query_name, query_args, query_payload_type)
                    else:
                        mutation_schemas[query_name] = QuerySchema(query_name, query_args, query_payload_type)
            else:
                fields = {}
                for field in type['fields']:
                    field = parse_field(field, True)
                    fields[field.name] = field
                object_schemas[name] = ObjectInfo(name, fields)

        elif kind == 'ENUM':
            values = [value['name'] for value in type['enumValues']]
            enum_schemas[name] = Enum(name, values)

        elif kind == 'INPUT_OBJECT':
            inputs = [parse_field(input, True) for input in type['inputFields']]
            input_object_schemas[name] = InputObjectInfo(name, inputs)

        elif kind == 'UNION':
            possible_types = [parse_field(possible_type, False) for possible_type in type['possibleTypes']]
            union_schemas[name] = Union(name, possible_types)

        elif kind == 'INTERFACE':
            pass
        elif kind == 'SCALAR':
            pass
        else:
            print('Not handling {} kind, name {}'.format(kind, name))

    #print_queries(mutation_schemas)
    #print_queries(query_schemas)

def parse_payload_fields(current_payload_field, lexer, file_dir):
    if lexer.current_token() != '{' or lexer.peek_next_token() in ['{', '}']:
        return None

    lexer.move_to_next_token()

    fragment_names = []

    while lexer.current_token() != '}':
        if lexer.current_token() == '...':
            lexer.move_to_next_token()

            if not is_valid_gql_name(lexer.current_token()):
                print 'Unexpected token after ...'
                return None

            if lexer.current_token() == 'on':
                # Union
                lexer.move_to_next_token()
                if not is_valid_gql_name(lexer.current_token()):
                    print 'Could not parse a union type'
                    return None
                current_payload_field.child_union_fields.append(PayloadField(lexer.current_token()))
                lexer.move_to_next_token()
                parse_payload_fields(current_payload_field.child_union_fields[-1], lexer, file_dir)
            else:
                # Fragment
                current_payload_field.fragment_fields.append(lexer.current_token())
                fragment_name = lexer.current_token().lower()
                fragment_names.append(fragment_name)
                lexer.move_to_next_token()

        elif lexer.current_token() == '{':
            child_fragment_names = parse_payload_fields(current_payload_field.child_fields[-1], lexer, file_dir)
            if child_fragment_names is None:
                return None
            fragment_names.extend(child_fragment_names)
        elif lexer.current_token() == '(':
            while lexer.current_token() != ')':
                if not lexer.move_to_next_token():
                    print 'Payload fields ended unexpectedly after a "("'
                    return None
            lexer.move_to_next_token()

        elif is_valid_gql_name(lexer.current_token()):
            current_payload_field.child_fields.append(PayloadField(lexer.current_token()))
            lexer.move_to_next_token()
        else:
            print 'Unexpected token: "{}"'.format(lexer.current_token())
            return None

    lexer.move_to_next_token()

    if(len(current_payload_field.child_union_fields) > 0):
        for child_field in current_payload_field.child_fields:
            if child_field.name == '__typename':
                break
        else:
            current_payload_field.child_fields.append(PayloadField(name='__typename'))
            # Field(name='__typename', type='SCALAR', is_list=False, nullable=False))

    return fragment_names

def parse_graphql_fragment_file(file_dir, file_name):
    with open(os.path.join(file_dir, file_name)) as file:
        fragment_contents = file.read()

    # TODO: Currently making a lot of assumptions here about the contents of the fragment as well as
    # not validating if the fragment's type matches
    lexer = GraphQLLexer(fragment_contents)

    lexer.move_to_next_token()
    if lexer.current_token() != 'fragment':
        print 'Error: {} does not start with "fragment"'.format(file_path)
        return

    lexer.move_to_next_token()
    fragment_name = lexer.current_token()
    if not is_valid_gql_name(fragment_name):
        print 'Error: could not parse fragment name in {}'.format(file_path)
        return

    lexer.move_to_next_token()
    if lexer.current_token() != 'on' or not is_valid_gql_name(lexer.peek_next_token()):
        print 'Error: could not parse the type of the fragment in {}'.format(file_path)
        return

    lexer.move_to_next_token()

    type = lexer.current_token()

    lexer.move_to_next_token()

    if lexer.current_token() != '{':
        print 'Error: could not parse the fields of the fragment in {}'.format(file_path)
        return

    top_level_payload = PayloadField(fragment_name)
    fragment_names = parse_payload_fields(top_level_payload, lexer, file_dir)
    if fragment_names is None:
        return
    # print_payload_fields(top_level_payload, '  ')

    return GQLFragment(fragment_name, type, top_level_payload.child_fields, file_name, fragment_names)

def parse_graphql_file(file_dir, file_name):
    file_path = os.path.join(file_dir, file_name)
    with open(file_path) as file:
        contents = file.read()

    gen_full = False
    if contents.startswith(('#gen_full', '# gen_full')):
        gen_full = True

    lexer = GraphQLLexer(contents)

    lexer.move_to_next_token()
    query_word = lexer.current_token()

    if query_word == 'mutation':
        isMutation = True
    elif query_word == 'query':
        isMutation = False
    else:
        print 'Error: {} is not a mutation or a query'.format(file_path)
        return

    lexer.move_to_next_token()
    query_name = lexer.current_token()
    if not is_valid_gql_name(query_name):
        print 'Error: Please give {} a name after "query" or "mutation".'.format(file_dir)
        return

    args = {} # string name of arg -> name of type of the argument, found in (input_)object_schemas dict
    if lexer.peek_next_token() == '(':
        lexer.move_to_next_token()
        error = False
        if not lexer.move_to_next_token():
            error = True
        while lexer.peek_next_token() != '{':
            if lexer.current_token() != '$':
                error = True
                break

            lexer.move_to_next_token()
            arg_name = lexer.current_token()
            if not is_valid_gql_name(arg_name):
                error = True
                break

            if lexer.peek_next_token() != ':':
                error = True
                break

            lexer.move_to_next_token()
            lexer.move_to_next_token()

            arg_type = lexer.current_token()
            if not arg_type:
                error = True
                break

            args[arg_name] = arg_type

            if lexer.peek_next_token() == ',':
                lexer.move_to_next_token()
                lexer.move_to_next_token()
            elif lexer.peek_next_token() == ')':
                lexer.move_to_next_token()
            else:
                error = True
                break

        if error:
            print 'Error: could not parse query inputs in {}'.format(file_dir)
            return

    if lexer.peek_next_token() != '{':
        print 'Error: {} is missing the body of the query'.format(file_dir)
        return

    lexer.move_to_next_token()
    lexer.move_to_next_token()

    query_function = lexer.current_token()
    if not is_valid_gql_name(query_function):
        print 'Error: could not parse the query function name in {}'.format(file_dir)

    # TODO: possibly validate the inputs to query function for the correct types. Otherwise, inputs can be ignored
    while lexer.current_token() != '{':
        # If the query has parameters, and therefore an open-paren, we need to find the close-paren before looking for an open-bracket.
        # ex: `unfollowGame(input: {gameID:$gameID})`. The open-bracket must be ignored.
        if lexer.current_token() == '(':
            while lexer.current_token() != ')':
                if not lexer.move_to_next_token():
                    raise Exception('Error: {} ended unexpectedly'.format(file_dir))
                    return

        if not lexer.move_to_next_token():
            print Exception('Error: {} ended unexpectedly'.format(file_dir))
            return

    top_level_payload = PayloadField('Payload:')
    fragment_names = parse_payload_fields(top_level_payload, lexer, file_dir)
    if fragment_names is None:
        return
    # print_payload_fields(top_level_payload, '  ')

    # TODO: fragment fields in the top-level payloads are not parsed,
    # causing the payload type to not be generated if the only field at the top level is the fragment
    # To fix, the entire top_level_payload should be passed into GQLQuery, or top-level fragments should be stored separately
    return GQLQuery(isMutation, query_name, args, query_function, top_level_payload.child_fields, fragment_names, file_name, gen_full)

# GQL TO C++ TRANSLATION FUNCTIONS
def gql_get_recursive_type_suffix(field):
    name_concatenation = ""
    for name in recursive_name_stack:
        name_concatenation += name

    name_type_concatenation = name_concatenation + field.type

    if field.type not in type_name_map:
        # If we have not seen this type before add it to the type_name_map to record it
        type_name_map[field.type] = [name_concatenation]
        type_name_suffix_map[name_type_concatenation] = 0
    else:
        # If we have seen this type before we need to check to see if we have seen this specific name
        if name_concatenation not in type_name_map[field.type]:
            # If we have not seen this name-type combo then we need a new suffix for the type to uniquely identify it
            type_name_map[field.type] += [name_concatenation]
            type_name_suffix_map[name_type_concatenation] = len(type_name_map[field.type]) - 1

    suffix = type_name_suffix_map[name_type_concatenation]
    if suffix:
        return str(suffix)
    else:
         return ''

def gql_field_to_c_type(field):
    if field.type in gql_to_c_basic_types:
        c_type = gql_to_c_basic_types[field.type]
    elif gql_kind_map[field.type] == 'OBJECT':
        c_type = field.type + gql_get_recursive_type_suffix(field)
    else:
        c_type = field.type

    if field.nullable:
        c_type = 'ttv::Optional<{}>'.format(c_type)

    if field.is_list:
        c_type = 'std::vector<{}>'.format(c_type)
        if field.list_nullable:
            c_type = 'ttv::Optional<{}>'.format(c_type)

    return c_type

# Returns a tuple, first element represents struct, second element represents enums
# structs are representated as list of tuples (name of struct, list of (c type, var name))
# enums are represented as a list of tuples (name of enum, list of enum values)
def gql_payloads_to_cpp(payload_fields, payload_object, recursive_suffix = "", top_level_payload = False, fragments = []):
    structs = []
    enums = []
    payload_variables = []
    union_variables = []
    union_schemas = []
    union_members = []
    for field in payload_fields:
        if field.name not in payload_object.fields:
            return
        field_info = payload_object.fields[field.name]
        c_field_type = gql_field_to_c_type(field_info)
        suffix = gql_get_recursive_type_suffix(field_info)
        if field_info.type not in gql_to_c_basic_types:
            if gql_kind_map[field_info.type] == 'OBJECT':
                recursive_name_stack.append(field_info.name)
                result = gql_payloads_to_cpp(field.child_fields, object_schemas[field_info.type], suffix, False, field.fragment_fields)
                recursive_name_stack.pop()
                if result is None:
                    return
                if result[0]:
                    structs.extend(result[0])
                if result[1]:
                    enums.extend(result[1])
                if result[2]:
                    union_schemas.extend(result[2])
            elif gql_kind_map[field_info.type] == 'ENUM':
                enums.append((field_info.type, enum_schemas[field_info.type].values))
            elif gql_kind_map[field_info.type] == 'UNION':
                # TODO: deal with unions that only have one possible type

                # TODO: a better way to do this?
                for child_field in field.child_fields:
                    if child_field.name == '__typename':
                        union_variables.append(('std::string', '__typename'))
                        break

                variant_type = 'ttv::Variant<'
                for union_field in field.child_union_fields:
                    # TODO: currently assuming it must be an object type
                    result = gql_payloads_to_cpp(union_field.child_fields, object_schemas[union_field.name])
                    if result is None:
                        return
                    if result[0]:
                        structs.extend(result[0])
                    if result[1]:
                        enums.extend(result[1])
                    if result[2]:
                        union_schemas.extend(result[2])

                    variant_type += '{}, '.format(union_field.name)
                    union_members.append(union_field.name)

                variant_type = variant_type[:-2] + '>'

                union_variables.append((variant_type, '{}Variant'.format(lowercase(field.name))))

                structs.append((field_info.type, union_variables))
                union_schemas.append((field_info.type + field.name[:1].upper() + field.name[1:] + 'SchemaSelector', union_members))
            else:
                return

        payload_variables.append((c_field_type, field.name))

    for fragment_name in fragments:
        payload_variables.append((fragment_name, lowercase(fragment_name)))

    # Not an error if no payload variables due to unions
    if payload_variables:
        if top_level_payload:
            structs.append((PAYLOAD_STRUCT_NAME, payload_variables))
        else:
            structs.append((payload_object.name + recursive_suffix, payload_variables))

    return (structs, enums, union_schemas)

def gql_to_cpp_header(gql_query, namespaces, header_include_path):
    gql_header_info = GQLQueryHeader(gql_query, namespaces)

    # Header
    if gql_query.fragment_names:
        gql_header_info.headers = [header_include_path + "{}{}".format(fragment, 'info.h') for fragment in gql_query.fragment_names]
    # TODO: should only add these when you need it instead of always
    gql_header_info.headers.append('twitchsdk/core/optional.h')
    gql_header_info.headers.append('twitchsdk/core/variant.h')
    gql_header_info.headers.append('twitchsdk/core/json/jsonserialization.h')

    types_to_generate = []

    # Input args struct
    input_param_variables = [('std::string', AUTH_TOKEN_FIELD_NAME)]
    for arg_name, arg_type in gql_query.args.iteritems():
        if arg_type.endswith('!'):
            c_arg_type = arg_type[:-1]
            optional = False
        else:
            c_arg_type = arg_type
            optional = True

        if c_arg_type in gql_to_c_basic_types:
            c_arg_type = gql_to_c_basic_types[c_arg_type]
        else:
            types_to_generate.append(c_arg_type)

        if optional:
            c_arg_type = 'ttv::Optional<{}>'.format(c_arg_type)

        input_param_variables.append((c_arg_type, arg_name))

    index = 0
    while index < len(types_to_generate):
        gql_type = types_to_generate[index]

        # if gql_type in gql_header_info.structs:
        if any(struct[0] == gql_type for struct in gql_header_info.structs):
            continue

        struct_variables = []
        field_infos = []

        if gql_kind_map[gql_type] == 'ENUM':
            gql_header_info.enums[gql_type] = enum_schemas[gql_type].values
        else:
            if gql_kind_map[gql_type] == 'OBJECT':
                field_infos = object_schemas[gql_type].fields.values()
            elif gql_kind_map[gql_type] == 'INPUT_OBJECT':
                field_infos = input_object_schemas[gql_type].input_fields
            else:
                print "Unknown type {} needed in {}".format(gql_type, gql_query.source_file_name)
                return

            for field_info in field_infos:
                if field_info.type not in gql_to_c_basic_types:
                    types_to_generate.append(field_info.type)
                struct_variables.append((gql_field_to_c_type(field_info), field_info.name))

            # gql_header_info.structs[gql_type] = struct_variables
            gql_header_info.structs.insert(0, (gql_type, struct_variables))

        index += 1

    # Payload struct
    # TODO: how to deal with payload types that are lists or optional on queries? Or if we query on multiple things in one file?
    if gql_query.is_mutation:
        payload_object = object_schemas[mutation_schemas[gql_query.query_function].payload.type]
    else:
        payload_object = object_schemas[query_schemas[gql_query.query_function].payload.type]

    payload_classes = gql_payloads_to_cpp(gql_query.payload, payload_object, "", True)

    if not payload_classes:
        print 'Error parsing payload in {}'.format(gql_query.query_name)
        return

    gql_header_info.structs.extend(payload_classes[0])
    for enum in payload_classes[1]:
        gql_header_info.enums[enum[0]] = enum[1]

    if payload_classes[2]:
        gql_header_info.union_schemas.extend(payload_classes[2])

    gql_header_info.structs.append((INPUT_PARAM_STRUCT_NAME, input_param_variables))

    return gql_header_info

def fragment_to_cpp_header(gql_fragment, namespaces, header_include_path):
    fragment_header_info = GQLFragmentHeader(gql_fragment, namespaces)

    # Header
    if gql_fragment.fragment_names:
        fragment_header_info.headers = [header_include_path + "{}{}".format(fragment, 'info.h') for fragment in gql_fragment.fragment_names]
    # TODO: should only add these when you need it instead of always
    fragment_header_info.headers.append('twitchsdk/core/optional.h')
    fragment_header_info.headers.append('twitchsdk/core/json/jsonserialization.h')

    fragment_object = object_schemas[gql_fragment.type]

    fragment_classes = gql_payloads_to_cpp(gql_fragment.payload, fragment_object)
    if not fragment_classes:
        print 'Error parsing payload in {}'.format(gql_fragment.name)
        return

    fragment_header_info.structs = fragment_classes[0]
    for enum in fragment_classes[1]:
        fragment_header_info.enums[enum[0]] = enum[1]

    return fragment_header_info

# C++ QUERY INFO FILE GENERATION FUNCTIONS
def back_search_for_typename(lines, position):
    while position >= 0:
        line = lines[position].strip()
        if line == '__typename':
            return True

        if line.startswith('}'):
            # we only care about the leading paramter
            return True

        if line.endswith('{'):
            return False

        position = position - 1

    return False

def generate_query_raw_string(source_graphql_files_dir, file_name, indentation, indentation_level):
    output = 'R"(\n'
    indentation_level += 1
    files = [file_name]
    index = 0

    while index < len(files):
        file_name = files[index]
        with open(os.path.join(source_graphql_files_dir, file_name)) as file:
            lines = file.readlines()

        for pos, line in enumerate(lines):
            line = line.strip()

            if line.startswith('}'):
                indentation_level -= 1

            if line.startswith('...'):
                words = line.split()
                if words[1] != 'on' and words[1] not in files:
                    files.append('{}.graphql'.format(words[1]))

                # We want to ensure the unions of __typename to select on
                if words[1] == 'on' and not back_search_for_typename(lines, pos - 1):
                    output += indentation * indentation_level + '__typename\n'

            if line:
                output += indentation * indentation_level + '{}\n'.format(line)

            if line.endswith('{'):
                indentation_level += 1

        index += 1
        if index < len(files):
            output += '\n'

    indentation_level -= 1
    output += indentation * indentation_level + ')"'

    return output

def generate_query_info_header(gql_header_info, indentation, common_output, source_graphql_files_dir):
    output = common_output
    file_path = os.path.join(source_graphql_files_dir, gql_header_info.query_info.source_file_name)
    output += '// Output determined from {}\n\n'.format(file_path[file_path.find('modules/'):])

    if gql_header_info.query_info.gen_full:
        output += '#pragma once\n\n'

        for header in gql_header_info.headers:
            output += '#include "{}"\n'.format(header)
        output += '\n'

    indentation_level = 0
    for namespace in gql_header_info.namespaces:
        output += indentation * indentation_level + 'namespace {}\n'.format(namespace)
        output += indentation * indentation_level + '{\n'
        indentation_level += 1

    if not gql_header_info.query_info.gen_full:
        # Only generating the function to grab the query string
        output += indentation * indentation_level + 'constexpr const char* Get'
        output += gql_header_info.query_info.query_name

        if gql_header_info.query_info.is_mutation:
            output += 'Mutation'
        else:
            output += 'Query'

        output += '()\n'
        output += indentation * indentation_level + '{\n'
        indentation_level += 1

        raw_string = generate_query_raw_string(source_graphql_files_dir, gql_header_info.query_info.source_file_name, indentation, indentation_level)
        if raw_string is not None:
            output += indentation * indentation_level + 'return {};\n'.format(raw_string)
        else:
            print 'Failed to generate the raw string get function for {}'.format(file_path)
            return

    else:
        output += indentation * indentation_level + 'namespace json\n'
        output += indentation * indentation_level + '{\n'
        indentation_level += 1

        for struct in gql_header_info.structs:
            output += indentation * indentation_level + 'struct {}{};\n'.format(gql_header_info.query_info.query_name, struct[0])
        for enum in gql_header_info.enums:
            output += indentation * indentation_level + 'struct {}{};\n'.format(gql_header_info.query_info.query_name, enum)
        for union_schema in gql_header_info.union_schemas:
            output += indentation * indentation_level + 'struct {}{};\n'.format(gql_header_info.query_info.query_name, union_schema[0])
        indentation_level -= 1
        output += indentation * indentation_level + '}\n\n'

        # Generating all the structs and serialization
        output += indentation * indentation_level + 'struct {}QueryInfo\n'.format(gql_header_info.query_info.query_name)
        output += indentation * indentation_level + '{\n'
        indentation_level += 1

        for enum_name, enum_values in gql_header_info.enums.iteritems():
            output += indentation * indentation_level + 'enum class {}\n'.format(enum_name)
            output += indentation * indentation_level + '{\n'
            indentation_level += 1
            for value in enum_values:
                output += indentation * indentation_level + '{},\n'.format(value)
            indentation_level -= 1
            output += indentation * indentation_level + '};\n\n'

        for struct in gql_header_info.structs:
            struct_name = struct[0]
            struct_variables = struct[1]

            output += indentation * indentation_level + 'struct {}\n'.format(struct_name)
            output += indentation * indentation_level + '{\n'
            indentation_level += 1
            for variable in struct_variables:
                output += indentation * indentation_level + '{} {};\n'.format(variable[0], variable[1])
            indentation_level -= 1

            output += indentation * indentation_level + '};\n\n'

        output += indentation * indentation_level + 'static constexpr const char* kTaskName = "GraphQLTask: {}";\n'.format(gql_header_info.query_info.query_name)

        raw_string = generate_query_raw_string(source_graphql_files_dir, gql_header_info.query_info.source_file_name, indentation, indentation_level)
        if raw_string is not None:
            output += indentation * indentation_level + 'static constexpr auto kQuery = {};\n'.format(raw_string)
        else:
            print 'Failed to generate the raw string get function for {}'.format(file_path)
            return

        # close the main struct
        indentation_level -= 1
        output += indentation * indentation_level + '};\n'

    while indentation_level > 0:
        indentation_level -= 1
        output += indentation * indentation_level + '}\n'
    output += '\n'

    if gql_header_info.query_info.gen_full:
        namespace_scope = '::'.join(gql_header_info.namespaces)

        for struct in gql_header_info.structs:
            struct_name = struct[0]
            struct_variables = struct[1]
            output += indentation * indentation_level + '\nstruct {}::json::{}{}\n'.format(namespace_scope, gql_header_info.query_info.query_name, struct_name)
            output += indentation * indentation_level + '{\n'
            indentation_level += 1

            variable_name = lowercase(struct_name)
            output += indentation * indentation_level + 'template <typename GraphQLType>\n'
            output += indentation * indentation_level + 'static auto BindFields(GraphQLType& {})\n'.format(variable_name)
            output += indentation * indentation_level + '{\n'
            indentation_level += 1

            output += indentation * indentation_level + 'using namespace ttv::json;\n\n'
            output += indentation * indentation_level + 'return std::make_tuple\n'
            output += indentation * indentation_level + '(\n'
            indentation_level += 1
            didOutput = False

            for variable in struct_variables:
                if struct_name != INPUT_PARAM_STRUCT_NAME or variable[1] != AUTH_TOKEN_FIELD_NAME:
                    didOutput = True
                    if variable[0].startswith('ttv::Variant'):
                        make_type = 'make_variant_selector'
                        # no key for variant selector
                        key_path = ''
                    else:
                        make_type = 'make_field'

                        if struct_name == PAYLOAD_STRUCT_NAME:
                            key_path = 'MakeKeyPath("{}", "{}"), '.format(gql_header_info.query_info.query_function, variable[1])
                        else:
                            key_path = '"{}", '.format(variable[1])

                    if variable[0].startswith('ttv::Optional'):
                        required_type = 'OptionalField'
                    else:
                        required_type = 'RequiredField'

                    member_var = '{}.{}'.format(variable_name, variable[1])

                    output += indentation * indentation_level + '{}<{}>({}{}),\n'.format(make_type, required_type, key_path, member_var)

            if didOutput is True:
                output = output[:-2] + '\n'

            indentation_level -= 1
            output += indentation * indentation_level + ');\n'

            if didOutput is False:
                # Failing to do this will result in an 'unused parameter' warning for some compilers
                output += indentation * indentation_level + '(void)'+ variable_name + ';\n'

            indentation_level -= 1
            output += indentation * indentation_level + '}\n'
            indentation_level -= 1
            output += indentation * indentation_level + '};\n\n'

        for enum_name, enum_values in gql_header_info.enums.iteritems():
            output += indentation * indentation_level + '\nstruct {}::json::{}{}\n'.format(namespace_scope, gql_header_info.query_info.query_name, enum_name)
            output += indentation * indentation_level + '{\n'
            indentation_level += 1

            variable_name = lowercase(enum_name)
            output += indentation * indentation_level + 'static auto EnumMap()\n'
            output += indentation * indentation_level + '{\n'
            indentation_level += 1

            output += indentation * indentation_level + 'using namespace ttv::json;\n\n'
            output += indentation * indentation_level + 'return std::make_tuple\n'
            output += indentation * indentation_level + '(\n'
            indentation_level += 1

            for enum_value in enum_values:
                output += indentation * indentation_level + 'make_enum_mapping_case_insensitive("{}", {}QueryInfo::{}::{}),\n'.format(enum_value, gql_header_info.query_info.query_name, enum_name, enum_value)

            output = output[:-2] + '\n'
            indentation_level -= 1

            output += indentation * indentation_level + ');\n'
            indentation_level -= 1
            output += indentation * indentation_level + '}\n'
            indentation_level -= 1
            output += indentation * indentation_level + '};\n\n'

        for schema_name, selection_types in gql_header_info.union_schemas:
            qualified_selection_types = ['{}QueryInfo::{}'.format(gql_header_info.query_info.query_name, x) for x in selection_types ]

            output += indentation * indentation_level + '\nstruct {}::json::{}{}\n'.format(namespace_scope, gql_header_info.query_info.query_name, schema_name)
            output += indentation * indentation_level + '{\n'
            indentation_level += 1

            output += indentation * indentation_level + 'static auto SelectSchema(const ttv::json::Value& value, ttv::Variant<{}>& output)\n'.format(', '.join(qualified_selection_types))
            output += indentation * indentation_level + '{\n'
            indentation_level += 1

            output += indentation * indentation_level +'auto& selectorKeyValue  = value["__typename"];\n'
            output += indentation * indentation_level +'if(selectorKeyValue.isNull() || !selectorKeyValue.isString()) {\n'
            indentation_level += 1
            output += indentation * indentation_level +'return false;\n'

            indentation_level -= 1
            output += indentation * indentation_level + '}\n\n'

            output += indentation * indentation_level + 'auto selectorKey = selectorKeyValue.asString();\n\n'

            for selection_type in selection_types:
                output += indentation * indentation_level + 'if(selectorKey == "{}") {{\n'.format(selection_type)
                indentation_level += 1
                output += indentation * indentation_level + 'output = {}QueryInfo::{}();\n'.format(gql_header_info.query_info.query_name, selection_type)
                output += indentation * indentation_level + 'auto& data = output.As<{}QueryInfo::{}>();\n'.format(gql_header_info.query_info.query_name, selection_type)
                output += indentation * indentation_level + 'return ttv::json::ToObject<{}QueryInfo::{}>(value, data);\n'.format(gql_header_info.query_info.query_name, selection_type)  
                indentation_level -= 1
                output += indentation * indentation_level + '}\n\n'

            output += indentation * indentation_level + 'return false;\n'
            indentation_level -= 1
            output += indentation * indentation_level + '}\n'
            indentation_level -= 1
            output += indentation * indentation_level + '};\n\n'

        for struct in gql_header_info.structs:
            struct_name = struct[0]
            output += '\ntemplate<>\nstruct ttv::json::DefaultSchemaProvider<{}::{}QueryInfo::{}>\n{{\n'.format(namespace_scope, gql_header_info.query_info.query_name, struct[0])
            output += indentation + 'using Type = ObjectSchema<{}::json::{}{}>;\n}};\n\n'.format(namespace_scope, gql_header_info.query_info.query_name, struct_name)

        for enum_name in gql_header_info.enums:
            output += '\ntemplate<>\nstruct ttv::json::DefaultSchemaProvider<{}::{}QueryInfo::{}>\n{{\n'.format(namespace_scope, gql_header_info.query_info.query_name, enum_name)
            output += indentation + 'using Type = EnumSchema<{}::json::{}{}>;\n}};\n\n'.format(namespace_scope, gql_header_info.query_info.query_name, enum_name)

        for schema_name, selection_types in gql_header_info.union_schemas:
            full_selection_types = ['{}::{}QueryInfo::{}'.format(namespace_scope, gql_header_info.query_info.query_name, x) for x in selection_types ]
            output += '\ntemplate<>\nstruct ttv::json::DefaultSchemaProvider<ttv::Variant<{}>>\n{{\n'.format(', '.join(full_selection_types))
            output += indentation + 'using Type = {}::json::{}{};\n}};\n\n'.format(namespace_scope, gql_header_info.query_info.query_name, schema_name)

    return output


# TODO: move the shared code with generate_query_info_header to a function
def generate_fragment_info(fragment_header_info, indentation, common_output, source_graphql_files_dir):
    output = common_output
    file_path = os.path.join(source_graphql_files_dir, fragment_header_info.fragment_info.source_file_name)
    output += '// Output determined from {}\n\n'.format(file_path[file_path.find('modules/'):])

    output += '#pragma once\n\n'

    for header in fragment_header_info.headers:
        output += '#include "{}"\n'.format(header)
    output += '\n'

    indentation_level = 0
    for namespace in fragment_header_info.namespaces:
        output += indentation * indentation_level + 'namespace {}\n'.format(namespace)
        output += indentation * indentation_level + '{\n'
        indentation_level += 1

    output += indentation * indentation_level + 'namespace json\n'
    output += indentation * indentation_level + '{\n'
    indentation_level += 1

    for struct in fragment_header_info.structs:
        output += indentation * indentation_level + 'struct {}{};\n'.format(fragment_header_info.fragment_info.name, struct[0])
    for enum in fragment_header_info.enums:
        output += indentation * indentation_level + 'struct {}{};\n'.format(fragment_header_info.fragment_info.name, enum)
    indentation_level -= 1
    output += indentation * indentation_level + '}\n\n'

    # Generating all the structs and serialization
    output += indentation * indentation_level + 'struct {}\n'.format(fragment_header_info.fragment_info.name)
    output += indentation * indentation_level + '{\n'
    indentation_level += 1

    # TODO: make a function for outputting enums and structs
    for enum_name, enum_values in fragment_header_info.enums.iteritems():
        output += indentation * indentation_level + 'enum class {}\n'.format(enum_name)
        output += indentation * indentation_level + '{\n'
        indentation_level += 1
        for value in enum_values:
            output += indentation * indentation_level + '{},\n'.format(value)
        indentation_level -= 1
        output += indentation * indentation_level + '};\n\n'

    for struct in fragment_header_info.structs:
        struct_name = struct[0]
        struct_variables = struct[1]

        if struct_name != fragment_header_info.fragment_info.type:
            output += indentation * indentation_level + 'struct {}\n'.format(struct_name)
            output += indentation * indentation_level + '{\n'
            indentation_level += 1
            for variable in struct_variables:
                output += indentation * indentation_level + '{} {};\n'.format(variable[0], variable[1])
            indentation_level -= 1

            output += indentation * indentation_level + '};\n\n'

    if fragment_header_info.structs[-1][0] != fragment_header_info.fragment_info.type:
        print 'WHY ARE THEY DIFFERENT'
        return

    for variable in fragment_header_info.structs[-1][1]:
        output += indentation * indentation_level + '{} {};\n'.format(variable[0], variable[1])

    # close the main struct
    indentation_level -= 1
    output += indentation * indentation_level + '};\n'

    while indentation_level > 0:
        indentation_level -= 1
        output += indentation * indentation_level + '}\n'
    output += '\n'\


    namespace_scope = '::'.join(fragment_header_info.namespaces)
    for struct in fragment_header_info.structs:
        struct_name = struct[0]
        struct_variables = struct[1]

        if struct_name == fragment_header_info.fragment_info.type:
            output += indentation * indentation_level + 'struct {}::json::{}{}\n'.format(namespace_scope, fragment_header_info.fragment_info.name, struct_name)
        else:
            output += indentation * indentation_level + 'struct {}::json::{}{}\n'.format(namespace_scope, fragment_header_info.fragment_info.name, struct_name)
        output += indentation * indentation_level + '{\n'
        indentation_level += 1

        variable_name = lowercase(struct_name)
        output += indentation * indentation_level + 'template <typename GraphQLType>\n'
        output += indentation * indentation_level + 'static auto BindFields(GraphQLType& {})\n'.format(variable_name)
        output += indentation * indentation_level + '{\n'
        indentation_level += 1

        output += indentation * indentation_level + 'using namespace ttv::json;\n\n'
        output += indentation * indentation_level + 'return std::make_tuple\n'
        output += indentation * indentation_level + '(\n'
        indentation_level += 1

        for variable in struct_variables:
            output += indentation * indentation_level + 'make_field<'
            # TODO: a better way to check if the field is optional
            if variable[0].startswith('ttv::Optional'):
                output += 'OptionalField'
            else:
                output += 'RequiredField'
            output += '>("{}", {}.{}),\n'.format(variable[1], variable_name, variable[1])

        output = output[:-2] + '\n'
        indentation_level -= 1

        output += indentation * indentation_level + ');\n'
        indentation_level -= 1
        output += indentation * indentation_level + '}\n'
        indentation_level -= 1
        output += indentation * indentation_level + '};\n\n'

    for enum_name, enum_values in fragment_header_info.enums.iteritems():
        output += indentation * indentation_level + '\nstruct {}::json::{}{}\n'.format(namespace_scope, fragment_header_info.fragment_info.name, enum_name)
        output += indentation * indentation_level + '{\n'
        indentation_level += 1

        variable_name = lowercase(enum_name)
        output += indentation * indentation_level + 'static auto EnumMap()\n'
        output += indentation * indentation_level + '{\n'
        indentation_level += 1

        output += indentation * indentation_level + 'using namespace ttv::json;\n\n'
        output += indentation * indentation_level + 'return std::make_tuple\n'
        output += indentation * indentation_level + '(\n'
        indentation_level += 1

        for enum_value in enum_values:
            output += indentation * indentation_level + 'make_enum_mapping_case_insensitive("{}", {}::{}::{}),\n'.format(enum_value, fragment_header_info.fragment_info.name, enum_name, enum_value)

        output = output[:-2] + '\n'
        indentation_level -= 1

        output += indentation * indentation_level + ');\n'
        indentation_level -= 1
        output += indentation * indentation_level + '}\n'
        indentation_level -= 1
        output += indentation * indentation_level + '};\n\n'

    for struct in fragment_header_info.structs:
        struct_name = struct[0]
        output += '\ntemplate<>\nstruct ttv::json::DefaultSchemaProvider<{}::{}::{}>\n{{\n'.format(namespace_scope, fragment_header_info.fragment_info.name, struct[0])
        output += indentation + 'using Type = ObjectSchema<{}::json::{}{}>;\n}};\n\n'.format(namespace_scope, fragment_header_info.fragment_info.name, struct_name)

    for enum_name in fragment_header_info.enums:
        output += '\ntemplate<>\nstruct ttv::json::DefaultSchemaProvider<{}::{}::{}>\n{{\n'.format(namespace_scope, fragment_header_info.fragment_info.name, enum_name)
        output += indentation + 'using Type = EnumSchema<{}::json::{}{}>;\n}};\n\n'.format(namespace_scope, fragment_header_info.fragment_info.name, enum_name)

    return output

def generate_graphql_query_files(schema_file_path, source_graphql_files_dir, files_to_generate, copyright_file_path, header_include_path, output_dir, output_base_file_name, namespaces, indentation):
    
    fragment_names = []
    fragment_files = []
    gql_files = []
    gql_headers = []

    # reset global state
    global input_object_schemas
    input_object_schemas = {}
    global object_schemas
    object_schemas = {}
    global enum_schemas
    enum_schemas = {}
    global union_schemas
    union_schemas = {}
    global mutation_schemas
    mutation_schemas = {}
    global query_schemas
    query_schemas = {}
    global gql_kind_map
    gql_kind_map = {}
    global type_name_map
    type_name_map = dict()
    global type_name_suffix_map
    type_name_suffix_map = dict()
    global recursive_name_stack
    recursive_name_stack = []

    parse_schema(schema_file_path)

    if header_include_path[-1] != '/':
        header_include_path += '/'

    for file_name in os.listdir(source_graphql_files_dir):
        if file_name in files_to_generate:
            print 'Parsing ' + file_name + '...'
            gql_file_info = parse_graphql_file(source_graphql_files_dir, file_name)
            if gql_file_info:
                gql_files.append(gql_file_info)
                for fragment_name in gql_file_info.fragment_names:
                    if fragment_name not in fragment_names:
                        fragment_names.append(fragment_name)
            else:
                print "Failed to generate file info for {}".format(file_name)

    for fragment in fragment_names:
        # TODO better way to check if the corresponding fragment file exists
        if '{}.graphql'.format(fragment.lower()) not in os.listdir(source_graphql_files_dir):
            print 'Cannot find fragment file for {}'.format(fragment)
            return

        fragment_info = parse_graphql_fragment_file(source_graphql_files_dir, '{}.graphql'.format(fragment))
        if fragment_info:
            fragment_files.append(fragment_info)

    fragment_files = [fragment_to_cpp_header(fragment, namespaces, header_include_path) for fragment in fragment_files]
    gql_headers = [gql_to_cpp_header(query, namespaces, header_include_path) for query in gql_files]

    with open(copyright_file_path) as file:
        copyright_block = file.read()

    common_output = '{}\n'.format(copyright_block)
    common_output += '// DO NOT EDIT THIS FILE - it is machine generated\n'

    for gql_header in gql_headers:
        file_output = generate_query_info_header(gql_header, indentation, common_output, source_graphql_files_dir)

        if output_dir:
            source_file_name_stripped = gql_header.query_info.source_file_name.split('.')[0]
            output_file_name = source_file_name_stripped + 'info.h'
            file_output_path = os.path.join(output_dir, output_file_name)
            with open(file_output_path, 'w') as fileout:
                fileout.write(file_output)

            print 'Generated ' + output_file_name
        else:
            print file_output


    for fragment_header in fragment_files:
        file_output = generate_fragment_info(fragment_header, indentation, common_output, source_graphql_files_dir)

        if output_dir:
            source_file_name_stripped = fragment_header.fragment_info.source_file_name.split('.')[0]
            output_file_name = source_file_name_stripped + 'info.h'
            file_output_path = os.path.join(output_dir, output_file_name)
            with open(file_output_path, 'w') as fileout:
                fileout.write(file_output)

            print 'Generated ' + output_file_name
        else:
            print file_output
