# Building the Twitch SDK

## Overview
The ```build``` directory contains custom Twitch SDK build generation scripts are used to allow developers to conveniently generate platform-specific projects that contain exactly the set of features that they need for their product.  This README describes how to build the Twitch SDK library.

For information on building samples see [Samples](Samples.md).

For information on building tests see [Tests](Tests.md).

## Environment Setup
In order to generate project files the following are required:
* Install Python 2.7.x
* Install [pip](https://pip.pypa.io/en/stable/installing/)
* ```pip install gitpython```, prefix with ```sudo -H``` on OSX
* Install CMake 3.4.3 or higher (for Windows, Mac and iOS project generation)
  * On OSX, you'll need to "Allow apps from anywhere" in Security and Privacy options, then add cmake to your path via
  ```sudo /Applications/CMake.app/Contents/bin/cmake-gui --install=/usr/local/bin```
* On Windows, install openssl by running the appropriate script in /modules/core/dependencies/openssl
  * Installing openssl will require [Strawberry Perl](http://strawberryperl.com)


## Project Generation
The projects are generated using custom Python scripts.  Projects shuold be regenerated any time you move to a new commit in the Twitch SDK repositories.

At a high level the generation process for the Twitch SDK library works as follows:
* The search paths will be searched to find fragments which are pertinent based on the given options.
* A static library project file will be generated for each module (core, chat, social, etc).
* A glue library project is generated that links against the module static libraries that were generated.

### Scripts
```generate_twitchsdk.py``` is used to generate the platform-specific Twitch SDK library projects.  It takes a number of options and switches to tailor the resulting projects to the platform and feature-set required by your application.

```generate_test.py``` is used to generate the platform-specific test projects.  See [Tests](Tests.md) for more information on building tests.

```generate_sample.py``` is used to generate the platform-specific sample projects.  See [Samples](Samples.md) for more information on building samples.


### Command-Line Usage

|   Switch       |           Values           | Allows multiple | Optional | Description |
|----------------|----------------------------|-----------------|----------|-------------|
| --module       |   \<module\>                 |         Y       |    Y     |             |
| --platform     |   \<platform\>               |         N       |    N     |             |
| --target       |   \<target\>                 |         N       |    N     | Specified if the platform has different possible targets (such as OS X or iOS).  The platform fragment documentation should specify possible values. |
| --compiler     |   \<compiler\>               |         N       |    N     |             |
| --language     |   \<language\>               |         N       |    Y     | The optional language bindings to include. |
| --feature      |   \<feature-name\>           |         Y       |    Y     | The optional additional features to include.  Looks for fragments with the given name in each module. |
| --dynamic      |   \<dynamic\>                |         N       |    Y     | Specifies a dynamic library.  This is the default. |
| --static       |   \<static\>                 |         N       |    Y     | Specifies a static library. |
| --output-dir   |   \<path\>                   |         N       |    N     | The root directory for project generation. |
| --settings     |   \<switch\> [\<switch\> ...]  |         N       |    Y     | Specifies switches that have meaning depending on the platform, language bindings and modules.  See module and fragment documentation for possible values. |


## Supported Platforms
Out of the box the following platforms are supported:
* Windows
* OSX
* iOS
* Android
Support more platforms can be easily added.  See _Adding support for new platforms_.

### Windows
* Uses CMake to write Visual Studio project files.

### Mac OS X and iOS (Darwin)
* Uses CMake to generate Xcode projects.

### Android
* Uses custom generator to write out NDK makefiles.


## Project Generation Examples
These examples assume you have the following command line variables set:
* ```OUTPUT_DIR```: The location you want to output the generated project files.

### Windows
```
python generate_twitchsdk.py --dynamic --module=core --module=chat --module=social --platform=win32 --compiler=vs2012 --arch=x86 --output-dir=%OUTPUT_DIR% --project-name=mylibrary -settings TTV_USEALL_WIN32_IMPLEMENTATIONS TTV_USE_STD_OPENSSL_SOCKET TTV_USE_STD_WEBSOCKET TTV_USE_STD_THREAD_API
```

### Mac OS X
```
python generate_twitchsdk.py --dynamic --module=core --module=chat --module=social --platform=darwin --compiler=xcode --arch=x64 --output-dir=$OUTPUT_DIR --project-name=mylibrary -settings TTV_USEALL_DARWIN_IMPLEMENTATIONS TTV_USE_STD_MUTEX TTV_USE_STD_SYSTEM_CLOCK TTV_USE_STD_THREAD TTV_USE_STD_THREAD_SYNC TTV_USE_STD_TRACER TTV_USE_STD_WEBSOCKET TTV_USE_STD_THREAD_API
```

### iOS
#### Make an Xcode project via CMake
```
python generate_twitchsdk.py --static --module=core --module=chat --module=social --platform=darwin --target=ios --compiler=xcode --arch=x64 --output-dir=$OUTPUT_DIR --project-name=mylibrary -settings TTV_USEALL_DARWIN_IMPLEMENTATIONS TTV_USE_STD_MUTEX TTV_USE_STD_SYSTEM_CLOCK TTV_USE_STD_THREAD TTV_USE_STD_THREAD_SYNC TTV_USE_STD_TRACER TTV_USE_STD_WEBSOCKET TTV_USE_STD_THREAD_API
```
#### Generate Cocoapods
```
python generate_twitchsdk.py --static --module=core --module=chat --module=social --module=broadcast --platform=darwin --compiler=xcode --target=ios --generator=cocoapods --arch=x64 --output-dir=$OUTPUT_DIR --project-name=twitchsdk -settings TTV_USEALL_DARWIN_IMPLEMENTATIONS TTV_USE_STD_WEBSOCKET TTV_USE_STD_THREAD_API TTV_USE_STD_MUTEX TTV_USE_STD_SYSTEM_CLOCK TTV_USE_STD_THREAD TTV_USE_STD_THREAD_SYNC TTV_USE_STD_TRACER
```

### Android
#### Generate a CMakeLists.txt file for Gradle
```
python generate_twitchsdk.py --dynamic --cmake-no-generate --force-monolithic-project=true --language=java --module=core --module=chat --module=social --module=broadcast --compiler=clang --platform=android --arch=x64 --output-dir=$OUTPUT_PATH --project-name=twitchsdk -settings TTV_USEALL_ANDROID_IMPLEMENTATIONS TTV_USE_STD_THREAD_API TTV_USE_STD_MUTEX TTV_USE_STD_SYSTEM_CLOCK TTV_USE_STD_THREAD TTV_USE_STD_THREAD_SYNC TTV_USE_ANDROID_TRACER TTV_USE_STD_RAW_SOCKET
```
#### Android Studio Steps
After installing Android Studio, there are a few extra steps required for setup. Through the Android Studio SDK Manager, download NDK, LLDB, and CMake. In the project, create a twitchsdk module. The java binding files can be added directly; however, for the cpp source files to be added, a few extra lines need to be added to the build.gradle file:
```
android {
 defaultConfig {
  externalNativeBuild{
   cmake {
    cppFlags "-std=c++14 -frtti -fexceptions"
    arguments "-DANDROID_TOOLCHAIN=clang", "-DANDROID_STL=c++_shared"
    abiFilters 'arm64-v8a', 'armeabi-v7a', 'x86', 'x86_64'
   }
  }
 }
 externalNativeBuild {
  cmake{
   path "folder_path_to/CMakeLists.txt"
  }
 }
}
```
After a gradle sync / build, gradle will add the cpp files to the twitchsdk module and use the CMakeLists.txt file to build the Twitch SDK.

More comprehensive instructions can be found on Android's [official documentation](https://developer.android.com/studio/projects/add-native-code.html) on adding C++ code to Android projects. Steps 1 & 2 (Creating new native source files and configuring CMake) have already been done by the SDK, so after downloading the necessary tools you can go straight to [configuring gradle](https://developer.android.com/studio/projects/gradle-external-native-builds.html) as mentioned above.
