# Coding Standards
Here are some of the coding standards to follow when performing development on the SDK code itself.

## General
* We currently use tabs instead of spaces.  We'd like to move to spaces and will do this at some time when there are no large outstanding branches open.
* Curly braces go on their own lines.
    ```
    if (x > 0)
    {
        ...
    }
    ```

* Naming conventions
    * Class, function and method names start with a capital letter.
    * Member variables of classes start with 'm' and are camel case.
    * Local variables and function parameters are camel case.
    * Interface names start with 'I'.
    ```
    class MyClass
    {
    public:
        void SomeMethod(int blah);

    private:
        int mSomeMember;
    };
    ```

* The names for .h and .cpp files for C++ code is generally all lower case.  This helps avoid issues with inclusion of files.  This rule does not apply to files for other languages or files that have been generated.

* Simple data structures passed to the client should be packed as POD into structs.  This keeps the language binding interface as simple as possible.
    ```
    struct TTV_SomeData
    {
        const utf8char* name;
        TTV_SomeEnum blah;

        const TTV_SomethingElse* interestingArray;
        uint32_t interestingArrayLength;
    };
    ```

* There are naming conventions to indicate how a member is to be used.  This is done partly for self-documentation and also to simplify the language binding generation.
* Members that use enumerations as bitfields must be named '*Bitfield'.  If the member represents just a single value of the bitfield it can be named whatever you want.
* Members that represent an array must be named '*Array' and also be accompanied by another member named '*ArrayLength' which specifies the length of the array.
    ```
    enum TTV_SomeEnum
    {
        TTV_SOMEENUM_A = 1 << 0,
        TTV_SOMEENUM_B = 1 << 1,
        TTV_SOMEENUM_C = 1 << 2,
        TTV_SOMEENUM_D = 1 << 3
    };

    struct TTV_SomethingElse
    {
        int someValue;
    };

    struct TTV_SomeData
    {
        TTV_SomeEnum someValue;                      // Indicates only one value will be set here.
        TTV_SomeEnum someEnumBitfield;               // Indicates that the value is a bitfield of TTV_SomeEnum values.

        const TTV_SomethingElse* interestingArray;   // Indicates that this is an array and not just a pointer to a single value.
        uint32_t interestingArrayLength;             // Accompanies a '*Array' member and specifies the number of elements in the array.
    };
    ```

* Don't use static variables in .cpp files.  Use anonymous namespaces.  This confines variable scope to the file in a clean way.
    ```
    namespace
    {
        int x = 0;
    }
    ```

* Don't use raw pointers unless you have a really good reason.  Use std::shared_ptr and std::unique_ptr.  However, use std::enable_shared_from_this sparingly since it usually implies a design issue.
* Use const references in function definitions to avoid expensive copies of non-primitive types.
    ```
    void SomeFunction(int a, float b, const std::string& avoidCopyingMe);
    ```

## Public Interface
* All methods on public classes return an error code as the return value.
* If you want to return data from the function use a const reference parameter.
    ```
    TTV_ErrorCode GetEmoticonSet(EmoticonSetId emoticonSetId, const TTV_ChatEmoticonSet*& result);
    ```
* Public types are fully documented in the code.

## Code Quality
* We enable warnings as errors as much as possible.  All warnings on all platforms are generally considered bugs.  If there is a good reason to do so a warning may be disabled at the file or project level.  Disabling at the project level requires configuring the project generation scripts.


## Cross-Platform Concerns
* Code that is platform-specific should be put in its own file in the source fragment for the language.  Do not use preprocessor definition to conditionally compile platform-specific code.
