# Source Code Documentation Standards
We will be using [Doxygen](http://www.stack.nl/~dimitri/doxygen/) to generate our source documentation.  It supports a mixture of Javadoc and Markdown which is what we'll be using.

See `/modules/core/core_common/include/socket.h` for a good example of complete documentation.

## C++

### Style

#### Functions and Class Methods
Here is an example of the markup to be used when documenting functions and methods.
```
/**
 * Attempts to create an IWebSocket instance that supports the given URI.
 *
 * When creating a new IWebSocket the registered factories will be visited in order of newest to oldest
 * registration until one is found that supports the requested protocol.
 *
 * @see RegisterWebSocketFactory
 * @param[in] uri The full uri to create the socket for.
 * @param[out] result The resulting IWebSocket instance if it was able to be created.
 * @return
 *   - TTV_EC_SUCCESS: The result has been populated with an IWebSocket instance to use.
 *   - TTV_EC_NOT_INITIALIZED: The socket library has not yet been initialized.
 *   - TTV_EC_NO_FACTORIES_REGISTERED: There are no registed factories.
 *   - TTV_EC_UNIMPLEMENTED: There are factories registered but none of them support the protocol.
 */
TTV_ErrorCode CreateWebSocket(const std::string& uri, std::shared_ptr<IWebSocket>& result);
```

All possible error codes should be enumerated if known.  If not possible, explicitly state it.

#### Classes and Interfaces
Here is an example of the markup to be used when documenting classes and interfaces.  Interfaces are similar to classes but do not have any detail specific to implementation.
```
/**
 * This module provides the ability to broadcast to Twitch by accepting video and audio data from
 * a game/app, encoding it and sending it to the Twitch backend in realtime.
 *
 * The following properties must be set before calling Initialize().
 * - SetCoreApi()
 * - SetListener()
 *
 * In order to broadcast several services must be configured using the following methods.
 * - SetVideoCapturer(): Sets the component that will supply video frames for encoding.
 * - SetVideoEncoder(): Sets the H.264 video encoder.  This will be platform specific.
 * - SetAudioCapturer(): Sets the components that will supply audio for encoding.  This
 *   may be called multiple times, such as for microphone and system audio.
 * - SetAudioEncoder(): Sets the audio encoder.  This will be platform specific.
 *
 * The video parameters must also be set via one of the 2 following methods.
 * - SetVideoParams(): Sets parameters when you know your maximum bandwidth.
 * - SetAdvancedVideoParams(): Sets parameters when you don't care about bandwidth and want
 *   to ensure you have a specific resolution.  This is not recommended for live streams.
 *
 * Currently, only one user may be broadcasting at a time.  In order to specify which user will
 * broadcast SetActiveUser() must be used.  The active user must be set and the user logged in
 * before configuring any of the encoders, capturers and other settings.  If the active user is
 * changed or logged out then the configuration is lost.
 */
class ttv::broadcast::BroadcastAPI : public ttv::ModuleBase
{
    ...
    std::shared_ptr<CoreApiClient> mCoreApiClient; //!< The client instance registered with CoreAPI.
};
```

#### Enumerations
An example enumeration.
```
/**
 * The type of data in a frame.
 */
enum class MessageType
{
    /** No data. */
    None,
    /** A binary frame. */
    Binary,
    /** A text frame. */
    Text,
    /** An unknown/unhandled frame type. */
    Unknown
};
```

#### Source Comments
Comments in .cpp files should use the following styles.  All start with an upper case letter.

##### Single Line
`// This is some interesting code`

##### Multi-Line
```
// This is some extremely interesting stuff which
// has a few details to be noted.
```

#### TODOs
`// TODO: This really should be cleaned up`

#### Workarounds and Hacks
`// HACK: This needs to be done to work around this issue.  The issue is...`

### Public Headers
All headers which are are not under an `internal` directory tree are considered public since they can be included directly by the client.  These headers have a higher standard for documentation than internal headers.

These headers should be fully documented.  This means that all classes, methods, members, functions, enumerations, typedefs/usings and macros should be marked up.

### Internal Headers
With the exception of the `core` module, all modules have an `internal` directory which contains all the headers that clients should not be using.  If we were to prepare a binary release this directory would not be included.

These headers do not need to be fully marked up unless they contain elements that are non-trivial.

## Java

### Style

Java should follow the same style as C++ where it applies.  The difference is that there are only .java files and no .h/.cpp files.  Only `public` and `protected` members need to be marked up.
