# Language Bindings

The following language bindings are provided out of the box:
* Java
* C++/CX for UWP

# Java Bindings

The SDK provides bindings to allow it to be used from Java. This document describes the usage patterns utilized in the SDK for this layer.

### Java Development Kit
Java itself provides tools for interacting with the virtual machine from a C or C++ environment, which are the foundation of the SDK's Java layer. The JDK comes with a library called **Java Native Interface**, which is a C/C++ library for working with the Java environment.

The `JavaVM` class is a thread-safe class from JNI. It can be used to attach the virtual machine to the current thread, which produces a `JNIEnv` object.

`JNIEnv` is a class that allows you to create objects, call methods, set fields, and do memory management on the Java side of the fence from C++. It is a thread confined object. There are three different ways we get access to a `JNIEnv` object when we need it:

1. When Java calls into C++, it will pass a JNIEnv object as a parameter to the function itself.
2. For methods that are called in the main threading context of the SDK, we can often use a `JNIEnv` object that we have previously cached in a global variable.
3. When we have no other Java context or cached `JNIEnv` object, we can grab the global `JavaVM` object and produce a `JNIEnv` object from it.

Each of these cases will be described in more detail later in this document.

##### javac/javap
The JDK also provides some command line tools with some essential functionality. The `javac` is Java's compiler itself. It takes `.java` files (which contain Java source code) and produces binary `.class` files that are directly used by the virtual machine at runtime, and is used by IDEs like Eclipse, IntelliJ and Android Studio.

The `javap` tool is a disassembler. This tool takes the binary `.class` files and can output metadata about the Java classes inside. For classes that have methods marked "native", it can also create the corresponding C header declarations for the functions that the virtual machine will link against to provide the native implementation of these methods (for when Java needs to call into C++ code). Both of these features are used by our scripts, which is detailed later in this document.

### SDK Java Support Files

Inside each module directory of the sdk, there is a java fragment directory that contains the code that supports the java bindings. For example, the `Core` java fragment can be found in `modules/core/core_java` under the root of the SDK repository. They contain a few different key folders and files:

* `bindings` directory - Contains all of the `.java` files for the classes and interfaces used by clients
* `include` directory - Contains the headers for the C++ code that supports the bindings.
* `source` directory - Contains all the source files for the C++ code that supports the bindings.
* `tests` directory - Contains some rough test code (C++ and java) for exercising the Java bindings. The code in here is not official or complete, it's mostly just a scratch area to test new bindings or changes to the interface.
* `tools` directory - Contains some useful scripts for generating some of the Java bindings code.
* Each module has a java_utils file (`java_coreutil` for `Core`, `java_chatutil` for `Chat`, etc) that contains some manually written tools for use by the bindings code. These live in the include/source directories mentioned above.

### JavaClassInfo

`JavaClassInfo` is a class defined by the SDK that can contain all the metadata about a given Java class. The object itself holds a few key pieces of information:
* The `klass`, which is an opaque identifer for the Java class used in JNI code.
* A map of instance method names to method ids, for use when invoking instance methods on Java objects.
* A map of static method names to method ids, for use when invoking static methods on Java classes.
* A map of field names to field ids, for use when getting or setting the value of a field on a Java object.

`JavaClassInfo` objects are needed for any class that will be created, modified, or whose methods will be invoked from C++. Each class that will be used in this way needs a corresponding `GetJavaClassInfo_[JavaClassname]` function (i.e. `GetJavaClassInfo_ChatChannelInfo`), the implementation of which can be automatically generated with a script in the SDK in `modules/core/core_java/tools/generate_class_info_func.py`. The output of this should go into the `generated` directories inside the `include` and `source` directories to indicate that they are automatically generated.

There is also a helper script in the java fragment directory of some modules inside the tools subdirectory called `generate_class_info.py`, which relies on the `generate_class_info_func.py` script and helps automate its use. These scripts have some hard-coded values that may need to be modified to work on any particular configuration.

Finally, in the java_utils file for each module, there is a `LoadAllJavaClassInfo` function that calls the `GetJavaClassInfo_*` functions for each of the classes that have one. Running this function ensures that the C++ side can properly load the metadata about the class.

### C++ calling into Java
For situations in which the C++ code has to call into Java, we usually have a proxy object in C++ that holds a global Java reference to its corresponding Java object and invokes methods onto it when needed. Before any methods can be invoked on the java object, however, we must acquire a `JNIEnv` first, which can be done in two ways. If we are in the main execution context of the SDK, it means we should have a `JNIEnv` object already cached in the `gActiveJavaEnvironment` variable. This is common for listener objects and callbacks. If instead we are in a different execution context, such as a dedicated thread that is created by the SDK, we need to grab the `JavaVM` object and create a `JNIEnv` object for that. The `AutoJEnv` object in the SDK is a helper class that will automatically generate a `JNIEnv` from the global `JavaVM` object and dispose of it when it leaves scope.

### Java calling into C++
When Java needs an entrypoint for calling into C++, it can mark a method as `native` in its code. This indicates that the implementation of this method will be provided in C++, not in Java. The `modules/core/core_java/tools/java_tools.py` script helps us here, and has two jobs, depending on the command line options.

First, `java_tools.py` can be invoked with the `--generate-header` parameter, which will cause it to use the JDK-provided `javap` tool to produce a header file. This header file will contain the declarations of the C functions that the java virtual machine will attempt to dynamically link on for its native methods. Here is an example invocation:
```
java_tools.py --generate-header --class="tv.twitch.social.SocialAPI" --class-path=%SDK_JAVA_BIN% --output=%SDK_ROOT%/modules/social/social_java/include/twitchsdk/social/generated/java_socialapi.h
```
Where `SDK_JAVA_BIN` is the directory that contains the compiled .class files, and `SDK_ROOT` is the root of the SDK repo. Each of these functions need to be manually implemented to translate the calls into the C++ APIs.

The other use for `java_tools.py` is with the `--generate-exports` parameter, which generates an `exports.txt` file that is used by the build scripts to produce the correct `.exp` and `.def` files. This makes sure that the projects build and export the correct symbols for Java to link on. Here is an example of its use:
```
java_tools.py --generate-exports --source-dir="%SDK_ROOT%/modules/core/core_java/include/twitchsdk/core" --source-dir="%SDK_ROOT%/modules/core/core_java/include/twitchsdk/core/generated" --source-dir="%SDK_ROOT%/modules/core/core_java/tests/unit/include/twitchsdk/core" --output="%SDK_ROOT%/modules/core/core_java/source/exports.txt"
```
Where `SDK_ROOT` is the root of the SDK repo. Since this just produces a text file that is later consumed by the build script, its important to re-run the build scripts after generating the exports.

##### NativeProxy
For these objects that are called into, we have a class in Java called `NativeProxy` that is subclassed in order to provide API entry points. The constructor for `NativeProxy` calls a native method called `createNativeInstance`. The C++ implementation of this native method creates the analagous C++ object, stores it in a registry, and passes a pointer to it back to Java as a `long` which `NativeProxy` stores as an instance variable. There is also a `disposeNativeInstance` method that evicts the C++ object from the registry.

For each C++ method that this `NativeProxy` exposes, the SDK convention is to have two Java methods:
* The native method - marked private, native, in capitalized camel case, and takes the native object pointer as a parameter. For example:
```
private native ErrorCode SetUserThreadsListener(long nativeObjectPointer, int userId, IChatUserThreadsListener listener);
```
* The forwarding method - marked public, lower camel case. Forwards to the capitalized method with the native object pointer. For example:
```
public ErrorCode setUserThreadsListener(int userId, IChatUserThreadsListener listener)
{
    return SetUserThreadsListener(getNativeObjectPointer(), userId, listener);
}
```
On the C++ side, the equivalent C function prototype that is being linked would look something like this:
```
/*
* Class:     tv_twitch_chat_ChatAPI
* Method:    SetListener
* Signature: (JLtv/twitch/chat/IChatAPIListener;)Ltv/twitch/ErrorCode;
*/
JNIEXPORT jobject JNICALL Java_tv_twitch_chat_ChatAPI_SetListener
(JNIEnv *, jobject, jlong, jobject);
```
Java automatically passes in a `JNIEnv` object for use in this method. In methods that are meant to be invoked in the main threading context of the SDK, we want to cache the `JNIEnv` in the global variable for use in listeners and callbacks further down the stack (see the **C++ calling into Java** section above). To do this, we can use the `ScopedJavaEnvironmentCacher` object, which sets the global variable in its constructor and clears out the reference in its destructor when it leaves scope.

The C++ implementation of these methods typically does a `reinterpret_cast` on the native object pointer in order to call into the main C++ code of the SDK. There is often a macro for doing so, such as `GET_CHATAPI_PTR`.

### Marshalling objects
When C++ calls into Java or vice versa, we usually need to do some translation between the C++ datatypes and the Java data types. For scalars, they can simply be `static_cast` directly between each other, but for objects, we have a few patterns we use for translating between them.
* `GetJavaInstance_[JavaClassName]` - For getting a Java object from an equivalent C++ object
* `GetNativeInstance_[CppClassName]` - For getting a C++ object from an equivalient Java object

These functions are in the java_utils file for whichever module is pertinent to these objects.

##### Result container
Unlike C++, Java does not have a built-in way of returning objects by reference. The SDK provides a means to do this via the templated `ResultContainer` class. Instead of passing in a reference to be modified, the Java method uses a `ResultContainer` as a method parameter. The method can then set the `result` of the `ResultContainer` for the caller to use once the method returns, much the same way returning values by pointer or by reference works in C++.

##### Memory management
Java uses garbage collection for memory management under the hood. When coding in Java, this is all dealt with transparently, but in C++, we need to be more explicit when dealing with java objects.

Upon creation, Java objects have a single local reference associated with them. Java objects are also assumed to have a single local reference associated with them when being returned from a method. But there are often times when we need to make sure to release our local reference to the object because we will no longer be using it. This is common for objects we are passing as parameters into callbacks or listener interfaces, or for objects we are assigning to a `ResultContainer` object. For these scenarios, we have a class called `JavaLocalReferenceDeleter` that releases the the bound object in its destructor when it leaves scope. Typically, we use the `AUTO_DELETE_LOCAL_REF` and `AUTO_DELETE_LOCAL_REF_NO_DECLARE` macros to achieve this. These macros automatically create a `JavaLocalReferenceDeleter` as a shadow variable alongside the object itself. `AUTO_DELETE_LOCAL_REF` creates the Java object itself as part of the invocation, while `AUTO_DELETE_LOCAL_REF_NO_DECLARE` just creates a shadow deleter for an already-declared variable.

In some cases, we may need to keep an object around longer than our current scope. In these cases, we have a `JavaGlobalObjectReference` helper that creates a global reference to the object, and releases the global reference in its destructor. These references are commonly needed for listener objects and callback objects that may still need to exist outside the local scope of the function.
