#!/usr/bin/python
#
# This script performs standard builds of the SDK as triggered by Jenkins
#
# Example quick incremental commit build
#   build.py --build-type=commit --platform=win32
#
# Example full nightly build
#   build.py --build-type=nightly --platform=win32
#
#

# Import standard modules
import argparse
import imp
import os
import platform
import shutil
import subprocess
import sys
import tempfile
import time
import uuid
import git
import distutils
from distutils import dir_util

# Capture paths
jenkins_dir = os.path.realpath(os.path.dirname(__file__))
repo_root = os.path.realpath(os.path.join(jenkins_dir, '..'))
repo_build_dir = os.path.realpath(os.path.join(repo_root, 'build'))

# Allow twitchcore python scripts to be found
sys.path.append(repo_build_dir)

import build_tools
import build_types
import generate_twitchsdk
import generate_test
import generate_sample


class TwitchSdkGenerationParameters:

    def __init__(self):
        self.project_name = None
        self.configuration_output_root_dir = None
        self.build_configurations = None
        self.output_object_type = None
        self.platform = None
        self.target = None
        self.architecture = None
        self.compiler = None
        self.language = None
        self.modules = []
        self.features = []
        self.fragment_search_paths = []
        self.repo_dirs = []
        self.settings = {}


def deduce_common_parameter_combinations(output_object_types, build_configurations, architectures, platform, targets, compilers, modulesets, languages, featuresets, settings, repo_dirs, fragment_search_paths, output_root_dir, handler):

    if targets is None or len(targets) == 0:
        targets = [None]
    if languages is None or len(languages) == 0:
        languages = [None]
    if featuresets is None or len(featuresets) == 0:
        featuresets = [None]
    if settings is None or len(settings) == 0:
        settings = [{}]

    for output_object_type in output_object_types:
        for arch in architectures:
            for target in targets:
                for compiler in compilers:
                    for modules in modulesets:
                        if not isinstance(modules, list): modules = [modules]
                        for features in featuresets:
                            if not isinstance(features, list): features = [features]
                            for setting_map in settings:
                                for language in languages:

                                    generation_params = TwitchSdkGenerationParameters()

                                    generation_params.output_object_type = output_object_type
                                    generation_params.build_configurations = build_configurations
                                    generation_params.configuration_output_root_dir = os.path.join(output_root_dir, platform, compiler, arch, output_object_type)
                                    generation_params.architecture = arch
                                    generation_params.compiler = compiler
                                    generation_params.platform = platform
                                    generation_params.target = target
                                    generation_params.language = language
                                    generation_params.modules = modules
                                    generation_params.features = features
                                    generation_params.repo_dirs = repo_dirs
                                    generation_params.fragment_search_paths = fragment_search_paths
                                    generation_params.settings = setting_map

                                    handler(generation_params)


def get_win32_config(build_type):

    languages = ['java']
    features = [
        ['intel-video-encoder', 'lame-audio-encoder']
    ]
    settings = [
    {
        'TTV_USEALL_WIN32_IMPLEMENTATIONS': None,
        'TTV_USE_STD_OPENSSL_SOCKET': None,
        'TTV_USE_STD_WEBSOCKET': None,
        'TTV_USE_STD_THREAD_API': None,
        'TTV_USE_STD_BACKGROUND_EVENT_SCHEDULER': None
    }]

    class NightlyResult:
        def __init__(self):
            self.platform = 'win32'
            self.targets = []
            self.build_configurations = ['Debug', 'Release']
            self.architectures = ['x86', 'x64']
            self.compilers = ['vs2015', 'vs2017']
            self.output_object_types = ['static', 'dynamic']
            self.languages = languages
            self.features = features
            self.settings = settings

    class CommitResult:
        def __init__(self):
            self.platform = 'win32'
            self.targets = []
            self.build_configurations = ['Release']
            self.architectures = ['x64']
            self.compilers = ['vs2015']
            self.output_object_types = ['static']
            self.languages = languages
            self.features = features
            self.settings = settings

    if build_type == 'nightly':
        return NightlyResult()
    elif build_type == 'commit':
        return CommitResult()


def get_osx_config(build_type):

    class Result:
        def __init__(self):
            self.platform = 'darwin'
            self.targets = [
                'osx'
            ]
            self.architectures = [
                'x64'
            ]
            self.compilers = [
                'xcode'
            ]
            self.output_object_types = [
                'static'
            ]
            self.features = [
                [
                    'intel-video-encoder',
                    'lame-audio-encoder'
                ]
            ]
            self.languages = [
                None
            ]
            self.settings = [
            {
                'TTV_USEALL_DARWIN_IMPLEMENTATIONS': None,
                'TTV_USE_STD_MUTEX': None,
                'TTV_USE_STD_SYSTEM_CLOCK': None,
                'TTV_USE_STD_THREAD': None,
                'TTV_USE_STD_THREAD_SYNC': None,
                'TTV_USE_STD_TRACER': None,
                'TTV_USE_STD_WEBSOCKET': None,
                'TTV_USE_STD_THREAD_API': None
            }
        ]

    return Result()


def get_ios_config(build_type):

    class Result:
        def __init__(self):
            self.platform = 'darwin'
            self.targets = [
                'ios'
            ]
            self.architectures = [
                'x64'
            ]
            self.compilers = [
                'xcode'
            ]
            self.output_object_types = [
                'static'
            ]
            self.features = [
                [
                    'intel-video-encoder',
                    'lame-audio-encoder'
                ]
            ]
            self.languages = [
                None
            ]
            self.settings = [
            {
                'TTV_USEALL_DARWIN_IMPLEMENTATIONS': None,
                'TTV_USE_STD_MUTEX': None,
                'TTV_USE_STD_SYSTEM_CLOCK': None,
                'TTV_USE_STD_THREAD': None,
                'TTV_USE_STD_THREAD_SYNC': None,
                'TTV_USE_STD_TRACER': None,
                'TTV_USE_STD_WEBSOCKET': None,
                'TTV_USE_STD_THREAD_API': None
            }
        ]

    return Result()


def get_android_config(build_type):

    class Result:
        def __init__(self):
            self.platform = 'android'
            self.targets = [
            ]
            self.architectures = [
                'x64'
            ]
            self.compilers = [
            ]
            self.output_object_types = [
                'static'
            ]
            self.features = [
            ]
            self.languages = [
                'java'
            ]
            self.settings = [
            {
                'TTV_USEALL_ANDROID_IMPLEMENTATIONS': None,
                'TTV_USE_STD_MUTEX': None,
                'TTV_USE_STD_SYSTEM_CLOCK': None,
                'TTV_USE_STD_THREAD': None,
                'TTV_USE_STD_THREAD_SYNC': None,
                'TTV_USE_STD_TRACER': None,
                'TTV_USE_STD_WEBSOCKET': None,
                'TTV_USE_STD_THREAD_API': None
            }
        ]

    return Result()


def get_common_command_line_params(generation_params, project_dir):

    command_line = []

    command_line.append('--platform=' + generation_params.platform)
    command_line.append('--arch=' + generation_params.architecture)
    command_line.append('--compiler=' + generation_params.compiler)
    command_line.append('--output-dir=' + os.path.join(generation_params.configuration_output_root_dir, project_dir))
    command_line.append('--project-name=' + generation_params.project_name)

    if generation_params.target:
        command_line.append('--target=' + generation_params.target)

    if generation_params.language:
        command_line.append('--language=' + generation_params.language)

    for module in generation_params.modules:
        command_line.append('--module=' + module)

    for feature in generation_params.features:
        command_line.append('--feature=' + feature)

    for path in generation_params.repo_dirs:
        command_line.append('--repo-dir=' + path)

    for path in generation_params.fragment_search_paths:
        command_line.append('--fragments=' + path)

    if len(generation_params.settings) > 0:
        command_line.append('-settings')

        for key, value in generation_params.settings.items():
            if value:
                command_line.append(key + '=' + str(value))
            else:
                command_line.append(key)

    return command_line


def build_project(path, config):

    """
    Builds the specifed project which contains a CMakeLists.txt file.
    If the build fails an exception will be thrown.
    """

    print('Building project "' + path + '" with configuration ' + config + '...')

    cmake_path = build_tools.find_cmake()

    if not cmake_path:
        raise Exception('Could not find cmake')

    shell_args = [cmake_path]
    shell_args.extend(['--build', path])
    shell_args.extend(['--config', config])

    proc = subprocess.Popen(shell_args)
    proc.wait()

    if proc.returncode != 0:
        raise Exception('Project failed to build: ' + path)

    print('Done building project.')


def ensure_dir(directory):
    if not os.path.exists(directory):
        os.makedirs(directory)

def copy_dir(from_path, to_path):
    ensure_dir(from_path)
    ensure_dir(to_path)
    distutils.dir_util.copy_tree(from_path, to_path)
    print('Copied directory\n' + from_path + ' to\n' + to_path)
    

def last_commit_relative_dir(path):
    path = os.path.abspath(path)
    print(path)

    """Get the nearest git repo from the path, searching its parent directories"""
    try:
        g = git.Git(repo_root)
        last_commit_hash = g.execute('git log -n 1 --format="%h" -- ' + path)
        last_commit_hash.strip()

        if not last_commit_hash:
            return "unknown_hash"

        return last_commit_hash
    except:
        print("Could not find root: " + path)

    return "unknown_hash"

def build_core_dependencies(platform_config):

    """
    Builds core dependencies if appropriate for the platform.  Does not build if already built.
    """

    # OpenSSL
    if platform_config.platform == 'win32':

        openssl_root = os.path.join(repo_root, 'modules', 'core', 'dependencies', 'openssl')
        cache_lib_root = os.path.realpath('/dev/lib/sdk/')

        for compiler in platform_config.compilers:
            for arch in platform_config.architectures:

                build = True

                last_commit_hash = last_commit_relative_dir(openssl_root)
                print('OpenSSL commit hash: ' + last_commit_hash)

                # use cache if available
                lib_path = os.path.join(openssl_root, 'lib', compiler, arch)
                cache_openssl_lib_path = os.path.join(cache_lib_root, 'openssl', compiler, arch, last_commit_hash)

                include_path = os.path.join(openssl_root, 'include', compiler, arch)        
                cache_openssl_include_path = os.path.join(cache_lib_root, 'include', compiler, arch, last_commit_hash)
                
                print("Copying OpenSSL from cache")
                copy_dir(cache_openssl_lib_path, lib_path)
                copy_dir(cache_openssl_include_path, include_path)

                if not os.path.exists(os.path.join(lib_path, 'libcrypto.lib') or not os.path.exists(os.path.join(lib_path, 'libssl.lib'))):

                    print('Building OpenSSL ' + compiler + ' ' + arch + '...')

                    cmd_path = os.path.join(openssl_root, 'build_' + compiler + '_' + arch + '.cmd')

                    proc = subprocess.Popen([cmd_path], cwd=openssl_root)
                    proc.wait()

                    if proc.returncode != 0:
                        raise Exception('OpenSSL failed to build: ' + cmd_path)
                    else:
                        print('Copying OpenSSL to cache lib')
                        copy_dir(lib_path, cache_openssl_lib_path)
                        copy_dir(include_path, cache_openssl_include_path)
                else:
                    print('OpenSSL ' + compiler + ' ' + arch + ' already built')


def create_temp_dir():

    if platform.system() == 'Windows':
        tempfile.tempdir = 'c:/jenkins_sdk_temp'

        if not os.path.exists(tempfile.tempdir):
            os.makedirs(tempfile.tempdir)

    return tempfile.mkdtemp().replace('\\', '/')


def run_tests(executable_path, working_dir):

    print('Running tests: "' + executable_path + '"')

    shell_args = [executable_path]
    proc = subprocess.Popen(executable_path, cwd=working_dir)
    proc.wait()

    if proc.returncode != 0:
        raise Exception('Tests failed with error code: ' + str(proc.returncode))

    print('Done running tests.')


if __name__ == "__main__":

    parser = argparse.ArgumentParser(description='Jenkins tasks.')

    parser.add_argument(
        '--build-type',
        required=True,
        metavar='<build-type>',
        help='Specifies the type of build to perform.  Options are "commit", "nightly"'
    )

    parser.add_argument(
        '--platform',
        required=True,
        metavar='<platform>',
        help='Specifies the platform to build for.  This maps directly to the platform parameter used for project generation.'
    )

    parser.add_argument(
        '--target',
        required=False,
        metavar='<target>',
        help='Specifies the target to build for.  This maps directly to the target parameter used for project generation.'
    )

    parser.add_argument(
        '--output-dir',
        required=False,
        metavar='<output-dir>',
        help='The output directory to build into.  If not specified then a random one will be created.'
    )

    parser.add_argument(
        '--keep-output',
        required=False,
        action='store_true',
        help="Don't delete the generated output when done."
    )

    args = parser.parse_args(None)

    build_type = args.build_type
    if build_type not in [ 'commit', 'nightly' ]: raise Exception('Unknown build-type: ' + build_type)

    if args.output_dir:
        output_root_dir = args.output_dir
    else:
        output_root_dir = create_temp_dir()

    print('Generating to ' + output_root_dir);

    try:
        all_modules = [
            'ads',
            'broadcast',
            'chat',
            'experiment',
            'social',
            'tracking'
        ]

        modulesets = [
            all_modules
        ]

        # # Build each module on its own as well
        # for m in all_modules:
        #   modules.append([m])

        repo_dirs = []
        fragment_search_paths = []

        target_platform = args.platform
        target = args.target

        # Get the config for the active platform
        platform_config = None

        if target_platform == 'win32':
            platform_config = get_win32_config(build_type)
        elif target_platform == 'darwin':
            if target == 'ios':
                platform_config = get_ios_config(build_type)
            else:
                platform_config = get_osx_config(build_type)
        elif target_platform == 'android':
            platform_config = get_android_config(build_type)
        else:
            raise Exception('Unhandled platform: ' + platform_name)

        # Build dependencies
        build_core_dependencies(platform_config)

        def handler(generation_params):

            unit_test_project_name = 'test_unit'
            sample_cli_project_name = 'sample_cli'
            twitchsdk_path = os.path.join(generation_params.configuration_output_root_dir, 'twitchsdk')
            sample_cli_path = os.path.join(generation_params.configuration_output_root_dir, 'sample_cli')
            test_unit_path = os.path.join(generation_params.configuration_output_root_dir, unit_test_project_name)

            # Generate the SDK project
            print('Generating twitchsdk project "' + twitchsdk_path + '"')

            generation_params.project_name = 'twitchsdk'
            command_line = get_common_command_line_params(generation_params, 'twitchsdk')
            command_line.append('--' + generation_params.output_object_type)

            generate_options = generate_twitchsdk.TwitchSdkOptions()
            generate_twitchsdk.parse_command_line(command_line, generate_options)
            generate_twitchsdk.generate(generate_options)

            # Build the SDK
            # for config in generation_params.build_configurations:
            #     build_project(gtwitchsdk_path, config)

            # Generate the cli sample tests
            print('Generating cli sample project "' + sample_cli_path + '"')

            generation_params.project_name = sample_cli_project_name
            command_line = get_common_command_line_params(generation_params, sample_cli_project_name)
            command_line.append('--sample=cli')
            command_line.append('--twitchsdk-dir=' + twitchsdk_path)

            generate_options = generate_sample.SampleOptions()
            generate_sample.parse_command_line(command_line, generate_options)
            generate_sample.generate(generate_options)

            # Build cli sample
            for config in generation_params.build_configurations:
                build_project(sample_cli_path, config)

            # Generate the unit tests
            print('Generating unit test project "' + test_unit_path + '"')

            generation_params.project_name = unit_test_project_name
            command_line = get_common_command_line_params(generation_params, unit_test_project_name)
            command_line.append('--test=unit')
            command_line.append('--twitchsdk-dir=' + twitchsdk_path)

            generate_options = generate_test.TestOptions()
            generate_test.parse_command_line(command_line, generate_options)
            generate_test.generate(generate_options)

            # Build the unit tests
            for config in generation_params.build_configurations:
                build_project(test_unit_path, config)

                if platform.system() == 'Windows':
                    test_executable_path = os.path.join(test_unit_path, config, unit_test_project_name)

                # TODO: Handle non-Windows unit tests
                else:
                    raise Exception('Add support for running unit tests on non-Windows platforms')

                run_tests(test_executable_path, os.path.join(test_unit_path, 'data'))


        deduce_common_parameter_combinations(
            output_object_types=platform_config.output_object_types,
            build_configurations=platform_config.build_configurations,
            architectures=platform_config.architectures,
            platform=platform_config.platform,
            targets=platform_config.targets,
            compilers=platform_config.compilers,
            modulesets=modulesets,
            languages=platform_config.languages,
            featuresets=platform_config.features,
            settings=platform_config.settings,
            repo_dirs=repo_dirs,
            fragment_search_paths=fragment_search_paths,
            output_root_dir=output_root_dir,
            handler=handler)

    finally:

        # Get out of the temp directory we're going to delete
        os.chdir(os.path.join(output_root_dir, '..'))

        # Delete generated projects and binaries
        if args.keep_output:
            print('Keeping output at ' + output_root_dir);
        else:
            print('Deleting output at ' + output_root_dir);

            # Try and delete the directory
            for x in range(0, 12):
                if not os.path.isdir(output_root_dir):
                    break
                try:
                    shutil.rmtree(output_root_dir)
                except Exception as e:
                    print('Error deleting temp dir: ' + str(e))
                    time.sleep(5)

            if os.path.isdir(output_root_dir):
                print('ERROR: Failed to delete directory at ' + output_root_dir);
