/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/ads/adtypes.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/module.h"

namespace ttv {
class TaskRunner;

namespace ads {
class AdsAPI;
}
}  // namespace ttv

/**
 * Provides client-side VAST functionality.
 *
 * The following properties must be set before calling Initialize().
 * - SetCoreApi()
 *
 * There are 2 main methods.
 * - FetchAds() will fetch ads to be played on the client.
 * - ReportAdEvent() will take a tracking url and perform macro substitutions before making the request.
 */
class ttv::ads::AdsAPI : public ttv::ModuleBase {
 public:
  using FetchAdsCallback = std::function<void(TTV_ErrorCode ec, std::vector<Ad>&& result)>;
  using ReportAdEventsCallback = std::function<void(TTV_ErrorCode ec)>;

 public:
  AdsAPI();

  // IModule implementation
  virtual std::string GetModuleName() const override;
  virtual TTV_ErrorCode Initialize(const InitializeCallback& callback) override;
  virtual TTV_ErrorCode Shutdown(const ShutdownCallback& callback) override;
  virtual TTV_ErrorCode Update() override;

  TTV_ErrorCode SetCoreApi(const std::shared_ptr<CoreAPI>& coreApi);

  /**
   * Fetches a pod of ads using the given config and info.
   * See AdConfiguration and AdFetchRequestInfo for options.
   * MediaFiles within LinearCreatives will be returned sorted by decreasing width.
   */
  TTV_ErrorCode FetchAds(
    const AdConfiguration& config, const AdFetchRequestInfo& info, const FetchAdsCallback& callback);

  /**
   * Report tracking info to the passed-in URL. This can be used for the following cases:
   *  - Impression URLs, when first displaying the ad
   *  - Click event URLs, when the user clicks on the ad
   *  - Event tracking URLs, which correspond to particular events when displaying the ad (Pause, Mute,
   * Progress/Quartile events, etc.)
   *  - Error URLs, if there is an issue with fetching or playing an ad.
   *  - User Agent for endpoint identification, some ad providers require matching user agent to the FetchAdRequest
   *
   * The URLs provided by the VAST server often include square-bracket-enclosed macro strings for which the client can
   * provide replacement values. The client can provide these values with the macroSubstitutions parameter, with the
   * keys being lower-cased strings of the macro name. The values are automatically percent encoded by the request
   * before being injected into the URL. Any values not provided are replaced with an empty string, with the exception
   * of the "cachebusting" macro which is automatically replaced with an 8-digit random number as per the VAST
   * specification.
   */
  TTV_ErrorCode ReportAdEvent(const std::string& url, const std::map<std::string, std::string>& macroSubstitutions,
    const std::string& adUserAgent, const ReportAdEventsCallback& callback);

  /**
   * Similar to ReportAdEvent but handles a batch of urls.  There is no success or failure callback.  If you require
   * confirmation of success use ReportAdEvent in a loop to know which urls are failing.
   */
  TTV_ErrorCode ReportAdEvents(const std::vector<std::string>& urls,
    const std::map<std::string, std::string>& macroSubstitutions, const std::string& adUserAgent);

 protected:
  // ModuleBase overrides
  virtual bool CheckShutdown() override;
  virtual void CompleteShutdown() override;

 private:
  class CoreApiClient : public ICoreApiClient {
   public:
    // ICoreApiClient implementation
    virtual std::string GetClientName();
  };

  std::shared_ptr<CoreAPI> mCoreApi;
  std::shared_ptr<TaskRunner> mTaskRunner;
  std::shared_ptr<IEventScheduler> mMainEventScheduler;
  std::shared_ptr<CoreApiClient> mCoreApiClient;
};
